import {defineConfig} from 'vite'
import vue from '@vitejs/plugin-vue'
import copy from 'rollup-plugin-copy'

// https://vitejs.dev/config/

/**
 * Wrap JS entry chunks in an IIFE to avoid top-level variable collisions
 * when multiple Vite-built scripts load on the same page.
 */
function wrapInIIFE() {
    return {
        name: 'wrap-iife',
        enforce: 'post',
        generateBundle(_, bundle) {
            for (const chunk of Object.values(bundle)) {
                if (chunk.type === 'chunk' && chunk.isEntry) {
                    chunk.code = `(function(){${chunk.code}})();\n`;
                }
            }
        },
    };
}

export default defineConfig({
    plugins: [
        vue(),
        wrapInIIFE(),
        copy({
            targets: [
                { src: 'static/icons/*', dest: 'assets/icons' },
            ],
            hook: 'writeBundle',
        }),
    ],

    css: {
        preprocessorOptions: {
            scss: {
                api: 'modern-compiler',
            },
        },
    },

    build: {
        minify: 'terser', // here we set up terser instead default esbuild
        terserOptions: {
            mangle: {
                reserved: ['wp', 'Wp'] // here we set up variable we want to exclude //from naming
            }
        },
        manifest: true,
        outDir: 'assets',
        assetsDir: '',
        // publicDir: 'public',
        emptyOutDir: true, // delete the contents of the output directory before each build

        // https://rollupjs.org/guide/en/#big-list-of-options
        rollupOptions: {
            input: [
                'admin/main.js',
                'admin/styles/main.scss',
                'frontend/autocomplete/main.js',
                'frontend/styles/autocomplete.scss',
                'frontend/integrations/woocommerce/filters.js',
                'frontend/integrations/woocommerce/filters.css',
                'frontend/views/search/main.js',
                'frontend/styles/search-view.scss',
                'frontend/styles/highlight.css',
                'frontend/blocks/search/index.js',
                'frontend/blocks/search/editor.css',
            ],
            output: {
                // chunkFileNames: 'js/[name].js',
                entryFileNames: (assetInfo) => {
                    // Admin scripts
                    if (assetInfo?.facadeModuleId.includes('/admin/') && assetInfo.name === 'main') {
                        return 'admin/scripts.js';
                    }
                    // Frontend autocomplete
                    if (assetInfo?.facadeModuleId.includes('/frontend/autocomplete/') && assetInfo.name === 'main') {
                        return 'frontend/autocomplete.js';
                    }
                    // Frontend WooCommerce shop filters
                    if (assetInfo?.facadeModuleId.includes('/frontend/integrations/woocommerce/') && assetInfo.name === 'filters') {
                        return 'frontend/integrations/woocommerce.js';
                    }
                    // Frontend search view
                    if (assetInfo?.facadeModuleId.includes('/frontend/views/search/') && assetInfo.name === 'main') {
                        return 'frontend/search-view.js';
                    }
                    // Block: search editor
                    if (assetInfo?.facadeModuleId.includes('/frontend/blocks/search/') && assetInfo.name === 'index') {
                        return 'blocks/search/index.js';
                    }
                    return '[name].js';
                },
                assetFileNames: (assetInfo) => {
                    if (/\.(gif|jpe?g|png|svg)$/.test(assetInfo.name ?? '')) {
                        return 'images/[name][extname]';
                    }
                    // Admin styles
                    if (assetInfo.originalFileName && assetInfo.originalFileName.includes('admin/') && assetInfo.originalFileName.includes('main.scss')) {
                        return 'admin/styles.css';
                    }
                    // Frontend autocomplete styles
                    if (assetInfo.originalFileName && assetInfo.originalFileName.includes('frontend/') && assetInfo.originalFileName.includes('autocomplete.scss')) {
                        return 'frontend/autocomplete.css';
                    }
                    // Frontend WooCommerce shop filter styles
                    if (assetInfo.originalFileName && assetInfo.originalFileName.includes('frontend/integrations/woocommerce/') && assetInfo.originalFileName.includes('filters.css')) {
                        return 'frontend/integrations/woocommerce.css';
                    }
                    // Frontend search view styles
                    if (assetInfo.originalFileName && assetInfo.originalFileName.includes('frontend/styles/') && assetInfo.originalFileName.includes('search-view.scss')) {
                        return 'frontend/search-view.css';
                    }
                    // Frontend highlight styles
                    if (assetInfo.originalFileName && assetInfo.originalFileName.includes('frontend/styles/') && assetInfo.originalFileName.includes('highlight.css')) {
                        return 'frontend/highlight.css';
                    }
                    // Block: search editor styles
                    if (assetInfo.originalFileName && assetInfo.originalFileName.includes('frontend/blocks/search/') && assetInfo.originalFileName.includes('editor.css')) {
                        return 'blocks/search/editor.css';
                    }
                    if (/\.css$/.test(assetInfo.name ?? '')) {
                        return 'css/[name][extname]';
                    }
                    return '[name][extname]';
                },
            },
        },
    },

    resolve: {
        alias: {
            'vue': 'vue/dist/vue.esm-bundler.js',
        },
    }
})

