<?php

namespace CelerSearch\Views;

defined( 'ABSPATH' ) || exit;

/**
 * Abstract base class for portable view components
 *
 * Views encapsulate rendering, asset management, and configuration.
 * They can be consumed by shortcodes, Elementor widgets, Gutenberg blocks, etc.
 */
abstract class AbstractView {

	/**
	 * Instance counter for unique IDs
	 *
	 * @var int
	 */
	private static int $counter = 0;

	/**
	 * Instance ID
	 *
	 * @var int
	 */
	protected int $instance_id;

	/**
	 * View configuration
	 *
	 * @var array
	 */
	protected array $config;

	/**
	 * Constructor
	 *
	 * @param array $config View configuration.
	 */
	public function __construct( array $config ) {
		self::$counter++;
		$this->instance_id = self::$counter;
		$this->config      = $this->configure( $config );
	}

	/**
	 * Merge and validate config with defaults
	 *
	 * @param array $config Raw configuration.
	 *
	 * @return array Processed configuration.
	 */
	abstract protected function configure( array $config ): array;

	/**
	 * Validate the view configuration
	 *
	 * @return array Array of error messages. Empty means valid.
	 */
	abstract public function validate(): array;

	/**
	 * Render the view HTML
	 *
	 * @return string
	 */
	abstract public function render(): string;

	/**
	 * Get JS configuration for this view instance
	 *
	 * @return array
	 */
	abstract public function get_js_config(): array;

	/**
	 * Get the asset handle prefix for this view type
	 *
	 * @return string
	 */
	abstract public function get_asset_handle(): string;

	/**
	 * Get the instance ID
	 *
	 * @return int
	 */
	public function get_instance_id(): int {
		return $this->instance_id;
	}

	/**
	 * Get a unique ID for this view instance
	 *
	 * @return string
	 */
	public function get_unique_id(): string {
		return $this->get_asset_handle() . '-' . $this->instance_id;
	}
}
