<?php

namespace CelerSearch\Shortcodes;

defined( 'ABSPATH' ) || exit;

/**
 * Abstract base class for shortcodes
 *
 * Handles attribute parsing, sanitization, validation, and delegates rendering to a View.
 */
abstract class AbstractShortcode {

	/**
	 * Get the shortcode tag name
	 *
	 * @return string
	 */
	abstract public function get_tag(): string;

	/**
	 * Get default attribute values
	 *
	 * @return array
	 */
	abstract protected function get_defaults(): array;

	/**
	 * Render the shortcode output
	 *
	 * @param array       $atts    Sanitized attributes.
	 * @param string|null $content Shortcode content.
	 *
	 * @return string
	 */
	abstract protected function render( array $atts, ?string $content ): string;

	/**
	 * Handle the shortcode
	 *
	 * @param array|string $atts    Raw shortcode attributes.
	 * @param string|null  $content Shortcode content.
	 *
	 * @return string
	 */
	public function handle( $atts = [], ?string $content = null ): string {
		$atts = shortcode_atts( $this->get_defaults(), $atts, $this->get_tag() );

		/**
		 * Filter shortcode attributes
		 *
		 * @param array  $atts Parsed attributes.
		 * @param string $tag  Shortcode tag.
		 */
		$atts = apply_filters( 'celersearch_shortcode_atts', $atts, $this->get_tag() );

		return $this->render( $atts, $content );
	}

	/**
	 * Register this shortcode with WordPress
	 *
	 * @return void
	 */
	public function register(): void {
		add_shortcode( $this->get_tag(), [ $this, 'handle' ] );
	}
}
