<?php

namespace CelerSearch\SearchAreas;

/**
 * WooCommerce Product search area
 *
 * Handles autocomplete for WooCommerce product search widgets and blocks
 * anywhere on the frontend of the site.
 */
class WooCommerceProductSearchArea extends AbstractSearchArea {

	/**
	 * Get the unique type identifier
	 *
	 * @return string
	 */
	public function get_type(): string {
		return 'woocommerce_product_search';
	}

	/**
	 * Get the human-readable label
	 *
	 * @return string
	 */
	public function get_label(): string {
		return __( 'WooCommerce Product Search', 'celersearch' );
	}

	/**
	 * Get the CSS selector for search inputs
	 *
	 * Targets WooCommerce product search widgets and blocks anywhere on the site.
	 *
	 * @return string
	 */
	public function get_input_selector(): string {
		$selectors = [
			'.woocommerce-product-search input[type="search"]',
			'.wc-block-product-search input[type="search"]',
			'.wc-block-product-search__field',
			'.wp-block-search__input',
		];

		$selector = implode( ', ', $selectors );

		return (string) apply_filters( 'celersearch_woocommerce_product_search_input_selector', $selector );
	}

	/**
	 * Check if this search area should attach in the current context
	 *
	 * @return bool
	 */
	public function should_attach(): bool {
		// Only attach on frontend when WooCommerce is active
		if ( is_admin() || wp_doing_ajax() ) {
			return false;
		}

		// WooCommerce must be active
		if ( ! class_exists( 'WooCommerce' ) ) {
			return false;
		}

		// Allow filtering
		return (bool) apply_filters( 'celersearch_woocommerce_product_search_should_attach', true, $this );
	}

	/**
	 * Get the context this area operates in
	 *
	 * @return string
	 */
	public function get_context(): string {
		return 'frontend';
	}
}
