<?php

namespace CelerSearch\SearchAreas;

/**
 * WooCommerce Orders search area
 *
 * Handles autocomplete for WooCommerce orders search in the admin area.
 * Supports both HPOS (High-Performance Order Storage) and legacy CPT orders.
 */
class WooCommerceOrdersSearchArea extends AbstractSearchArea {

	/**
	 * Get the unique type identifier
	 *
	 * @return string
	 */
	public function get_type(): string {
		return 'woocommerce_orders_search';
	}

	/**
	 * Get the human-readable label
	 *
	 * @return string
	 */
	public function get_label(): string {
		return __( 'WooCommerce Admin Orders Search', 'celersearch' );
	}

	/**
	 * Get the CSS selector for search inputs
	 *
	 * HPOS uses #orders-search-input-search-input (table ID + '-search-input' suffix).
	 *
	 * @return string
	 */
	public function get_input_selector(): string {
		// HPOS orders page input: the table ID is 'orders-search-input' and WP appends '-search-input'
		// Full ID: orders-search-input-search-input
		return (string) apply_filters( 'celersearch_woocommerce_orders_input_selector', '#orders-search-input-search-input' );
	}

	/**
	 * Check if this search area should attach in the current context
	 *
	 * @return bool
	 */
	public function should_attach(): bool {
		if ( ! is_admin() || ! class_exists( 'WooCommerce' ) ) {
			return false;
		}

		$screen = function_exists( 'get_current_screen' ) ? get_current_screen() : null;
		if ( ! $screen ) {
			return false;
		}

		// HPOS: screen ID is 'woocommerce_page_wc-orders'
		// Legacy: screen base is 'edit' with post_type 'shop_order'
		$is_hpos_orders   = $screen->id === 'woocommerce_page_wc-orders';
		$is_legacy_orders = $screen->base === 'edit' && $screen->post_type === 'shop_order';

		if ( $is_hpos_orders || $is_legacy_orders ) {
			return (bool) apply_filters( 'celersearch_woocommerce_orders_should_attach', true, $this );
		}

		return false;
	}

	/**
	 * Get the context this area operates in
	 *
	 * @return string
	 */
	public function get_context(): string {
		return 'admin';
	}
}
