<?php

namespace CelerSearch\SearchAreas;

use CelerSearch\Utilities\Logger;

/**
 * Admin Post Type search area
 *
 * Handles autocomplete for post type list table searches
 * in the WordPress admin area. Parameterized by post type.
 */
class AdminPostTypeSearchArea extends AbstractSearchArea {

	/**
	 * The post type this area is for
	 *
	 * @var string
	 */
	protected string $post_type = 'post';

	/**
	 * Constructor
	 *
	 * @param array $config Configuration from settings.
	 */
	public function __construct( array $config = [] ) {
		parent::__construct( $config );

		// Extract post_type from config or from type string
		if ( ! empty( $config['post_type'] ) ) {
			$this->post_type = $config['post_type'];
		} elseif ( ! empty( $config['type'] ) && preg_match( '/^admin_(.+)_search$/', $config['type'], $matches ) ) {
			$this->post_type = $matches[1];
		}
	}

	/**
	 * Get the unique type identifier
	 *
	 * @return string
	 */
	public function get_type(): string {
		return 'admin_' . $this->post_type . '_search';
	}

	/**
	 * Get the human-readable label
	 *
	 * @return string
	 */
	public function get_label(): string {
		$post_type_obj = get_post_type_object( $this->post_type );
		$label         = $post_type_obj ? $post_type_obj->labels->name : ucfirst( $this->post_type );

		/* translators: %s: post type name */
		return sprintf( __( 'Admin %s Search', 'celersearch' ), $label );
	}

	/**
	 * Get the CSS selector for search inputs
	 *
	 * @return string
	 */
	public function get_input_selector(): string {
		$selector = '#post-search-input';

		return (string) apply_filters( 'celersearch_admin_post_type_input_selector', $selector, $this->post_type );
	}

	/**
	 * Check if this search area should attach in the current context
	 *
	 * @return bool
	 */
	public function should_attach(): bool {
		Logger::channel( Logger::CHANNEL_DEBUG )->debug( 'AdminPostTypeSearchArea::should_attach()', [
			'configured_post_type' => $this->post_type,
		] );

		// Only attach in admin
		if ( ! is_admin() ) {
			Logger::channel( Logger::CHANNEL_DEBUG )->debug( 'Not in admin - returning false' );
			return false;
		}

		// Check if we're on an edit screen (list table)
		$screen = function_exists( 'get_current_screen' ) ? get_current_screen() : null;

		if ( ! $screen ) {
			Logger::channel( Logger::CHANNEL_DEBUG )->debug( 'get_current_screen() returned null' );
			return false;
		}

		Logger::channel( Logger::CHANNEL_DEBUG )->debug( 'Current screen info', [
			'base'      => $screen->base,
			'post_type' => $screen->post_type,
		] );

		// Must be on the edit screen (list table) for this specific post type
		if ( $screen->base !== 'edit' ) {
			Logger::channel( Logger::CHANNEL_DEBUG )->debug( 'Screen base is not "edit"', [ 'base' => $screen->base ] );
			return false;
		}

		// Must match our post type
		if ( $screen->post_type !== $this->post_type ) {
			Logger::channel( Logger::CHANNEL_DEBUG )->debug( 'Post type mismatch', [
				'screen_post_type' => $screen->post_type,
				'configured'       => $this->post_type,
			] );
			return false;
		}

		Logger::channel( Logger::CHANNEL_DEBUG )->debug( 'should_attach() returning true' );

		// Allow filtering
		return (bool) apply_filters( 'celersearch_admin_post_type_should_attach', true, $this, $screen );
	}

	/**
	 * Get the context this area operates in
	 *
	 * @return string
	 */
	public function get_context(): string {
		return 'admin';
	}

	/**
	 * Get the post type this area is for
	 *
	 * @return string
	 */
	public function get_post_type(): string {
		return $this->post_type;
	}
}
