<?php

namespace CelerSearch\SearchAreas;

/**
 * Abstract base class for search areas
 *
 * Search areas define where and how autocomplete is attached
 * to search inputs in different contexts (frontend, admin, etc.)
 */
abstract class AbstractSearchArea {

	/**
	 * Configuration from settings
	 *
	 * @var array
	 */
	protected array $config = [];

	/**
	 * The index ID to use for this area
	 *
	 * @var int
	 */
	protected int $index_id = 0;

	/**
	 * Constructor
	 *
	 * @param array $config Configuration from settings.
	 */
	public function __construct( array $config = [] ) {
		$this->config   = $config;
		$this->index_id = (int) ( $config['index_id'] ?? 0 );
	}

	/**
	 * Get the unique type identifier for this search area
	 *
	 * @return string
	 */
	abstract public function get_type(): string;

	/**
	 * Get the human-readable label for this search area
	 *
	 * @return string
	 */
	abstract public function get_label(): string;

	/**
	 * Get the CSS selector for search inputs to enhance
	 *
	 * @return string
	 */
	abstract public function get_input_selector(): string;

	/**
	 * Check if this search area should attach in the current context
	 *
	 * @return bool
	 */
	abstract public function should_attach(): bool;

	/**
	 * Get the context this area operates in ('frontend' or 'admin')
	 *
	 * @return string
	 */
	abstract public function get_context(): string;

	/**
	 * Check if this search area supports autocomplete
	 *
	 * @return bool
	 */
	public function supports_autocomplete(): bool {
		return true;
	}

	/**
	 * Get the debounce delay in milliseconds
	 *
	 * @return int
	 */
	public function get_debounce_ms(): int {
		return (int) apply_filters( 'celersearch_autocomplete_debounce', 300, $this );
	}

	/**
	 * Get the minimum characters required to trigger search
	 *
	 * @return int
	 */
	public function get_min_chars(): int {
		return (int) apply_filters( 'celersearch_autocomplete_min_chars', 2, $this );
	}

	/**
	 * Get the maximum number of results to show
	 *
	 * @return int
	 */
	public function get_max_results(): int {
		return (int) apply_filters( 'celersearch_autocomplete_max_results', 5, $this );
	}

	/**
	 * Get the index ID for this area
	 *
	 * @return int
	 */
	public function get_index_id(): int {
		return $this->index_id;
	}

	/**
	 * Get the area ID from config
	 *
	 * @return string
	 */
	public function get_id(): string {
		return $this->config['id'] ?? '';
	}

	/**
	 * Check if autocomplete is enabled for this area
	 *
	 * @return bool
	 */
	public function is_autocomplete_enabled(): bool {
		return ! empty( $this->config['autocomplete'] );
	}

	/**
	 * Check if this area is enabled
	 *
	 * @return bool
	 */
	public function is_enabled(): bool {
		return ! empty( $this->config['enabled'] );
	}

	/**
	 * Check if a specific feature is enabled for this area
	 *
	 * @param string $feature_key The feature key to check.
	 *
	 * @return bool
	 */
	public function is_feature_enabled( string $feature_key ): bool {
		$feature = $this->config['features'][ $feature_key ] ?? false;
		if ( is_array( $feature ) ) {
			return ! empty( $feature['enabled'] );
		}

		return ! empty( $feature );
	}

	/**
	 * Get the configuration array for JavaScript
	 *
	 * @return array
	 */
	public function get_js_config(): array {
		$config = [
			'id'         => $this->get_id(),
			'type'       => $this->get_type(),
			'selector'   => $this->get_input_selector(),
			'debounce'   => $this->get_debounce_ms(),
			'minChars'   => $this->get_min_chars(),
			'maxResults' => $this->get_max_results(),
			'indexId'    => $this->get_index_id(),
			'context'    => $this->get_context(),
		];

		return (array) apply_filters( 'celersearch_search_area_config', $config, $this );
	}
}
