<?php

namespace CelerSearch\Indices;

defined( 'ABSPATH' ) || exit;

/**
 * Base class for post-based indices (posts, products, etc.)
 *
 * Provides common autocomplete formatting for WordPress post types.
 */
abstract class BasePostIndex extends BaseIndex {

	/**
	 * Format a post-based hit for autocomplete display
	 *
	 * @param array $hit Raw hit from search index
	 *
	 * @return array|null Formatted hit or null to skip
	 */
	public function format_autocomplete_hit( array $hit ): ?array {
		$post_id = $hit['id'] ?? $hit['post_id'] ?? 0;

		if ( empty( $post_id ) ) {
			return null;
		}

		$post = get_post( $post_id );

		if ( ! $post ) {
			return null;
		}

		$post_type_obj = get_post_type_object( $post->post_type );

		return [
			'id'              => $post_id,
			'post_type'       => $post->post_type,
			'post_type_label' => $post_type_obj ? $post_type_obj->labels->singular_name : $post->post_type,
			'title'           => $hit['title'] ?? $hit['post_title'] ?? $post->post_title,
			'excerpt'         => $this->get_autocomplete_excerpt( $hit, $post ),
			'url'             => get_permalink( $post_id ),
			'edit_url'        => get_edit_post_link( $post_id, 'raw' ),
			'thumbnail'       => $this->get_autocomplete_thumbnail( $post_id ),
			'variations'      => $hit['variations'] ?? [],
			'_formatted'      => $this->sanitize_formatted_fields( $hit['_formatted'] ?? [] ),
		];
	}

	/**
	 * Get excerpt for autocomplete hit
	 *
	 * @param array    $hit
	 * @param \WP_Post $post
	 *
	 * @return string
	 */
	protected function get_autocomplete_excerpt( array $hit, \WP_Post $post ): string {
		if ( ! empty( $hit['content'] ) ) {
			return wp_trim_words( wp_strip_all_tags( $hit['content'] ), 15 );
		}

		if ( ! empty( $hit['short_description'] ) ) {
			return wp_trim_words( wp_strip_all_tags( $hit['short_description'] ), 15 );
		}

		if ( ! empty( $post->post_excerpt ) ) {
			return wp_trim_words( wp_strip_all_tags( $post->post_excerpt ), 15 );
		}

		return wp_trim_words( wp_strip_all_tags( $post->post_content ), 15 );
	}

	/**
	 * Get thumbnail URL for autocomplete hit
	 *
	 * @param int $post_id
	 *
	 * @return string|null
	 */
	protected function get_autocomplete_thumbnail( int $post_id ): ?string {
		$thumbnail_id = get_post_thumbnail_id( $post_id );

		if ( ! $thumbnail_id ) {
			return null;
		}

		$thumbnail_url = wp_get_attachment_image_url( $thumbnail_id, 'thumbnail' );

		return $thumbnail_url ?: null;
	}

	/**
	 * Get status filter for post-based autocomplete search
	 *
	 * @param string $context 'frontend' or 'admin'
	 *
	 * @return array|null
	 */
	public function get_autocomplete_status_filter( string $context ): ?array {
		if ( $context === 'admin' ) {
			$statuses = [ 'publish', 'draft', 'pending', 'private', 'future' ];
		} else {
			$statuses = [ 'publish' ];
		}

		$statuses = apply_filters( 'celersearch_autocomplete_statuses', $statuses, $context );

		if ( empty( $statuses ) ) {
			return null;
		}

		if ( count( $statuses ) === 1 ) {
			return [
				'field'    => 'status',
				'operator' => '=',
				'value'    => $statuses[0],
			];
		}

		return [
			'field'    => 'status',
			'operator' => 'IN',
			'value'    => $statuses,
		];
	}
}
