<?php

namespace CelerSearch\Factories;

class SearchAreaFactory {

	/**
	 * Get available search area types
	 *
	 * @return array
	 */
	public static function get_available_types(): array {
		$types = [
			// Frontend types
			[
				'type'        => 'wordpress_public_search',
				'label'       => __( 'WordPress Public Search', 'celersearch' ),
				'description' => __( 'Replace the default WordPress search form on the frontend', 'celersearch' ),
				'icon'        => 'dashicons-search',
				'category'    => 'frontend',
				'features'    => [
					[
						'key'         => 'autocomplete',
						'label'       => __( 'Autocomplete', 'celersearch' ),
						'description' => __( 'Show search suggestions as the user types', 'celersearch' ),
					],
					[
						'key'         => 'search_page',
						'label'       => __( 'Replace Search Results Page', 'celersearch' ),
						'description' => __( 'Redirect WordPress search to a page with the CelerSearch shortcode', 'celersearch' ),
						'options'     => [
							[
								'key'     => 'page_id',
								'label'   => __( 'Search Results Page', 'celersearch' ),
								'type'    => 'select',
								'default' => '',
								'choices' => self::get_page_choices(),
							],
						],
					],
				],
			],
		];

		// Add WooCommerce product search if WooCommerce is active
		if ( class_exists( 'WooCommerce' ) ) {
			$types[] = [
				'type'        => 'woocommerce_product_search',
				'label'       => __( 'WooCommerce Product Search', 'celersearch' ),
				'description' => __( 'Replace WooCommerce product search widgets and blocks', 'celersearch' ),
				'icon'        => 'dashicons-cart',
				'category'    => 'frontend',
				'features'    => [
					[
						'key'         => 'autocomplete',
						'label'       => __( 'Autocomplete', 'celersearch' ),
						'description' => __( 'Show search suggestions as the user types', 'celersearch' ),
					],
				],
			];

			$types[] = [
				'type'        => 'woocommerce_shop_browse',
				'label'       => __( 'WooCommerce Shop Browse', 'celersearch' ),
				'description' => __( 'Power shop, category, and tag archive pages with Meilisearch', 'celersearch' ),
				'icon'        => 'dashicons-store',
				'category'    => 'frontend',
				'features'    => [
					[
						'key'         => 'filters',
						'label'       => __( 'Shop Filters', 'celersearch' ),
						'description' => __( 'Show faceted filter bar on shop pages', 'celersearch' ),
						'options'     => [
							[
								'key'     => 'style',
								'label'   => __( 'Display Style', 'celersearch' ),
								'type'    => 'select',
								'default' => 'drawer',
								'choices' => [
									[ 'value' => 'drawer', 'label' => __( 'Slide-out Drawer', 'celersearch' ) ],
									[ 'value' => 'inline', 'label' => __( 'Inline Bar', 'celersearch' ) ],
								],
							],
						],
					],
					[
						'key'         => 'search',
						'label'       => __( 'Search Field', 'celersearch' ),
						'description' => __( 'Show a search input in the toolbar to filter products by keyword', 'celersearch' ),
						'options'     => [],
					],
				],
			];

			$types[] = [
				'type'        => 'woocommerce_orders_search',
				'label'       => __( 'WooCommerce Admin Orders Search', 'celersearch' ),
				'description' => __( 'Replace WooCommerce orders search in admin area', 'celersearch' ),
				'icon'        => 'dashicons-clipboard',
				'category'    => 'admin',
				'features'    => [
					[
						'key'         => 'autocomplete',
						'label'       => __( 'Autocomplete', 'celersearch' ),
						'description' => __( 'Show search suggestions as the user types', 'celersearch' ),
					],
				],
			];
		}

		// Dynamically add admin types for each public post type
		$post_types = get_post_types( [ 'public' => true, 'show_ui' => true ], 'objects' );
		foreach ( $post_types as $post_type ) {
			// Skip attachments (media library has its own handling)
			if ( $post_type->name === 'attachment' ) {
				continue;
			}

			/* translators: %s: post type name */
			$label = sprintf( __( 'Admin %s Search', 'celersearch' ), $post_type->labels->name );
			/* translators: %s: post type name */
			$description = sprintf( __( 'Replace the %s search in WordPress admin', 'celersearch' ), strtolower( $post_type->labels->name ) );
			$types[] = [
				'type'        => 'admin_' . $post_type->name . '_search',
				'label'       => $label,
				'description' => $description,
				'icon'        => $post_type->menu_icon ?: 'dashicons-admin-post',
				'category'    => 'admin',
				'post_type'   => $post_type->name,
				'features'    => [
					[
						'key'         => 'autocomplete',
						'label'       => __( 'Autocomplete', 'celersearch' ),
						'description' => __( 'Show search suggestions as the user types', 'celersearch' ),
					],
				],
			];
		}

		// Add media library search (special handling, no features)
		$types[] = [
			'type'        => 'admin_media_search',
			'label'       => __( 'Admin Media Library Search', 'celersearch' ),
			'description' => __( 'Replace media library search in WordPress admin', 'celersearch' ),
			'icon'        => 'dashicons-admin-media',
			'category'    => 'admin',
			'features'    => [],
		];

		// REST API search
		$types[] = [
			'type'        => 'rest_api_search',
			'label'       => __( 'REST API Search', 'celersearch' ),
			'description' => __( 'Enable search via WordPress REST API', 'celersearch' ),
			'icon'        => 'dashicons-rest-api',
			'category'    => 'api',
			'features'    => [],
		];

		// Allow third-party plugins to add their own search area types
		return apply_filters( 'celersearch_area_types', $types );
	}

	/**
	 * Get a specific search area type by its type identifier
	 *
	 * @param string $type
	 *
	 * @return array|null
	 */
	public static function get_type( string $type ): ?array {
		$types = self::get_available_types();

		foreach ( $types as $area_type ) {
			if ( $area_type['type'] === $type ) {
				return $area_type;
			}
		}

		return null;
	}

	/**
	 * Validate if a search area type exists
	 *
	 * @param string $type
	 *
	 * @return bool
	 */
	public static function type_exists( string $type ): bool {
		return self::get_type( $type ) !== null;
	}

	/**
	 * Get published pages as choices for the search page dropdown
	 *
	 * @return array
	 */
	private static function get_page_choices(): array {
		$pages   = get_pages( [ 'post_status' => 'publish', 'sort_column' => 'post_title' ] );
		$choices = [
			[ 'value' => '', 'label' => __( '— Select a page —', 'celersearch' ) ],
		];
		foreach ( $pages ?: [] as $page ) {
			$choices[] = [ 'value' => (string) $page->ID, 'label' => $page->post_title ];
		}

		return $choices;
	}
}
