<?php

namespace CelerSearch\DataTransfer;

use CelerSearch\Abstracts\RepositoryItem;
use CelerSearch\Utilities\DateTimeUtilities;

class IndexConfig extends RepositoryItem {

	/**
	 * The index configuration value
	 * @var object
	 */
	protected object $value;

	/**
	 * Constructor
	 * @param object $value
	 */
	public function __construct( object $value ) {
		$this->value = $value;
		if ( isset( $this->value->config ) && is_array( $this->value->config ) ) {
			$this->value->config = (object) $this->value->config;
		}
	}

	/**
	 * Get the index ID
	 * @return string
	 */
	public function getId(): string {
		return $this->get( 'id' );
	}

	/**
	 * Get the index name
	 * @return string
	 */
	public function getName(): string {
		return $this->get( 'name' );
	}

	/**
	 * Get the index type
	 * @return string
	 */
	public function getType(): string {
		return $this->get( 'type' );
	}

	/**
	 * Get the index slug
	 * @return string
	 */
	public function getSlug(): string {
		return $this->get( 'slug' );
	}

	/**
	 * Get the index config
	 * @return null|object
	 */
	public function getConfig(): null|object {
		return $this->get( 'config' );
	}

	/**
	 * Get the index status
	 * @return null|string
	 */
	public function getStatus(): null|string {
		return $this->get( 'status' );
	}

	/**
	 * Get the service ID
	 * @return null|int
	 */
	public function getServiceId(): null|int {
		return $this->get( 'service_id' );
	}

	/**
	 * Get the total remote items
	 * @return null|int
	 */
	public function getTotalRemote(): null|int {
		return $this->get( 'total_remote' );
	}

	/**
	 * Get the total local items
	 * @return null|int
	 */
	public function getTotalLocal(): null|int {
		return $this->get( 'total_local' );
	}

	/**
	 * Get the checked at timestamp
	 * @return null|string
	 */
	public function getCheckedAt(): null|string {
		return $this->get( 'checked_at' );
	}

	/**
	 * Get the synced at timestamp
	 * @return null|string
	 */
	public function getSyncedAt(): null|string {
		return $this->get( 'synced_at' );
	}

	/**
	 * Get the created at timestamp
	 * @return null|string
	 */
	public function getCreatedAt(): null|string {
		return $this->get( 'created_at' );
	}

	/**
	 * Get a specific key from the index configuration
	 * @param string $key
	 * @return mixed
	 */
	public function get( $key ): mixed {
		return isset( $this->value->$key ) ? $this->value->$key : null;
	}

	/**
	 * Get the full value object
	 * @return object
	 */
	public function getValue() {
		return $this->value;
	}

	/**
	 * Get the formatted value object
	 * @return object
	 */
	public function getFormatted() {
		$value = $this->value;

		if ( ! empty( $value->checked_at ) ) {
			$value->checked_at = DateTimeUtilities::format( $value->checked_at );
		}
		if ( ! empty( $value->synced_at ) ) {
			$value->synced_at = DateTimeUtilities::format( $value->synced_at );
		}
		if ( ! empty( $value->created_at ) ) {
			$value->created_at = DateTimeUtilities::format( $value->created_at );
		}
		return $value;
	}

	/**
	 * Convert to array
	 * @return array
	 */
	public function toArray(): array {
		return (array) $this->value;
	}

	/**
	 * Convert to string
	 * @return string
	 */
	public function __toString(): string {
		return json_encode( $this->value );
	}
}