<?php

namespace CelerSearch\Autocomplete;

/**
 * Template loader for autocomplete templates
 *
 * Handles loading templates with theme override support.
 * Priority: child theme > parent theme > plugin templates
 */
class TemplateLoader {

	/**
	 * Plugin template directory
	 */
	const PLUGIN_TEMPLATE_DIR = 'templates/autocomplete';

	/**
	 * Theme template directory
	 */
	const THEME_TEMPLATE_DIR = 'celersearch/autocomplete';

	/**
	 * Load a template file with theme override support
	 *
	 * @param string $name Template name (without .php extension).
	 * @param array  $args Variables to pass to the template.
	 *
	 * @return string The rendered template content.
	 */
	public static function get_template( string $name, array $args = [] ): string {
		$template = self::locate_template( $name );

		if ( ! $template ) {
			return '';
		}

		ob_start();

		// Make args available to template via $template_args variable.
		// Theme overrides can access passed variables directly, e.g., $template_args['custom_var']
		$template_args = $args;
		unset( $args ); // Prevent variable pollution

		include $template;

		return ob_get_clean();
	}

	/**
	 * Locate a template file with theme override priority
	 *
	 * @param string $name Template name (without .php extension).
	 *
	 * @return string|null Full path to template or null if not found.
	 */
	public static function locate_template( string $name ): ?string {
		$template_file = $name . '.php';

		// Build list of locations to check (priority order)
		$locations = [
			// Child theme
			get_stylesheet_directory() . '/' . self::THEME_TEMPLATE_DIR . '/' . $template_file,
			// Parent theme
			get_template_directory() . '/' . self::THEME_TEMPLATE_DIR . '/' . $template_file,
			// Plugin
			\CELERSEARCH_PLUGIN_PATH . self::PLUGIN_TEMPLATE_DIR . '/' . $template_file,
		];

		// Allow filtering locations
		$locations = apply_filters( 'celersearch_template_locations', $locations, $name );

		foreach ( $locations as $location ) {
			if ( file_exists( $location ) ) {
				return $location;
			}
		}

		return null;
	}

	/**
	 * Get all templates as strings for JavaScript
	 *
	 * Templates use {{placeholder}} syntax for JS replacement.
	 *
	 * @return array
	 */
	public static function get_js_templates(): array {
		$templates = [
			'container'  => self::get_template( 'container' ),
			'suggestion' => self::get_template( 'suggestion' ),
			'header'     => self::get_template( 'header' ),
			'footer'     => self::get_template( 'footer' ),
			'empty'      => self::get_template( 'empty' ),
		];

		return apply_filters( 'celersearch_js_templates', $templates );
	}

	/**
	 * Check if a template exists
	 *
	 * @param string $name Template name.
	 *
	 * @return bool
	 */
	public static function template_exists( string $name ): bool {
		return self::locate_template( $name ) !== null;
	}

	/**
	 * Get the path to a template (for debugging/info)
	 *
	 * @param string $name Template name.
	 *
	 * @return string|null
	 */
	public static function get_template_path( string $name ): ?string {
		return self::locate_template( $name );
	}
}
