<?php

namespace CelerSearch\Autocomplete;

defined( 'ABSPATH' ) || exit;

use CelerSearch\Interfaces\IRegistrable;
use CelerSearch\SearchAreas\SearchAreaRegistry;

/**
 * Bootstrap class for the autocomplete system
 *
 * Initializes all components of the autocomplete feature:
 * - Search Area Registry
 * - REST API endpoint
 * - Asset management
 */
class Boot implements IRegistrable {

	/**
	 * REST API handler
	 *
	 * @var RestApi
	 */
	private RestApi $rest_api;

	/**
	 * Assets handler
	 *
	 * @var Assets
	 */
	private Assets $assets;

	/**
	 * Constructor
	 */
	public function __construct() {
		$this->rest_api = new RestApi();
		$this->assets   = new Assets();
	}

	/**
	 * Register hooks
	 *
	 * @return void
	 */
	public function register(): void {
		// Register default search area classes after all plugins have loaded
		// This ensures custom post types (like WooCommerce products) are available
		add_action( 'init', [ $this, 'register_search_areas' ], 20 );

		// Register REST API routes
		$this->rest_api->register();

		// Register asset enqueue hooks
		$this->assets->register();
	}

	/**
	 * Register search areas after all post types are registered
	 *
	 * @return void
	 */
	public function register_search_areas(): void {
		// Register default search area classes
		SearchAreaRegistry::register_defaults();

		// Allow plugins to register custom search areas
		do_action( 'celersearch_register_search_areas' );
	}
}
