<?php

namespace CelerSearch\Autocomplete;

defined( 'ABSPATH' ) || exit;

use CelerSearch\Interfaces\IRegistrable;
use CelerSearch\SearchAreas\SearchAreaRegistry;

/**
 * Asset management for autocomplete
 *
 * Handles enqueueing scripts and styles for the autocomplete feature
 * in both frontend and admin contexts.
 */
class Assets implements IRegistrable {

	/**
	 * Script handle
	 */
	const SCRIPT_HANDLE = 'celersearch-autocomplete';

	/**
	 * Style handle
	 */
	const STYLE_HANDLE = 'celersearch-autocomplete';

	/**
	 * Register hooks
	 *
	 * @return void
	 */
	public function register(): void {
		add_action( 'wp_enqueue_scripts', [ $this, 'enqueue_frontend' ], 20 );
		add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_admin' ], 20 );
	}

	/**
	 * Enqueue frontend scripts and styles
	 *
	 * @return void
	 */
	public function enqueue_frontend(): void {
		if ( ! $this->should_enqueue( 'frontend' ) ) {
			return;
		}

		$this->enqueue_scripts( 'frontend' );
	}

	/**
	 * Enqueue admin scripts and styles
	 *
	 * @return void
	 */
	public function enqueue_admin(): void {
		if ( ! $this->should_enqueue( 'admin' ) ) {
			return;
		}

		$this->enqueue_scripts( 'admin' );
	}

	/**
	 * Check if scripts should be enqueued for context
	 *
	 * @param string $context The context ('frontend' or 'admin').
	 *
	 * @return bool
	 */
	private function should_enqueue( string $context ): bool {
		$areas = SearchAreaRegistry::get_active_areas_for_context( $context );

		return ! empty( $areas );
	}

	/**
	 * Enqueue scripts and styles
	 *
	 * @param string $context The context ('frontend' or 'admin').
	 *
	 * @return void
	 */
	private function enqueue_scripts( string $context ): void {
		// Enqueue styles
		wp_enqueue_style(
			self::STYLE_HANDLE,
			\CELERSEARCH_PLUGIN_URL . 'assets/frontend/autocomplete.css',
			[],
			\CELERSEARCH_PLUGIN_VERSION
		);

		// Enqueue script
		wp_enqueue_script(
			self::SCRIPT_HANDLE,
			\CELERSEARCH_PLUGIN_URL . 'assets/frontend/autocomplete.js',
			[],
			\CELERSEARCH_PLUGIN_VERSION,
			true // Load in footer
		);

		// Localize script with config
		wp_localize_script(
			self::SCRIPT_HANDLE,
			'celerSearchAutocomplete',
			$this->get_script_config( $context )
		);
	}

	/**
	 * Get script configuration for localization
	 *
	 * @param string $context The context ('frontend' or 'admin').
	 *
	 * @return array
	 */
	private function get_script_config( string $context ): array {
		$areas        = SearchAreaRegistry::get_active_areas_for_context( $context );
		$area_configs = [];

		foreach ( $areas as $area ) {
			$area_configs[] = $area->get_js_config();
		}

		$config = [
			'restUrl'   => rest_url( 'celersearch/v1/autocomplete' ),
			'nonce'     => wp_create_nonce( 'wp_rest' ),
			'areas'     => $area_configs,
			'templates' => TemplateLoader::get_js_templates(),
			'i18n'      => [
				'noResults'    => __( 'No results found', 'celersearch' ),
				'searching'    => __( 'Searching...', 'celersearch' ),
				'viewAll'      => __( 'View all results', 'celersearch' ),
				'resultsLabel' => __( 'Search results', 'celersearch' ),
				'moreOptions'  => __( 'more options', 'celersearch' ),
			],
		];

		return apply_filters( 'celersearch_autocomplete_js_config', $config, $context );
	}
}
