<?php

namespace CelerSearch\Admin;

defined( 'ABSPATH' ) || exit;

use CelerSearch\Interfaces\IRegistrable;

class Assets implements IRegistrable {

	/**
	 * List of scripts
	 * @var array|array[]
	 */
	protected array $scripts = [];

	/**
	 * List of styles
	 * @var array|array[]
	 */
	protected array $styles = [];

	/**
	 * List of global styles (enqueued on all admin pages)
	 * @var array|array[]
	 */
	protected array $global_styles = [];

	public function __construct() {

		$this->scripts = [
			'celersearch-admin' => [
				'src'       => CELERSEARCH_PLUGIN_URL . 'assets/admin/scripts.js',
				'deps'      => [ 'wp-i18n' ],
				'version'   => CELERSEARCH_PLUGIN_VERSION,
				'in_footer' => true,
				'localize'  => [
					'object' => 'CelerSearchAdmin',
					'i18n'   => dirname( __FILE__ ) . '/Strings.php'
				]
			]
		];

		$this->styles = [
			'celersearch-admin' => [
				'src' => CELERSEARCH_PLUGIN_URL . 'assets/admin/styles.css'
			],
		];

		$this->global_styles = [];

	}


	/**
	 * Registers certain group of hooks (actions/filters)
	 * @return void
	 */
	public function register() : void {
		add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_scripts' ] );
		add_action( 'init', [ $this, 'reqister_scripts' ], 12 );
	}

	/**
	 * Enqueue scripts in admin
	 * @return void
	 */
	public function enqueue_scripts() {
		// Always enqueue global styles on all admin pages
		foreach ( $this->global_styles as $handle => $style ) {
			wp_enqueue_style( $handle );
		}

		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Just checking admin page context
		$page = isset( $_GET['page'] ) ? sanitize_text_field( wp_unslash( $_GET['page'] ) ) : '';
		if ( false === strpos( $page, 'celersearch' ) ) {
			return;
		}
		foreach ( $this->scripts as $key => $v ) {
			wp_enqueue_script( $key );
		}
		foreach ( $this->styles as $key => $v ) {
			wp_enqueue_style( $key );
		}

	}

	public function reqister_scripts(): void {
		$this->_register_scripts( $this->scripts );
		$this->_register_styles( $this->styles );
		$this->_register_styles( $this->global_styles );
	}

	/**
	 * Register scripts
	 *
	 * @param array $scripts
	 *
	 * @return void
	 */
	private function _register_scripts( $scripts ) {
		foreach ( $scripts as $handle => $script ) {
			$deps      = isset( $script['deps'] ) ? $script['deps'] : false;
			$in_footer = isset( $script['in_footer'] ) ? $script['in_footer'] : false;
			$version   = isset( $script['version'] ) ? $script['version'] : CELERSEARCH_PLUGIN_VERSION;

			wp_register_script( $handle, $script['src'], $deps, $version, $in_footer );
			if ( isset( $script['localize'] ) ) {
				$strings = include $script['localize']['i18n'];
				wp_localize_script( $handle, $script['localize']['object'], [
					'i18n' => $strings,
					'nonce' => wp_create_nonce( 'celersearch' ),
					'ajaxUrl' => admin_url( 'admin-ajax.php' ),
					'adminUrl' => admin_url(),
					'pluginUrl' => CELERSEARCH_PLUGIN_URL,
				] );
			}
		}
	}

	/**
	 * Register styles
	 *
	 * @param array $styles
	 *
	 * @return void
	 */
	public function _register_styles( $styles ) {
		foreach ( $styles as $handle => $style ) {
			$deps = isset( $style['deps'] ) ? $style['deps'] : false;

			wp_register_style( $handle, $style['src'], $deps, CELERSEARCH_PLUGIN_VERSION );
		}
	}

}