<?php
/**
 * The core plugin class.
 *
 * This is the main class that serves as the core of the plugin, orchestrating all 
 * functionality between the plugin's components and maintaining plugin state.
 *
 * @since      1.0.0
 * @package    CartShark
 * @subpackage CartShark/includes
 */

class CartShark {

    /**
     * The loader that's responsible for maintaining and registering all hooks that power
     * the plugin.
     *
     * @since    1.0.0
     * @access   protected
     * @var      CartShark_Loader    $loader    Maintains and registers all hooks for the plugin.
     */
    protected $loader;

    /**
     * The unique identifier of this plugin.
     *
     * @since    1.0.0
     * @access   protected
     * @var      string    $plugin_name    The string used to uniquely identify this plugin.
     */
    protected $plugin_name;

    /**
     * The current version of the plugin.
     *
     * @since    1.0.0
     * @access   protected
     * @var      string    $version    The current version of the plugin.
     */
    protected $version;

    /**
     * The CartShark API handler.
     *
     * @since    1.0.0
     * @access   protected
     * @var      CartShark_API    $api    Handles all API interactions.
     */
    protected $api;

    /**
     * Store active plugin integrations.
     *
     * @var array
     */
    private array $integrations = [];

    /**
     * Define the core functionality of the plugin.
     *
     * Set the plugin name and the plugin version that can be used throughout the plugin.
     * Load the dependencies, define the locale, and set the hooks for the admin area and
     * the public-facing side of the site.
     *
     * @since    1.0.0
     */
    public function __construct() {
        if ( defined( 'CARTSHARK_VERSION' ) ) {
            $this->version = CARTSHARK_VERSION;
        } else {
            $this->version = '1.0.0';
        }
        $this->plugin_name = 'cartshark';
    }

    /** 
     * Init the class
     *
     *
     * @since    1.0.0
     */
    public function init() {
        $this->load_dependencies();
        $this->define_admin_hooks();
        $this->define_public_hooks();
        $this->setup_integrations();
    }

    /**
     * Load the required dependencies for this plugin.
     *
     * Since the main plugin file already includes all the necessary files,
     * this method just initializes the loader and API components.
     *
     * @since    1.0.0
     * @access   private
     */
    private function load_dependencies() {
        // Dependencies are already loaded in the main plugin file
        // We just need to initialize our core components
        
        $this->loader = new CartShark_Loader();
        $this->api = new CartShark_API();
    }

    /**
     * Register all of the hooks related to the admin area functionality
     * of the plugin.
     *
     * @since    1.0.0
     * @access   private
     */
    private function define_admin_hooks() {

        $plugin_admin = new CartShark_Admin( $this->get_plugin_name(), $this->get_version() );

        // IMPORTANT: Add this line to handle redirects before headers are sent
        $this->loader->add_action( 'current_screen', $plugin_admin, 'handle_early_redirects' );
        
        $this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_styles' );
        $this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_scripts' );
        
        // Add menu items
        $this->loader->add_action( 'admin_menu', $plugin_admin, 'add_plugin_admin_menu' );
        
        // Settings
        $this->loader->add_action('admin_init', $plugin_admin, 'register_settings');

        // AJAX handlers for admin functions
        $plugin_admin->register_ajax_handlers();
        
    }

    /**
     * Register all of the hooks related to the public-facing functionality
     * of the plugin.
     *
     * @since    1.0.0
     * @access   private
     */
    private function define_public_hooks() {

        $plugin_public = new CartShark_Public( $this->get_plugin_name(), $this->get_version(), $this->api );

        $this->loader->add_action( 'wp_enqueue_scripts', $plugin_public, 'enqueue_styles' );
        $this->loader->add_action( 'wp_enqueue_scripts', $plugin_public, 'enqueue_scripts' );
        
        // Add the skimming detection script to the footer
        $this->loader->add_action( 'wp_head', $plugin_public, 'inject_tracker_code' );
    }

    /**
     * Set up integrations with other plugins like WooCommerce.
     *
     * @since    1.0.0
     * @access   private
     */
    private function setup_integrations() {
        require_once plugin_dir_path( __FILE__ ) . 'integrations/class-cartshark-woocommerce.php';

        if ( $this->is_woocommerce_active() ) {
            $woocommerce = new CartShark_WooCommerce();
            $woocommerce->init();
            $this->integrations[] = $woocommerce;
        }

        // Future: Add more integrations

    }

    /**
     * Run the loader to execute all of the hooks with WordPress.
     *
     * @since    1.0.0
     */
    public function run() {
        $this->loader->run();
    }

    /**
     * The name of the plugin used to uniquely identify it within the context of
     * WordPress and to define internationalization functionality.
     *
     * @since     1.0.0
     * @return    string    The name of the plugin.
     */
    public function get_plugin_name() {
        return $this->plugin_name;
    }

    /**
     * The reference to the class that orchestrates the hooks with the plugin.
     *
     * @since     1.0.0
     * @return    CartShark_Loader    Orchestrates the hooks of the plugin.
     */
    public function get_loader() {
        return $this->loader;
    }

    /**
     * Retrieve the version number of the plugin.
     *
     * @since     1.0.0
     * @return    string    The version number of the plugin.
     */
    public function get_version() {
        return $this->version;
    }


    /**
     * Check if WooCommerce is active.
     *
     * @since    1.0.0
     * @return   boolean    True if WooCommerce is active, false otherwise.
     */
    private function is_woocommerce_active() {
        include_once( ABSPATH . 'wp-admin/includes/plugin.php' );
        return is_plugin_active( 'woocommerce/woocommerce.php' );
    }

    /**
     * Get the CartShark API instance.
     *
     * @since    1.0.0
     * @return   CartShark_API    The API instance.
     */
    public function get_api() {
        return $this->api;
    }
}