<?php
/**
 * The tracker-specific functionality of the plugin.
 *
 * @since      1.0.0
 * @package    CartShark
 * @subpackage CartShark/includes
 */

class CartShark_Tracker {

    /**
     * The ID of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $plugin_name    The ID of this plugin.
     */
    private $plugin_name;

    /**
     * The version of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $version    The current version of this plugin.
     */
    private $version;

    /**
     * The API handler instance.
     *
     * @since    1.0.0
     * @access   private
     * @var      CartShark_API    $api    The API handler instance.
     */
    private $api;

    /**
     * The Option Key name
     *
     * @since    1.0.0
     * @access   private
     */
    private $option_key = 'cartshark_tracker_snippet';

    /**
     * The Option Key name for tracker account ID
     *
     * @since    1.0.0
     * @access   private
     */
    private $tracker_account_key = 'cartshark_tracker_account_id';
    
    /**
     * The Option Key name for tracker website ID
     *
     * @since    1.0.0
     * @access   private
     */
    private $tracker_website_key = 'cartshark_tracker_website_id';
    
    /**
     * Get the stored tracker snippet.
     *
     * @since    1.0.0
     * @return   string    The tracker snippet or empty string.
     */
    public function get_tracker_snippet() {
        return get_option($this->option_key, '');
    }
    
    /**
     * Check if tracker is present.
     *
     * @since    1.0.0
     * @return   bool    True if tracker exists, false otherwise.
     */
    public function is_tracker_present() {
        return !empty($this->get_tracker_snippet());
    }
    
    /**
     * Store tracker snippet along with associated account and website IDs.
     *
     * @since    1.0.0
     * @param    string    $snippet        The tracker snippet code.
     * @param    string    $account_id     The account ID used to generate this tracker.
     * @param    string    $website_id     The website ID used to generate this tracker.
     */
    public function store_tracker_snippet($snippet, $account_id = '', $website_id = '') {
        update_option($this->option_key, $snippet, false);
        
        if (!empty($account_id)) {
            update_option($this->tracker_account_key, $account_id, false);
        }
        
        if (!empty($website_id)) {
            update_option($this->tracker_website_key, $website_id, false);
        }
    }
    
    /**
     * Get the account ID that was used to generate the current tracker.
     *
     * @since    1.0.0
     * @return   string    The account ID or empty string if not found.
     */
    public function get_tracker_account_id() {
        return get_option($this->tracker_account_key, '');
    }
    
    /**
     * Get the website ID that was used to generate the current tracker.
     *
     * @since    1.0.0
     * @return   string    The website ID or empty string if not found.
     */
    public function get_tracker_website_id() {
        return get_option($this->tracker_website_key, '');
    }
    
    /**
     * Check if the stored tracker matches the current account.
     *
     * @since    1.0.0
     * @param    string    $current_account_id    The current account ID to check against.
     * @return   bool      True if tracker matches current account, false otherwise.
     */
    public function is_tracker_for_current_account($current_account_id) {
        if (empty($current_account_id)) {
            return false;
        }
        
        $stored_account_id = $this->get_tracker_account_id();
        
        // If no stored account ID, assume tracker needs refresh
        if (empty($stored_account_id)) {
            return false;
        }
        
        return $stored_account_id === $current_account_id;
    }
    
    /**
     * Check if tracker is valid for the current account and present.
     *
     * @since    1.0.0
     * @param    string    $current_account_id    The current account ID.
     * @return   bool      True if tracker is valid and present, false otherwise.
     */
    public function is_tracker_valid_for_account($current_account_id) {
        return $this->is_tracker_present() && $this->is_tracker_for_current_account($current_account_id);
    }
    
    /**
     * Insert tracker code if it's valid for the current session.
     *
     * @since    1.0.0
     * @return   string    The tracker code or empty string.
     */
    public function insert_tracker_code() {
        if (!$this->is_tracker_present()) {
            return '';
        }
        
        // Get current account ID from stored user data
        $current_account_id = get_option('cartshark_account_id', '');
        
        // Only return tracker if it matches the current account
        if ($this->is_tracker_for_current_account($current_account_id)) {
            return $this->get_tracker_snippet();
        }
        
        return '';
    }
    
    /**
     * Clear all tracker-related data.
     *
     * @since    1.0.0
     */
    public function clear_tracker_snippet() {
        delete_option($this->option_key);
        delete_option($this->tracker_account_key);
        delete_option($this->tracker_website_key);
    }
    
    /**
     * Check if tracker is enabled.
     *
     * @since    1.0.0
     * @return   bool    True if enabled, false otherwise.
     */
    public function is_tracker_enabled() {
        return get_option('cartshark_tracker_enabled', '1') === '1';
    }
}

