<?php
/**
 * Fired during plugin deactivation
 * This class defines all code necessary to run during the plugin's deactivation.
 *
 * @since      1.0.0
 * @package    CartShark
 * @subpackage CartShark/includes
 */
class CartShark_Deactivator {

    /**
     * Deactivate
     * 
     * @since    1.0.0
     */
    public static function deactivate() {
        // Clear all scheduled events
        self::clear_scheduled_events();
        
        // Clear all transients
        self::clear_transients();
        
        // Clear all plugin options
        self::clear_plugin_options();
        
        // Clear any user meta data
        self::clear_user_meta();
        
    }

    /**
     * Clear all scheduled WordPress cron events created by the plugin.
     *
     * @since    1.0.0
     * @access   private
     */
    private static function clear_scheduled_events() {

        // Clear any cron jobs that might be registered by the plugin
        // Add specific cron job names if your plugin uses them
        $cron_hooks = array(
            'cartshark_sync_data',
            'cartshark_refresh_tokens',
            'cartshark_cleanup_logs',
        );

        foreach ($cron_hooks as $hook) {
            $timestamp = wp_next_scheduled($hook);
            if ($timestamp) {
                wp_unschedule_event($timestamp, $hook);
            }
        }
    }

    /**
     * Clear all transients created by the plugin.
     *
     * @since    1.0.0
     * @access   private
     */
    private static function clear_transients() {
        // Clear authentication token transient
        delete_transient('cartshark_auth_token');
        
        // Clear any cached data transients
        delete_transient('cartshark_account_stats');
        delete_transient('cartshark_website_data');
        delete_transient('cartshark_subscription_data');
        
        // Clear any other transients that might be created
        global $wpdb;

        // Suppress the Plugin Check Linter
        // Direct DB call is safe here for one-time deactivation cleanup
        // No need for caching, as we're fetching keys for deletion only
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $transients = $wpdb->get_col(
            "SELECT option_name FROM {$wpdb->options} WHERE option_name LIKE '_transient_cartshark_%' OR option_name LIKE '_transient_timeout_cartshark_%'"
        );

        foreach ( $transients as $transient ) {
            // Strip _transient_ or _transient_timeout_ prefix
            $name = preg_replace( '/^_transient_(timeout_)?/', '', $transient );
            delete_transient( $name );
        }
    }


    /**
     * Clear all plugin options from the WordPress options table.
     *
     * @since    1.0.0
     * @access   private
     */
    private static function clear_plugin_options() {
        // Authentication and API related options
        delete_option('cartshark_public_key_encrypted');
        delete_option('cartshark_private_key_encrypted');
        delete_option('cartshark_encryption_key');
        delete_option('cartshark_user_data');
        delete_option('cartshark_account_id');
        delete_option('cartshark_website_id');
        delete_option('cartshark_api_keys_updated');
        
        // Settings options
        delete_option('cartshark_tracker_enabled');
        
        // Tracker related options
        delete_option('cartshark_tracker_snippet');
        delete_option('cartshark_tracker_version');
        delete_option('cartshark_tracker_last_updated');
        
        // Any cached data options
        delete_option('cartshark_last_sync');
        delete_option('cartshark_setup_complete');
        
        // Plugin version and activation data
        delete_option('cartshark_version');
        delete_option('cartshark_activation_date');
        delete_option('cartshark_first_activation');
        
        // Clear any other plugin-specific options
        global $wpdb;

        // Suppress the Plugin Check Linter
        // Direct DB call is safe here for one-time deactivation cleanup
        // No need for caching, as we're fetching keys for deletion only
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $options = $wpdb->get_col(
            "SELECT option_name FROM {$wpdb->options} WHERE option_name LIKE 'cartshark_%'"
        );

        foreach ( $options as $option ) {
            delete_option( $option );
        }
    }


    /**
     * Clear any user meta data created by the plugin.
     *
     * @since    1.0.0
     * @access   private
     */
    private static function clear_user_meta() {
        global $wpdb;

        // Suppress the Plugin Check Linter
        // Direct DB call is safe here for one-time deactivation cleanup
        // No need for caching, as we're fetching keys for deletion only
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $meta_keys = $wpdb->get_col(
            "SELECT DISTINCT meta_key FROM {$wpdb->usermeta} WHERE meta_key LIKE 'cartshark_%'"
        );

        if ( $meta_keys ) {
            foreach ( $meta_keys as $meta_key ) {
                delete_metadata( 'user', 0, $meta_key, '', true ); // Delete for all users
            }
        }
    }


    /**
     * Clear any capabilities or roles created by the plugin.
     *
     * @since    1.0.0
     * @access   private
     */
    private static function clear_capabilities() {
        // Remove any custom capabilities added by the plugin
        $role = get_role('administrator');
        if ($role) {
            $role->remove_cap('manage_cartshark');
        }
        
        // Remove any custom roles created by the plugin
        // remove_role('cartshark_manager');
    }

    /**
     * Clear any rewrite rules added by the plugin.
     *
     * @since    1.0.0
     * @access   private
     */
    private static function clear_rewrite_rules() {
        // Flush rewrite rules to remove any custom rules added by the plugin
        flush_rewrite_rules();
    }
}