<?php
/**
 * The API communication layer for CartShark platform interactions.
 *
 * This class handles all requests to CartShark's platform including authentication,
 * account creation, subscription management, website registration, and data retrieval.
 *
 * @since      1.0.0
 * @package    CartShark
 * @subpackage CartShark/includes
 */
class CartShark_API {

    /**
     * The API base URL.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $api_base_url    The base URL for API requests.
     */
    private $api_base_url;

    /**
     * The API version.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $api_version    The API version to use.
     */
    private $api_version;

    /**
     * The authentication token.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $auth_token    The current authentication token.
     */
    private $auth_token;

    /**
     * The Public API key.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $api_key    The Public API key for requests.
     */
    private $public_key;

    /**
     * The Private API key.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $api_key    The Private API key for requests.
     */
    private $private_key;
    
    /**
     * The Encryption key.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $api_key    The Encryption.
     */
    private $encryption_key;

    /**
     * Request timeout in seconds.
     *
     * @since    1.0.0
     * @access   private
     * @var      int    $timeout    Request timeout duration.
     */
    private $timeout;

    /**
     * Initialize the class and set its properties.
     *
     * @since    1.0.0
     */
    public function __construct() {
        $this->api_base_url = defined('CARTSHARK_API_URL') ? CARTSHARK_API_URL : 'https://api.rapidspike.com';
        $this->api_version = 'v1';
        $this->timeout = 30;
        
        // Generate or retrieve encryption key
        $this->encryption_key = $this->get_encryption_key();
        
        // Load stored credentials
        $this->load_credentials();
    }

    /**
     * *************************************************************************************************************** 
     * Endpoints
     * *************************************************************************************************************** 
     */

    /**
     * Authenticate with CartShark API using credentials
     * 
     * @since    1.0.0
     */
    public function authenticate($username, $password) {
        $endpoint = $this->build_endpoint('/users/login');
        
        $body = array(
            'email' => sanitize_email($username),
            'password' => $password
        );
        
        $response = $this->make_request('POST', $endpoint, $body);
        
        if (is_wp_error($response)) {
            return $response;
        }

        return $this->finalize_authentication($response);

    }


    /**
     * Authenticate with CartShark API using Short-lived Onboarding Token
     * 
     * @since    1.0.0
     */
    public function authenticate_with_onboarding_token($token) {

        $endpoint = $this->build_endpoint('/users/login-token-exchange');

        $response = $this->make_request('POST', $endpoint, null, array(
            'Authorization' => 'Bearer ' . sanitize_text_field($token)
        ));

        if (is_wp_error($response)) {
            return $response;
        }

        return $this->finalize_authentication($response);
    }


    /**
     * Called After Authentication, Stores Response and Retrieves API keys.
     * 
     * @since    1.0.0
     */
    private function finalize_authentication($response) {

        if (!isset($response['data']['auth']['token'])) {
            return new WP_Error('auth_failed', 'Authentication token not received');
        }

        $token = $response['data']['auth']['token'];
        $user_data = $response['data']['user'] ?? null;

        // Access check
        if (isset($user_data['brand_subscription_id']) && $user_data['brand_subscription_id'] === 0) {
            return new WP_Error('cartshark_rs_account_only', 'Login failed. This is a RapidSpike-only account; contact support@rapidspike.com if you would like access to CartShark.');
        }

        if (isset($user_data['acl']['level']) && $user_data['acl']['level'] < 30) {
            return new WP_Error('cartshark_insufficient_access', 'Login failed. Your account does not have permission to access CartShark. Please speak to your account owner for access.');
        }

        // Store data
        $this->store_auth_token($token);

        if ($user_data) {
            update_option('cartshark_user_data', $user_data, false);
        }

        if (isset($user_data['account_uuid'])) {
            update_option('cartshark_account_id', $user_data['account_uuid']);
        }

        // Get API keys
        $api_keys_result = $this->retrieve_api_keys();
        if (is_wp_error($api_keys_result)) {
            return $api_keys_result;
        }

        return $response;
    }


    /**
     * Retrieve API keys using the auth token.
     * 
     * @since    1.0.0
     */
    private function retrieve_api_keys() {

        if (empty($this->auth_token)) {
            return new WP_Error('no_auth_token', 'No authentication token available');
        }

        $endpoint = $this->build_endpoint('/accounts/api');
        
        // Step 1: Try to fetch existing API keys
        $response = $this->make_request('GET', $endpoint, null, array(
            'Authorization' => 'Bearer ' . $this->auth_token
        ));

        if (is_wp_error($response)) {
            return $response;
        }

        $public = $response['data']['public_key'] ?? null;
        $private = $response['data']['private_key'] ?? null;

        // Step 2: If keys exist, store and return
        if (!empty($public) && !empty($private)) {
            $this->store_api_keys($public, $private);
            return true;
        }

        // Step 3: If keys are missing, attempt to generate them
        $generate_response = $this->make_request('POST', $endpoint, null, array(
            'Authorization' => 'Bearer ' . $this->auth_token
        ));

        if (is_wp_error($generate_response)) {
            return $generate_response;
        }

        $new_public = $generate_response['data']['public_key'] ?? null;
        $new_private = $generate_response['data']['private_key'] ?? null;

        if (!empty($new_public) && !empty($new_private)) {
            $this->store_api_keys($new_public, $new_private);
            return true;
        }

        return new WP_Error('api_key_generation_failed', 'Unable to retrieve or generate API keys.');

    }


    /**
     * Get user's subscription plan information.
     *
     * @since    1.0.0
     * @return   array|WP_Error    Plan information or error.
     */
    public function get_subscription_plan() {
        $endpoint = $this->build_endpoint('/accounts/plan');
        return $this->make_authenticated_request('GET', $endpoint);
    }

    /**
     * Get user's registered websites.
     *
     * @since    1.0.0
     * @return   array|WP_Error    Websites array or error.
     */
    public function get_websites() {
        $endpoint = $this->build_endpoint('/websites');
        return $this->make_authenticated_request('GET', $endpoint);
    }

    /**
     * Generate a tracker code
     *
     * @since    1.0.0
     */
    public function fetch_tracker_snippet($account_id, $website_id) {
        $endpoint = $this->build_endpoint('/tracker');

        $body = array(
            'account' => $account_id,
            'website' => $website_id,
        );

        return $this->make_authenticated_request('POST', $endpoint, $body);
    }

    /**
     * Fetch high-level account stats
     * 
     * @since   1.0.0
     * */

    public function fetch_account_stats() {
        $endpoint = $this->build_endpoint('/magecartdetection/stats?days=30');
        return $this->make_authenticated_request('GET', $endpoint);
    }

    /**
     * Fetch pageviews (per site)
     * 
     * @since   1.0.0
     * */

    public function fetch_website_pageviews($websiteUuid) {
        $endpoint = $this->build_endpoint(sprintf('/websites/%s/rum/timeline?days=30&scale=days', $websiteUuid));
        return $this->make_authenticated_request('GET', $endpoint);
    }

    /**
     * Fetch endpoints list
     * 
     * @since   1.0.0
     * */

    public function fetch_account_endpoints() {
        $endpoint = $this->build_endpoint('/magecartdetection/whitelist?list=untrusted&days=30&filter=none');
        return $this->make_authenticated_request('GET', $endpoint);
    }


    /**
     * *************************************************************************************************************** 
     * API Accessors
     * *************************************************************************************************************** 
     */

    /**
     * Make the build_endpoint method public for admin class access.
     *
     * @since    1.0.0
     * @param    string    $path    API endpoint path.
     * @return   string             Full endpoint URL.
     */
    public function build_endpoint($path) {
        return trailingslashit($this->api_base_url) . $this->api_version . $path;
    }


    /**
     * Make authenticated API requests using API keys.
     * 
     * @since    1.0.0
     * @param    string         $method      HTTP method.
     */
    public function make_authenticated_request($method, $endpoint, $body = null) {
        $keys_check = $this->ensure_api_keys();
        if (is_wp_error($keys_check)) {
            return $keys_check;
        }

        $auth_params = $this->get_auth_query_params();

        // Append auth params to the endpoint URL
        $query_string = http_build_query($auth_params);
        $url_with_auth = $endpoint . (strpos($endpoint, '?') !== false ? '&' : '?') . $query_string;

        return $this->make_request($method, $url_with_auth, $body);

    }


    /**
     * Use keys to generate the query params.
     * 
     * @since    1.0.0
     * @access   private
     */
    private function get_auth_query_params() {
        $time = time();
        $package = $this->public_key . "\n" . $time;
        $signature = base64_encode(hash_hmac('sha1', $package, $this->private_key, true));

        return array(
            'public_key' => $this->public_key,
            'time' => $time,
            'signature' => $signature,
        );
    }


    /**
     * Make HTTP request to API.
     *
     * @since    1.0.0
     * @access   private
     * @param    string         $method      HTTP method.
     * @param    string         $url         API endpoint.
     * @param    array          $body        Request body.
     * @param    array          $headers     Additional headers.
     * @return   array|WP_Error              Response or error.
     */

    private function make_request($method, $url, $body = null, $headers = array()) {

        $args = array(
            'method' => $method,
            'timeout' => $this->timeout,
            'headers' => array_merge(array(
                'Content-Type' => 'application/json',
                'User-Agent' => 'CartShark-WordPress-Plugin/' . CARTSHARK_VERSION
            ), $headers)
        );

        if ($body && in_array($method, array('POST', 'PUT', 'PATCH'))) {
            $args['body'] = wp_json_encode($body);
        }

        $response = wp_remote_request($url, $args);

        if (is_wp_error($response)) {       
            return $response;
        }

        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);

        if ($response_code >= 400) {          
            $error_data = json_decode($response_body, true);
            $error_message = isset($error_data['message']) ? $error_data['message'] : 'API request failed';
            return new WP_Error('api_error', $error_message, array('response_code' => $response_code));
        }

        return json_decode($response_body, true);
    }


    /**
     * Check if user is logged in to CartShark.
     *
     * @since    1.0.0
     * @return   bool    True if logged in, false otherwise.
     */
    public function is_logged_in() {
        return $this->is_authenticated() && !empty(get_option('cartshark_user_data'));
    }


    /**
     * Logout user from CartShark.
     * No need to call a logout endpoint, just clear credentials and reset
     *
     * @since    1.0.0
     */
    public function logout() {
        $this->clear_credentials();
    }


    /**
     * *************************************************************************************************************** 
     * Getters
     * *************************************************************************************************************** 
     */


    /**
     * Get current account ID from stored user data.
     *
     * @since    1.0.0
     * @access   private
     * @return   int|null    Account ID or null if not found.
     */
    private function get_current_account_id() {
        $user_data = get_option('cartshark_user_data', array());
        return isset($user_data['account_id']) ? $user_data['account_id'] : null;
    }

    /**
     * Get stored user data.
     *
     * @since    1.0.0
     * @return   array    User data array.
     */
    public function get_user_data() {
        return get_option('cartshark_user_data', array());
    }


    /**
     * Check if current website domain is registered.
     *
     * @since    1.0.0
     * @return   bool    True if website is registered, false otherwise.
     */
    public function is_current_website_registered() {
        $current_domain = wp_parse_url(home_url(), PHP_URL_HOST);
        $websites_response = $this->get_websites();
        
        if (is_wp_error($websites_response)) {
            return false;
        }

        if (isset($websites_response['data']) && is_array($websites_response['data'])) {
            foreach ($websites_response['data'] as $website) {
                if (isset($website['domain']) && $website['domain'] === $current_domain) {
                    return true;
                }
            }
        }

        return false;
    }

    /**
     * Check if user has an active subscription.
     *
     * @since    1.0.0
     * @return   bool    True if subscription is active, false otherwise.
     */
    public function has_active_subscription() {
        $plan_response = $this->get_subscription_plan();
        
        if (is_wp_error($plan_response)) {
            return false;
        }

        return isset($plan_response['data']['plan_token']) && !empty($plan_response['data']['plan_token']);
    }



    /**
     * *************************************************************************************************************** 
     * Authentication / Validation
     * *************************************************************************************************************** 
     */

    /**
     * Store API keys (persistent, encrypted).
     * 
     * @since    1.0.0
     */
    private function store_api_keys($public_key, $private_key) {
        $this->public_key = $public_key;
        $this->private_key = $private_key;
        
        // Encrypt and store permanently
        update_option('cartshark_public_key_encrypted', $this->encrypt_data($public_key), false);
        update_option('cartshark_private_key_encrypted', $this->encrypt_data($private_key), false);
        
        // Store timestamp for potential key rotation
        update_option('cartshark_api_keys_updated', time(), false);
    }


    /**
     * Ensure we have valid API keys, refreshing if necessary.
     * 
     * @since    1.0.0
     */
    private function ensure_api_keys() {
        if ($this->is_authenticated()) {
            return true;
        }

        // If no auth token, authentication is required
        if (empty($this->auth_token)) {
            return new WP_Error('authentication_required', 'Please authenticate first');
        }

        // Try to retrieve API keys
        return $this->retrieve_api_keys();
    }


    /**
     * Check if API is authenticated.
     *
     * @since    1.0.0
     * @access   private
     * @return   bool    True if authenticated, false otherwise.
     */
    private function is_authenticated() {
        return !empty($this->public_key) && !empty($this->private_key);
    }


    /**
     * Get or generate encryption key for storing API keys securely.
     *
     * @since    1.0.0
     * @access   private  
     */
    private function get_encryption_key() {
        $key = get_option('cartshark_encryption_key');
        if (!$key) {
            $key = wp_generate_password(32, false);
            update_option('cartshark_encryption_key', $key, false);
        }
        return $key;
    }


    /**
     * Encrypt sensitive data before storage.
     *
     * @since    1.0.0
     * @access   private  
     */
    private function encrypt_data($data) {
        $cipher = "AES-256-CBC";
        $iv = openssl_random_pseudo_bytes(openssl_cipher_iv_length($cipher));
        $encrypted = openssl_encrypt($data, $cipher, $this->encryption_key, 0, $iv);
        return base64_encode($encrypted . '::' . $iv);
    }


    /**
     * Decrypt sensitive data after retrieval.
     *
     * @since    1.0.0
     * @access   private  
     */
    private function decrypt_data($data) {
        if (empty($data)) return '';
        
        $data = base64_decode($data);
        list($encrypted_data, $iv) = array_pad(explode('::', $data, 2), 2, null);
        
        if ($iv === null) return '';
        
        return openssl_decrypt($encrypted_data, "AES-256-CBC", $this->encryption_key, 0, $iv);
    }


    /**
     * Load stored credentials from WordPress options.
     *
     * @since    1.0.0
     * @access   private  
     */
    private function load_credentials() {
        // Auth token (temporary, stored as transient)
        $this->auth_token = get_transient('cartshark_auth_token');
        
        // API keys (persistent, encrypted)
        $encrypted_public = get_option('cartshark_public_key_encrypted');
        $encrypted_private = get_option('cartshark_private_key_encrypted');
        
        $this->public_key = $this->decrypt_data($encrypted_public);
        $this->private_key = $this->decrypt_data($encrypted_private);
    }

    /**
     * Store authentication token (temporary).
     *
     * @since    1.0.0
     * @access   private  
     */
    private function store_auth_token($token) {
        $this->auth_token = $token;
        // Store for 1 hour (3600 seconds). Can be up to 24h
        set_transient('cartshark_auth_token', $token, 3600);
    }


    /**
     * Clear all stored credentials (for logout/reset).
     */
    public function clear_credentials() {
        delete_transient('cartshark_auth_token');
        delete_option('cartshark_public_key_encrypted');
        delete_option('cartshark_private_key_encrypted');
        delete_option('cartshark_user_data');
        delete_option('cartshark_api_keys_updated');
        
        $this->auth_token = '';
        $this->public_key = '';
        $this->private_key = '';
    }

}