<?php
/**
 * Fired during plugin activation.
 *
 * This class defines all code necessary to run during the plugin's activation.
 * Handles default option setup and initial configuration checks for CartShark
 * web-skimming detection.
 *
 * @since      1.0.0
 * @package    CartShark
 * @subpackage CartShark/includes
 */
class CartShark_Activator {

	/**
	 * Plugin activation handler
	 *
	 * Executes all necessary tasks when the plugin is activated:
	 * - Sets default options
	 * - Performs system checks
	 * - Sets activation timestamp
	 *
	 * @since    1.0.0
	 */
	public static function activate() {
		// Set default plugin options
		self::set_default_options();
		
		// Perform system compatibility checks
		self::check_system_requirements();
		
		// Set activation timestamp
		update_option( 'cartshark_activated_time', current_time( 'timestamp' ) );
		
		// Set plugin version
		update_option( 'cartshark_version', CARTSHARK_VERSION );
		
		// Create user roles and capabilities
		self::create_user_roles();
		
		// Flush rewrite rules to ensure proper URL handling
		flush_rewrite_rules();
	}

	/**
	 * Set default plugin options
	 *
	 * @since    1.0.0
	 */
	private static function set_default_options() {
		// Core plugin settings - only set options that don't already exist
		$default_options = array(
			'cartshark_account_id' => '',
			'cartshark_website_id' => '',
			'cartshark_subscription_status' => 'inactive',
			'cartshark_tracker_status' => false,
			'cartshark_setup_complete' => false
		);

		foreach ( $default_options as $option_name => $default_value ) {
			if ( false === get_option( $option_name ) ) {
				update_option( $option_name, $default_value );
			}
		}
	}

	/**
	 * Check system requirements
	 *
	 * @since    1.0.0
	 */
	private static function check_system_requirements() {
		$requirements_met = true;
		$missing_requirements = array();

		// Check PHP version
		if ( version_compare( PHP_VERSION, '7.4', '<' ) ) {
			$requirements_met = false;
			$missing_requirements[] = sprintf(
				// Translators: %s is the PHP version number
				__( 'PHP version 7.4 or higher is required. Your version: %s', 'cartshark-security' ),
				PHP_VERSION
			);
		}

		// Check WordPress version
		global $wp_version;
		if ( version_compare( $wp_version, '5.0', '<' ) ) {
			$requirements_met = false;
			$missing_requirements[] = sprintf(
				// Translators: %s is the Wordpress version number
				__( 'WordPress version 5.0 or higher is required. Your version: %s', 'cartshark-security' ),
				$wp_version
			);
		}

		// Check for required PHP extensions
		$required_extensions = array( 'curl', 'json', 'openssl' );
		foreach ( $required_extensions as $extension ) {
			if ( ! extension_loaded( $extension ) ) {
				$requirements_met = false;
				$missing_requirements[] = sprintf(
					// Translators: %s is the PHP extension name
					__( 'PHP extension "%s" is required but not installed.', 'cartshark-security' ),
					$extension
				);
			}
		}

		// Store requirements check results
		update_option( 'cartshark_requirements_met', $requirements_met );
		update_option( 'cartshark_missing_requirements', $missing_requirements );

	}

	/**
	 * Create custom user roles and capabilities
	 *
	 * @since    1.0.0
	 */
	private static function create_user_roles() {
		// Add CartShark capabilities to administrator role
		$admin_role = get_role( 'administrator' );
		if ( $admin_role ) {
			$admin_role->add_cap( 'manage_cartshark' );
			$admin_role->add_cap( 'view_cartshark_dashboard' );
			$admin_role->add_cap( 'configure_cartshark' );
		}

		// Add view-only capabilities to shop manager role (if WooCommerce is active)
		$shop_manager_role = get_role( 'shop_manager' );
		if ( $shop_manager_role ) {
			$shop_manager_role->add_cap( 'view_cartshark_dashboard' );
		}
	}

	/**
	 * Get activation status
	 *
	 * @since    1.0.0
	 * @return   bool    True if activation was successful
	 */
	public static function is_activated() {
		return (bool) get_option( 'cartshark_activated_time', false );
	}

	/**
	 * Get missing requirements
	 *
	 * @since    1.0.0
	 * @return   array   Array of missing requirements
	 */
	public static function get_missing_requirements() {
		return get_option( 'cartshark_missing_requirements', array() );
	}

	/**
	 * Check if system requirements are met
	 *
	 * @since    1.0.0
	 * @return   bool    True if requirements are met
	 */
	public static function requirements_met() {
		return get_option( 'cartshark_requirements_met', false );
	}

	/**
	 * Check if setup is complete
	 *
	 * @since    1.0.0
	 * @return   bool    True if setup is complete
	 */
	public static function is_setup_complete() {
		return (bool) get_option( 'cartshark_setup_complete', false );
	}

	/**
	 * Check if tracker is active
	 *
	 * @since    1.0.0
	 * @return   bool    True if tracker is active
	 */
	public static function is_tracker_active() {
		return (bool) get_option( 'cartshark_tracker_status', false );
	}

	/**
	 * Get subscription status
	 *
	 * @since    1.0.0
	 * @return   string  Current subscription status
	 */
	public static function get_subscription_status() {
		return get_option( 'cartshark_subscription_status', 'inactive' );
	}
}