/**
 * CartShark Admin JavaScript
 *
 * Handles all admin panel interactions including login, logout,
 * tab switching, and account creation redirects.
 *
 * @package CartShark
 * @subpackage CartShark/admin/js
 */

(function($) {
    'use strict';

    $(document).ready(function() {
        
        /**
         * Tab switching functionality
         * 
         */
        $('.cartshark-tab-button').on('click', function() {
            var tab = $(this).data('tab');
            
            $('.cartshark-tab-button').removeClass('active');
            $('.cartshark-tab-content').removeClass('active');
            
            $(this).addClass('active');
            $('#' + tab + '-form').addClass('active');
        });

        /**
         * Login form submission
         * 
         */ 
        $('#cartshark-login-form').on('submit', function(e) {
            e.preventDefault();
            
            var $form = $(this);
            var $submit = $('#login-submit');
            var $spinner = $('#loading-spinner');
            var $messages = $('#login-messages');
            
            // Disable form and show spinner
            $submit.prop('disabled', true);
            $spinner.addClass('is-active');
            $messages.empty();
            
            // Get form data
            var email = $('#login-email').val();
            var password = $('#login-password').val();
            
            // Validate required fields
            if (!email || !password) {
                showMessage($messages, 'Please provide both email and password.', 'error');
                resetForm();
                return;
            }
            
            // Make AJAX request
            $.ajax({
                url: cartshark_ajax.ajax_url,
                type: 'POST',
                data: {
                    action: 'cartshark_login',
                    nonce: cartshark_ajax.nonce,
                    email: email,
                    password: password
                },
                success: function(response) {
                    if (response.success) {
                        showMessage($messages, response.data.message, 'success');
                        
                        // Check if redirect is needed
                        if (response.data.redirect_url) {
                            showMessage($messages, 'Redirecting to complete setup...', 'success');
                            setTimeout(function() {
                                window.location.href = response.data.redirect_url;
                            }, 1500);
                        } else {
                            // Reload page to show dashboard
                            setTimeout(function() {
                                window.location.reload();
                            }, 1500);
                        }
                    } else {
                        showMessage($messages, response.data.message, 'error');
                        resetForm();
                    }
                },
                error: function(xhr, status, error) {
                    showMessage($messages, 'An error occurred. Please try again.', 'error');
                    resetForm();
                }
            });
            
            function resetForm() {
                $submit.prop('disabled', false);
                $spinner.removeClass('is-active');
            }
        });

        /**
         * Create account button
         * 
         */ 
        $('#create-account-btn').on('click', function() {
            var $btn = $(this);
            var registerUrl = $btn.data('register-url');
            var returnUrl = encodeURIComponent($btn.data('return-url'));
            var siteDomain = encodeURIComponent($btn.data('site-domain'));
            var siteLabel = encodeURIComponent($btn.data('site-label'));
            
            var fullUrl = registerUrl + '?' + $.param({
                return_url: $btn.data('return-url'),
                site_domain: $btn.data('site-domain'),
                site_label: $btn.data('site-label'),
                source_label: 'Wordpress'
            });
            
            window.location.href = fullUrl;
        });


        /**
         * Logout button
         * 
         */ 
        $('#logout-btn').on('click', function() {

            if (!confirm('Are you sure you want to logout from CartShark?')) {
                return;
            }
            
            var $btn = $(this);
            $btn.prop('disabled', true).text('Logging out...');
            
            $.ajax({
                url: cartshark_ajax.ajax_url,
                type: 'POST',
                data: {
                    action: 'cartshark_logout',
                    nonce: cartshark_ajax.nonce
                },
                success: function(response) {
                    window.location.reload();
                },
                error: function(xhr) {
                    console.error("XHR error:", xhr.responseText); 
                    $btn.prop('disabled', false).text('Logout');
                    alert('An error occurred during logout. Please try again.');
                }
            });
        });


        /**
         * Show a message in the specified container
         *
         * @param {jQuery} $container - jQuery object of the message container
         * @param {string} message - Message to display
         * @param {string} type - Message type ('success' or 'error')
         */
        function showMessage($container, message, type) {
            var cssClass = type === 'success' ? 'cartshark-message cartshark-success' : 'cartshark-message cartshark-error';
            var messageHtml = '<div class="' + cssClass + '">' + escapeHtml(message) + '</div>';
            
            $container.html(messageHtml);
            
            // Auto-hide success messages after 5 seconds
            if (type === 'success') {
                setTimeout(function() {
                    $container.fadeOut();
                }, 5000);
            }
        }

        /**
         * Escape HTML to prevent XSS
         *
         * @param {string} text - Text to escape
         * @return {string} Escaped text
         */
        function escapeHtml(text) {
            var map = {
                '&': '&amp;',
                '<': '&lt;',
                '>': '&gt;',
                '"': '&quot;',
                "'": '&#039;'
            };
            
            return text.replace(/[&<>"']/g, function(m) {
                return map[m];
            });
        }

        // Auto-focus first input field
        $('#login-email').focus();

    })


    let currentDomain = window.cartshark_ajax.site_domain;
    
    try {
        currentDomain = new URL(currentDomain).hostname;
    } catch (e) {
        // fallback in case currentDomain isn't a full URL
        currentDomain = currentDomain.replace(/^https?:\/\//, '').split('/')[0];
    }

    document.addEventListener('DOMContentLoaded', function () {

        const statsDiv = document.getElementById('cartshark-stats');
        const loadingStatsDiv = document.getElementById('cartshark-loading-stats');
        const loadingPageviewsDiv = document.getElementById('cartshark-loading-pageviews');
        const loadingEndpointsDiv = document.getElementById('cartshark-loading-endpoints');
        const messageStatsDiv = document.getElementById('cartshark-message-ajax-stats');
        const messagePageviewsDiv = document.getElementById('cartshark-message-ajax-pageviews');
        const messageEndpointsDiv = document.getElementById('cartshark-message-ajax-endpoints');
        const endpointsDiv = document.getElementById('cartshark-endpoints');
        const pageviewsDiv = document.getElementById('cartshark-pageviews');

        if (!window.cartshark_ajax || !window.cartshark_ajax.ajax_url || !window.cartshark_ajax.nonce) {
            console.error('CartShark dashboard config missing.');
            return;
        }

        const dashboardRoot = document.querySelector('#cartshark-dashboard');
        if (!dashboardRoot) return; // Not the dashboard page, skip


        /**
         * Get Stats 
         * 
         */

        fetch(window.cartshark_ajax.ajax_url, {
            method: 'POST',
            headers: {'Content-Type': 'application/x-www-form-urlencoded'},
            body: new URLSearchParams({
                action: 'cartshark_get_stats',
                nonce: window.cartshark_ajax.nonce
            })
        })
        .then(res => res.json())
        .then(data => {

            loadingStatsDiv.style.display = 'none';

            if (data.success) {

                const stats = data.data.data.stats;

                document.getElementById('stat-untrusted').textContent = stats.total_untrusted;
                document.getElementById('stat-trusted').textContent = stats.total_trusted;
                document.getElementById('stat-pages').textContent = stats.total_account_resources.toLocaleString();

                statsDiv.style.display = 'block';

            } else {

                messageStatsDiv.style.display = 'block';
                messageStatsDiv.querySelectorAll("p")[0].textContent = 'Failed to load stats: ' + data.data;

            }
        })
        .catch(err => {
            loadingStatsDiv.textContent = 'Error loading stats.';
            console.error(err);
        });


        /**
         * Get Pageviews 
         * 
         */

        fetch(window.cartshark_ajax.ajax_url, {
            method: 'POST',
            headers: {'Content-Type': 'application/x-www-form-urlencoded'},
            body: new URLSearchParams({
                action: 'cartshark_get_pageviews',
                nonce: window.cartshark_ajax.nonce
            })
        })
        .then(res => res.json())
        .then(data => {

            loadingPageviewsDiv.style.display = 'none';

            if (data.success) {

                const labels = [];
                const pageViews = [];

                Object.values(data.data.data).forEach(entry => {
                    labels.push(entry.date);
                    pageViews.push(entry.total_requests);
                });

                const totalRequestsExist = Object.values(data.data.data).some(entry => {
                    return Number(entry.total_requests) > 0;
                });

                if (!totalRequestsExist) {
                    const messageBox = messagePageviewsDiv.querySelector('.cartshark-message');
                    messageBox.classList.remove('cartshark-error');
                    messageBox.classList.add('cartshark-warning');                    

                    messagePageviewsDiv.style.display = 'block';
                    messagePageviewsDiv.querySelectorAll("p")[0].textContent = 'No pageview data received in the last 30 days. Please allow a few minutes for a newly installed tracker to begin sending data.';
                    return;
                } 

                else {

                    pageviewsDiv.style.display = 'block';

                    const ctx = document.getElementById('pageviewsChart').getContext('2d');

                    const gradient = ctx.createLinearGradient(0, 0, 0, 250);
                    gradient.addColorStop(0, 'rgba(0, 152, 279, 0.8)'); // Top color
                    gradient.addColorStop(1, 'rgba(0, 152, 279, 0)');   // Bottom transparent

                    // Add a little breathing room, then round it
                    const maxValue = Math.max(...pageViews);
                    const paddedMax = Math.ceil(maxValue * 1.05);
                    const roundedMax = Math.ceil(paddedMax / 25) * 25;

                    new Chart(ctx, {
                        type: 'line',
                        data: {
                            labels: labels,
                            datasets: [{
                                label: 'Page Views',
                                data: pageViews,
                                fill: true,
                                backgroundColor: gradient,
                                borderColor: '#0098F9',
                                tension: 0.5,
                                pointRadius: 0,
                                pointHoverRadius: 0
                            }]
                        },
                        options: {
                            responsive: true,
                            interaction: {
                                mode: 'index',
                                intersect: false
                            },
                            plugins: {
                                legend: { display: false },
                                tooltip: {
                                    enabled: true,
                                    mode: 'nearest',
                                    intersect: false,
                                    animation: { duration: 0 },
                                    callbacks: {
                                        title: function(tooltipItems) {
                                            const rawLabel = tooltipItems[0].label;
                                            const date = new Date(rawLabel);

                                            return new Intl.DateTimeFormat('en-GB', {
                                                day: '2-digit',
                                                month: 'short',
                                                year: 'numeric'
                                            }).format(date); // e.g., "21 May 2025"
                                        }
                                    }                            
                                }
                            },
                            scales: {
                                y: {
                                    beginAtZero: true,
                                    ticks: { 
                                        color: '#ABB0B9', 
                                        maxTicksLimit: 2
                                    },
                                    max: roundedMax,
                                    title: { display: false },
                                    grid: {
                                        color: 'rgba(255,255,255,.35)'
                                    },
                                    border: {
                                        dash: [1,5],
                                    }, 
                                },
                                x: {
                                    ticks: { display: false },
                                    grid: { display: false },
                                    title: { display: false }
                                }
                            }
                        }
                    });

                }

            } else {

                messagePageviewsDiv.style.display = 'block';
                messagePageviewsDiv.querySelectorAll("p")[0].textContent = 'Failed to load pageviews';
                return;

            }

        })
        .catch(err => {
            loadingPageviewsDiv.textContent = 'Error loading pageviews.';
            console.error(err);
        });


        /**
         * Get Endpoints Data
         * 
         */

        fetch(window.cartshark_ajax.ajax_url, {
            method: 'POST',
            headers: {'Content-Type': 'application/x-www-form-urlencoded'},
            body: new URLSearchParams({
                action: 'cartshark_get_endpoints',
                nonce: window.cartshark_ajax.nonce
            })
        })
        .then(res => res.json())
        .then(data => { 

            loadingEndpointsDiv.style.display = 'none';

            if (!data.success || !Array.isArray(data.data.data.hosts)) {
                console.error('Unexpected endpoint format');
                messageEndpointsDiv.style.display = 'block';
                messageEndpointsDiv.querySelectorAll("p")[0].textContent = 'Failed to load endpoints: ' + data.data;
                return;
            }

            const matchingEndpoints = data.data.data.hosts.filter(endpoint => {
                if (!endpoint.websites) return false;

                // Get keys from the websites object
                return Object.values(endpoint.websites).some(site => {
                    try {
                        const parsed = new URL(site.domain_name);
                        return parsed.hostname === currentDomain;
                    } catch (e) {
                        return false;
                    }
                });
            });

            // Output the hostnames
            const listContainer = document.getElementById('endpoint-list');
            if (matchingEndpoints.length === 0) {
                listContainer.innerHTML = `
                    <tr>
                        <td colspan="4">
                            <div class="cartshark-message cartshark-warning">
                                <svg viewBox="0 0 32 32" xmlns="http://www.w3.org/2000/svg"><path d="M16 2.667C8.648 2.667 2.667 8.648 2.667 16S8.648 29.333 16 29.333 29.333 23.352 29.333 16 23.352 2.667 16 2.667zm4.943 16.39A1.332 1.332 0 0 1 20 21.333a1.33 1.33 0 0 1-.943-.39L16 17.885l-3.057 3.058a1.33 1.33 0 0 1-1.886 0 1.332 1.332 0 0 1 0-1.886L14.115 16l-3.058-3.057a1.332 1.332 0 0 1 0-1.886 1.332 1.332 0 0 1 1.886 0L16 14.115l3.057-3.058a1.332 1.332 0 0 1 1.886 0 1.332 1.332 0 0 1 0 1.886L17.885 16l3.058 3.057z"/></svg>
                                <p>No endpoints found for this website in last 30 days.</p>
                            </div>
                        </td>
                    </tr>`;
            } else {
                matchingEndpoints.forEach((ep, index) => {

                    let { threat, service } = extractProviderMeta(ep.provider_notes || '');

                    // Override for "Verified Safe"
                    if( ep.category_code === 1 ) threat = 'safe';

                    ep.threatMeta = assignStatus(threat);
                    ep.serviceMeta = serviceMapping[service] || null;

                    const tr = document.createElement('tr');
                    const providerNote = cleanProviderNotes(ep.provider_notes);
                    const uniqueId = `provider-note-${index}`;

                    const threatLabel = ep.threatMeta?.label || 'Unknown';
                    const threatClass = ep.threatMeta?.class || 'neutral';
                    const threatIcon = ep.threatMeta?.iconPath || null;
                    const serviceLabel = ep.serviceMeta?.short || 'Unknown';
                    const serviceIcon = ep.serviceMeta?.iconPath || null;

                    tr.innerHTML = `
                        <td>
                            <span class="threat-label ${threatClass}">
                                ${threatIcon ? `<img src="${threatIcon}" alt="${threatLabel}" class="threat-icon">` : ''} ${threatLabel}
                            </span>
                        </td>
                        <td>
                            <h5 class="gd gd--align-center gd--gap-5">
                                ${ep.host}
                            </h5>
                        </td>
                        <td>
                            <span class="service">
                                ${serviceIcon ? `<img src="${serviceIcon}" alt="${serviceLabel}" class="service-icon">` : ''}
                                <span class="service-label">${serviceLabel}</span>
                            </span>
                        </td>
                        <td style="text-align:right;">
                            ${providerNote ? `<button class="button button-primary button-small toggle-note" data-target="${uniqueId}">Details \u25BC</button>` : ''}
                        </td>
                    `;

                    listContainer.appendChild(tr);

                    // Collapsible row for provider notes
                    if (providerNote) {
                        const noteTr = document.createElement('tr');
                        noteTr.id = uniqueId;
                        noteTr.classList.add('provider-note-row');
                        noteTr.style.display = 'none';
                        noteTr.innerHTML = `
                            <td colspan="4">
                                <div class="provider-note"><p>${providerNote.replace(/\n/g, '<br>')}</p></div>
                            </td>
                        `;
                        listContainer.appendChild(noteTr);
                    }

                });
            }

            endpointsDiv.style.display = 'block';


            /**
             * Toggle functionality for displaying notes 
             */
            document.querySelectorAll('.toggle-note').forEach(btn => {
                btn.addEventListener('click', () => {

                    const targetId = btn.getAttribute('data-target');
                    const noteRow = document.getElementById(targetId);
                    if (noteRow.style.display === 'none') {
                        noteRow.style.display = '';
                        btn.textContent = 'Hide \u25B2';
                    } else {
                        noteRow.style.display = 'none';
                        btn.textContent = 'Details \u25BC';
                    }
                });
            });

        })
        .catch(err => {
            loadingEndpointsDiv.textContent = 'Error loading endpoints.';
            console.error(err);
        });

    });


    /**
     * Extract variables from provider notes
     * 
     * @param {string} - notes from endpoint data
     */
    function extractProviderMeta(providerNotes) {
        if (typeof providerNotes !== 'string') return {};

        const threatMatch = providerNotes.match(/\{\{threatAssessment\}\}\s*(.+)/i);
        const serviceMatch = providerNotes.match(/\{\{service\}\}\s*(.+)/i);

        const threat = threatMatch ? threatMatch[1].trim() : null;
        const service = serviceMatch ? serviceMatch[1].trim() : null;

        return { threat, service };
    }


    /**
     * Remove the variables from provider notes
     * 
     * @param {string} - notes from endpoint data
     */
    function cleanProviderNotes(notes) {
        if (!notes) return '';
        return notes
            .replace(/{{[^}]+}}\s*[^({{}})]+/g, '') // remove {{key}} value
            .replace(/\n{2,}/g, '\n')               // collapse extra newlines
            .trim();
    }


    /**
     * Handle "Status" of Threats
     * 
     * @param {string} - threat label
     */
    const assignStatus = (threat) => {
        const loweredThreat = (typeof threat === 'string') ? threat.toLowerCase().trim() : threat;

        const statuses = {
            'safe': { 
                label: 'Verified Safe', 
                class: 'success',
                iconPath: iconBasePath + 'CheckIcon.svg', 
            },
            'minimal risk': { 
                label: 'Minimal Risk', 
                class: 'info',
                iconPath: iconBasePath + 'InfoBlueIcon.svg', 
            },
            'low risk': { 
                label: 'Low Risk', 
                class: 'caution',
                iconPath: iconBasePath + 'InfoYellowIcon.svg', 
            },
            'medium risk': { 
                label: 'Medium Risk', 
                class: 'warning',
                iconPath: iconBasePath + 'ExclaimationIcon.svg', 
            },
            'high risk': { 
                label: 'High Risk', 
                class: 'danger',
                iconPath: iconBasePath + 'CrossIcon.svg', 
            },
            'magecart attack': { 
                label: 'Known Threat', 
                class: 'danger',
                iconPath: iconBasePath + 'CrossIcon.svg', 
            },
            'unknown': { 
                label: 'Unknown', 
                class: 'neutral', 
                iconPath: iconBasePath + 'InfoGreyIcon.svg', 
            }
        };

        return statuses[loweredThreat] || statuses['unknown'];
    };


    /**
     * Services Data Object
     * 
     */

    const iconBasePath = window.cartshark_ajax.iconBasePath;

    const serviceMapping = {
        'Analytics & Tracking': {
            label: 'Analytics & Tracking',
            short: 'Analytics',
            iconPath: iconBasePath + 'DocumentIcon.svg',
        },
        'Advertising': {
            label: 'Advertising',
            short: 'Advertising',
            iconPath: iconBasePath + 'MegaphoneIcon.svg',
        },
        'Content Delivery Networks (CDN)': {
            label: 'Content Delivery Networks (CDN)',
            short: 'CDN',
            iconPath: iconBasePath + 'CloudIcon.svg',
        },
        'Social Media Integration': {
            label: 'Social Media Integration',
            short: 'Social Media',
            iconPath: iconBasePath + 'ConversationIcon.svg',
        },
        'Live Chat & Customer Support': {
            label: 'Live Chat & Customer Support',
            short: 'Support',
            iconPath: iconBasePath + 'HeadsetIcon.svg',
        },
        'Marketing Automation': {
            label: 'Marketing Automation',
            short: 'Marketing',
            iconPath: iconBasePath + 'BoltIcon.svg',
        },
        'Payment Processing': {
            label: 'Payment Processing',
            short: 'Payments',
            iconPath: iconBasePath + 'CoinIcon.svg',
        },
        'Security & Anti-fraud': {
            label: 'Security & Anti-fraud',
            short: 'Security',
            iconPath: iconBasePath + 'ShieldIcon.svg',
        },
        'SEO Tools': {
            label: 'SEO Tools',
            short: 'SEO',
            iconPath: iconBasePath + 'SearchEngineIcon.svg',
        },
        'Comments & Discussion': {
            label: 'Comments & Discussion',
            short: 'Comments',
            iconPath: iconBasePath + 'CommentIcon.svg',
        },
        'A/B Testing & Personalisation': {
            label: 'A/B Testing & Personalisation',
            short: 'Testing',
            iconPath: iconBasePath + 'FlaskIcon.svg',
        },
        'Forms & Surveys': {
            label: 'Forms & Surveys',
            short: 'Forms',
            iconPath: iconBasePath + 'ChecklistIcon.svg',
        },
        'Ecommerce': {
            label: 'Ecommerce',
            short: 'Ecommerce',
            iconPath: iconBasePath + 'ShoppingCartIcon.svg',
        },
        'Entertainment': {
            label: 'Entertainment',
            short: 'Entertainment',
            iconPath: iconBasePath + 'StarIcon.svg',
        },
        'Cloud Services': {
            label: 'Cloud Services',
            short: 'Cloud',
            iconPath: iconBasePath + 'CloudIcon.svg',
        },
        'Other / Unknown': {
            label: 'Other / Unknown',
            short: 'Other',
            iconPath: iconBasePath + 'CircleQuestionMarkIcon.svg',
        },
    }


})(jQuery);