<?php

/**
 * The admin-specific functionality of the plugin.
 *
 * @since      1.0.0
 * @package    CartShark
 * @subpackage CartShark/admin
 */

class CartShark_Admin {

    /**
     * The ID of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $plugin_name    The ID of this plugin.
     */
    private $plugin_name;

    /**
     * The version of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $version    The current version of this plugin.
     */
    private $version;

    /**
     * The API instance.
     *
     * @since    1.0.0
     * @access   private
     * @var      CartShark_API    $api    The API communication layer.
     */
    private $api;

    /**
     * The tracker.
     *
     * @since    1.0.0
     * @access   private
     * @var      CartShark_Tracker    $tracker    The tracker class.
     */
    private $tracker;

    /**
     * The admin templates path.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $templates_path    Path to admin templates.
     */
    private $templates_path;


    /**
     * Initialize the class and set its properties.
     *
     * @since    1.0.0
     * @param      string    $plugin_name       The name of this plugin.
     * @param      string    $version    The version of this plugin.
     */
    public function __construct($plugin_name, $version) {
        $this->plugin_name = $plugin_name;
        $this->version = $version;
        $this->api = new CartShark_API();
        $this->tracker = new CartShark_Tracker();
        $this->templates_path = plugin_dir_path(__FILE__) . 'partials/';
    }


    /**
     * Handle early redirects before headers are sent.
     * This should be called on 'admin_init' hook.
     *
     * @since    1.0.0
     */
    public function handle_early_redirects() {

        // Only handle redirects on our admin pages
        if (!isset($_GET['page']) || $_GET['page'] !== 'cartshark') {
            return;
        }

        // Unslash and sanitize nonce once (if set)
        $nonce = isset($_GET['_wpnonce']) ? sanitize_text_field(wp_unslash($_GET['_wpnonce'])) : '';

        // Handle return from onboarding
        if (
            isset($_GET['action']) &&
            $_GET['action'] === 'complete-onboarding' &&
            isset($_GET['token']) &&
            !empty($nonce) &&
            wp_verify_nonce($nonce, 'cartshark_onboarding')
        ) {
            $token = sanitize_text_field(wp_unslash($_GET['token']));
            $this->handle_onboarding_return($token);
            return;
        }

        // Redirect to SSO into the app, handle start of onboarding (also behind a nonce)
        if (
            isset($_GET['action']) &&
            $_GET['action'] === 'start-onboarding' &&
            isset($_GET['type']) &&
            !empty($nonce) &&
            wp_verify_nonce($nonce, 'cartshark_onboarding')
        ) {
            $this->redirect_to_onboarding();
            return;
        }

    }


    /**
     * Register the stylesheets for the admin area.
     *
     * @since    1.0.0
     */
    public function enqueue_styles() {
        wp_enqueue_style(
            $this->plugin_name, 
            plugin_dir_url(__FILE__) . 'css/cartshark-admin.css', 
            array(), 
            $this->version, 
            'all'
        );

        wp_enqueue_style(
            'cartshark_google_fonts',
            'https://fonts.googleapis.com/css2?family=Karla:wght@400;700&family=Montserrat:wght@400;700&display=swap',
            array(),
            '1.0',
            'all'
        );        
    }


    /**
     * Register the JavaScript for the admin area.
     *
     * @since    1.0.0
     */
    public function enqueue_scripts() {

        wp_enqueue_script(
            'chartjs',
            plugin_dir_url(__FILE__) . 'js/chart.js', // latest v4
            [],
            '4.5.0',
            true
        );

        wp_enqueue_script(
            $this->plugin_name, 
            plugin_dir_url(__FILE__) . 'js/cartshark-admin.js', 
            array('jquery'), 
            $this->version, 
            false
        );
        
        // Localize script for AJAX
        wp_localize_script($this->plugin_name, 'cartshark_ajax', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('cartshark_nonce'),
            'cartshark_url' => defined('CARTSHARK_APP_URL') ? CARTSHARK_APP_URL : 'https://cartshark.rapidspike.com',
            'site_url' => home_url(),
            'site_domain' => wp_parse_url(home_url(), PHP_URL_HOST),
            'site_label' => get_bloginfo('name'),
            'iconBasePath' => plugins_url('icons/', __FILE__)
        ));
    }


    /**
     * Add admin menu pages.
     *
     * @since    1.0.0
     */
    public function add_plugin_admin_menu() {

        $svg_icon = '<svg fill="currentColor" style="color:rgba(240, 246, 252, 0.6);" viewBox="0 0 22 22" xmlns="http://www.w3.org/2000/svg"><path fill="currentColor" fill-rule="evenodd" clip-rule="evenodd" d="M9.99998 0C4.45652 0 0 4.42524 0 9.9298C0 14.8947 3.58695 18.9961 8.36955 19.7517C8.04346 18.9961 7.82606 17.8089 7.49998 17.0534C6.19563 14.463 4.7826 11.9805 4.7826 10.0377C4.7826 9.06634 5.10869 8.09495 5.54347 7.23149C5.10869 6.58389 3.47826 5.39663 3.47826 5.39663C3.47826 5.39663 4.6739 4.85697 7.0652 5.07284C8.26085 3.88558 9.78258 3.13005 10.8695 3.02211C12.7174 2.69832 14.4565 2.91418 15.1087 4.10144C15.2174 4.31731 14.0217 5.9363 14.0217 5.9363C14.0217 5.9363 13.6956 5.50457 12.7174 5.72043C11.8478 5.82836 11.0869 6.47596 11.0869 6.47596C11.0869 6.47596 11.8478 6.36803 12.5 6.58389C13.0434 6.69183 13.0434 6.90769 13.0434 6.90769C12.7173 7.55529 11.413 8.41875 10.3261 9.39014C10.6521 9.71394 11.0869 10.6853 10.8695 11.7647C9.99997 11.225 9.67389 10.7933 9.13041 10.6853C7.49998 12.7361 7.39128 14.5709 8.04345 16.5137C8.26085 17.3772 9.3478 16.7296 11.5217 16.082C13.2608 15.5423 15.1087 16.1899 14.8913 16.2978C12.2826 16.9454 11.5217 17.2692 10.6521 18.2406C10.2174 18.6723 10.1087 19.3199 10.1087 19.8596C15.5434 19.7517 19.9999 15.3264 19.9999 9.9298C19.9999 4.42524 15.4347 0 9.99998 0ZM9.78258 9.39014C9.99997 9.17428 9.56519 9.17428 8.69563 9.17428C7.60868 9.17428 6.73911 10.2536 6.73911 10.2536C6.73911 10.2536 7.82607 10.2536 8.36955 10.7933C9.13042 10.4695 9.2391 9.92981 9.78258 9.39014Z"/></svg>';
        
        $icon_data_uri = 'data:image/svg+xml;base64,' . base64_encode($svg_icon);
        
        add_menu_page(
            'CartShark',
            'CartShark',
            'manage_options',
            'cartshark',
            array($this, 'display_admin_page'),
            $icon_data_uri,
            30
        );

        add_submenu_page(
            'cartshark',                 // Parent slug
            'CartShark Dashboard',       // Page title
            'Dashboard',                 // Menu title
            'manage_options',
            'cartshark',                 // Same slug as top-level
            array($this, 'display_admin_page')
        );

        add_submenu_page(
            'cartshark',
            'CartShark Settings',
            'Settings',
            'manage_options',
            'cartshark-settings',
            array($this, 'display_settings_page')
        );

    }


    /**
     * Settings: Tracker
     *
     * @since    1.0.0
     */
    public function register_settings() {
        register_setting('cartshark_settings_group', 'cartshark_tracker_enabled', [
            'sanitize_callback' => function( $value ) {
                return $value === '1' ? '1' : '0';
            },
            'default'           => false,
        ]);      
    }    


    /**
     * Display the main admin page based on authentication status.
     *
     * @since    1.0.0
     */
    public function display_admin_page() {

        // Note: Early redirects are now handled in handle_early_redirects()
        // This method only handles page display logic

        // Check authentication status
        if ($this->api->is_logged_in()) {

            // Check if user needs onboarding
            $onboarding_check = $this->check_onboarding_status();
            
            if (is_wp_error($onboarding_check)) {
                $this->display_error_page($onboarding_check->get_error_message());
                return;
            }

            if ($onboarding_check['needs_onboarding']) {
                $this->display_onboarding_pending_page($onboarding_check['onboarding_type']);
                return;
            }

            // Check if tracker is installed / Validate the tracker is correct for the current account

            // Get current account ID (this should be fresh from authentication)
            $current_account_id = get_option('cartshark_account_id', '');
            
            if (empty($current_account_id)) {
                // Handle case where account ID is not available
                $tracker_account_error = "CartShark: No account ID available";
                $this->display_error_page($tracker_account_error);
                return;
            }

            // Check if we need to fetch/refresh the tracker
            if (!$this->tracker->is_tracker_valid_for_account($current_account_id)) {
                
                // We need to fetch a new tracker
                $this->fetch_and_store_tracker($current_account_id, $this->tracker);
                
            }

            // User is authenticated and onboarded - show dashboard
            $this->display_dashboard();
        } else {
            // User needs to login or create account
            $this->display_auth_page();
        }
    }


    /**
     * Fetch and store a new tracker for the current account.
     *
     * @since    1.0.0
     * @param    string              $account_id    The current account ID.
     * @param    CartShark_Tracker   $tracker       The tracker instance.
     */
    private function fetch_and_store_tracker($account_id, $tracker) {
        
        // Get the website ID for the current domain
        $website_id = get_option('cartshark_website_id') ?? null;
        
        if (empty($website_id)) {
            return;
        }
        
        // Fetch new tracker from API
        $tracker_response = $this->api->fetch_tracker_snippet($account_id, $website_id);
        
        if (is_wp_error($tracker_response)) {
            $tracker_missing_error = 'CartShark: Failed to fetch tracker snippet: ' . $tracker_response->get_error_message();
            $this->display_error_page($tracker_missing_error);
            return;
        }
        
        // Extract tracker code from response
        $tracker_code = isset($tracker_response['data']) ? $tracker_response['data'] : '';
        
        if (!empty($tracker_code)) {
            
            // Store the new tracker with account and website IDs
            $tracker->store_tracker_snippet($tracker_code, $account_id, $website_id);

        }
    }


    /**
     * Display the authentication page (login/register).
     *
     * @since    1.0.0
     */
    private function display_auth_page() {

        $nonce = wp_create_nonce('cartshark_onboarding');

        $template_data = array(
            'cartshark_url' => defined('CARTSHARK_APP_URL') ? CARTSHARK_APP_URL : 'https://cartshark.rapidspike.com',
            'site_domain' => wp_parse_url(home_url(), PHP_URL_HOST),
            'site_label' => get_bloginfo('name'),
            'return_url' => add_query_arg(array(
                'page' => 'cartshark',
                'action' => 'complete-onboarding',
                '_wpnonce' => $nonce
            ), admin_url('admin.php'))
        );
        
        $this->load_template('cartshark-admin-authentication.php', $template_data);
    }


    /**
     * Display the Pending page
     *
     * @since    1.0.0
     */
    private function display_onboarding_pending_page($onboarding_type) {

        $nonce = wp_create_nonce('cartshark_onboarding');

        $template_data = array(
            'onboarding_type' => $onboarding_type,
            'continue_url' => add_query_arg(array(
                'action' => 'start-onboarding',
                'type' => $onboarding_type,
                '_wpnonce' => $nonce,
            ), admin_url('admin.php?page=cartshark')),
            'logout_url' => wp_logout_url(admin_url('admin.php?page=cartshark'))
        );

        $this->load_template('cartshark-onboarding-pending.php', $template_data);
    }


    /**
     * Display the dashboard for authenticated users.
     *
     * @since    1.0.0
     */
    private function display_dashboard() {
        $user_data = $this->api->get_user_data();
        $template_data = array(
            'user_authenticated' => true,
            'user_data' => $user_data,
            'tracker_enabled' => $this->tracker->is_tracker_enabled(),
            'cartshark_url' => defined('CARTSHARK_APP_URL') ? CARTSHARK_APP_URL : 'https://cartshark.rapidspike.com'
        );
        
        $this->load_template('cartshark-admin-dashboard.php', $template_data);
    }


    /**
     * Display the Settings page.
     *
     * @since    1.0.0
     */
    public function display_settings_page() {

        if ($this->api->is_logged_in()) {
              
            $template_data = array(
                'tracker_enabled' => $this->tracker->is_tracker_enabled(),
                'cartshark_url' => defined('CARTSHARK_APP_URL') ? CARTSHARK_APP_URL : 'https://cartshark.rapidspike.com'
            );
            
            $this->load_template('cartshark-admin-settings.php', $template_data);

        } else {
            $this->display_auth_page(); // Redirect or display login page
        }

    }


    /**
     * Display error page.
     *
     * @since    1.0.0
     * @param    string    $message    Error message to display.
     */
    private function display_error_page($message) {
        $template_data = array(
            'error_message' => $message,
            'retry_url' => admin_url('admin.php?page=cartshark')
        );
        
        $this->load_template('cartshark-admin-error.php', $template_data);
    }


    /**
     * Load a template file with given data.
     *
     * @since    1.0.0
     * @param    string    $template    Template filename.
     * @param    array     $data        Data to pass to template.
     */
    private function load_template($template, $data = array()) {
        $template_path = $this->templates_path . $template;
        
        if (file_exists($template_path)) {
            // Extract data array to variables
            extract($data);
            include $template_path;
        } else {
            echo '<div class="wrap"><h1>CartShark</h1><div class="notice notice-error"><p>Template file not found: ' . esc_html($template) . '</p></div></div>';
        }
    }


    /**
     * Check if user needs onboarding (subscription or website setup).
     *
     * @since    1.0.0
     * @return   array|WP_Error    Onboarding status or error.
     */
    private function check_onboarding_status() {
        $current_domain = wp_parse_url(home_url(), PHP_URL_HOST);
        $current_normalized = $this->normalize_domain($current_domain);
        
        // Check subscription status
        $subscription_response = $this->api->make_authenticated_request('GET', $this->api->build_endpoint('/accounts/plan'));

        if (is_wp_error($subscription_response)) {
            return $subscription_response;
        }

        $has_subscription = isset($subscription_response['data']['plan']['token']) 
    && $subscription_response['data']['plan']['token'] !== 'pending';

        // Fetch websites: Check if website exists in CS
        $websites_response = $this->api->make_authenticated_request('GET', $this->api->build_endpoint('/websites'));
        
        if (is_wp_error($websites_response)) {
            return $websites_response;
        }

        $has_website = false;
        if (isset($websites_response['data']['websites']) && is_array($websites_response['data']['websites'])) {
            foreach ($websites_response['data']['websites'] as $website) {
                
                if (!isset($website['website']['domain_name'])) {
                    continue;
                }

                $website_normalized = $this->normalize_domain($website['website']['domain_name']);

                if ($website_normalized === $current_normalized) {
                    $has_website = true;
                    // Store the Website UUID locally
                    if (!empty($website['website']['uuid'])) {
                        update_option('cartshark_website_id', $website['website']['uuid']);
                    }
                    break;
                }
            }
        }

        // Determine onboarding needs
        $needs_onboarding = !$has_subscription || !$has_website;
        $onboarding_type = '';

        if (!$has_subscription && !$has_website) {
            $onboarding_type = 'full';
        } elseif (!$has_subscription) {
            $onboarding_type = 'subscription';
        } elseif (!$has_website) {
            $onboarding_type = 'website';
        }

        return array(
            'needs_onboarding' => $needs_onboarding,
            'onboarding_type' => $onboarding_type,
            'has_subscription' => $has_subscription,
            'has_website' => $has_website
        );
    }


    /**
     * Redirect user to CartShark app for onboarding.
     *
     * @since    1.0.0
     * @param    string    $onboarding_type    Type of onboarding needed.
     */
    private function redirect_to_onboarding() {

        // Generate temp token and redirect URL for onboarding
        $temp_token_response = $this->api->make_authenticated_request('POST', $this->api->build_endpoint('/users/login-sso'));

        if (is_wp_error($temp_token_response)) {
            $this->display_error_page('Unable to generate secure token for onboarding. Please try again.');
            return;
        }

        if (!isset($temp_token_response['data']['auth']['token'])) {
            $this->display_error_page('Invalid response from server. Please try again.');
            return;
        }

        if (!is_wp_error($temp_token_response) && isset($temp_token_response['data']['auth']['token'])) {

            if (!is_user_logged_in()) {
                wp_die('User not authenticated. Cannot generate nonce.');
            }

            $temp_token = $temp_token_response['data']['auth']['token'];

            $nonce = wp_create_nonce('cartshark_onboarding');

            $return_url = add_query_arg(array(
                'page' => 'cartshark',
                'action' => 'complete-onboarding',
                '_wpnonce' => $nonce
            ), admin_url('admin.php'));

            $site_domain = home_url('/');
            $site_label = get_bloginfo('name');
            $cartshark_url = defined('CARTSHARK_APP_URL') ? CARTSHARK_APP_URL : 'https://cartshark.rapidspike.com';

            $onboarding_url = $cartshark_url . '/platform-onboarding?' . http_build_query(array(
                'temp_token' => $temp_token,
                'return_url' => $return_url,
                'site_label' => $site_label,
                'site_domain' => $site_domain,
                'source_label' => 'WordPress'
            ));

            wp_redirect($onboarding_url);

        }

        exit;
    }


    /**
     * Handle return from onboarding process.
     *
     * @since    1.0.0
     */
    private function handle_onboarding_return($token) {

        $auth_response = $this->api->authenticate_with_onboarding_token($token);

        if (is_wp_error($auth_response)) {
            $this->display_error_page("Login failed after onboarding: " . $auth_response->get_error_message());
            return;
        }

        wp_redirect(admin_url('admin.php?page=cartshark'));
        exit;        

    }


    /**
     * Handle AJAX login request.
     *
     * @since    1.0.0
     */
    public function ajax_login() {
        
        // Check nonce is set and valid
        if ( ! isset( $_POST['nonce'] ) || 
             ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'cartshark_nonce' ) ) {
            wp_send_json_error( 'Invalid nonce', 400 );
        }

        // Sanitize and validate email
        $email = isset( $_POST['email'] ) ? sanitize_email( wp_unslash( $_POST['email'] ) ) : '';

        if (!is_email($email)) {
            wp_send_json_error(array('message' => 'Please provide a valid email address.'));
        }
        
        // Validate password
        // Passwords should not be sanitized using sanitize_text_field() or similar functions,
        // as they may alter the input (e.g., strip characters), leading to authentication issues. 
        // Instead, we unslash and perform specific validation checks below.
        // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
        $password = isset( $_POST['password'] ) ? wp_unslash( $_POST['password'] ) : '';
        
        // Basic password validation
        if (empty($password)) {
            wp_send_json_error(array('message' => 'Please provide both email and password.'));
        }
        
        // Validate password constraints (adjust as needed)
        if (strlen($password) < 6) {
            wp_send_json_error(array('message' => 'Password must be at least 6 characters long.'));
        }
        
        if (strlen($password) > 256) {
            wp_send_json_error(array('message' => 'Password is too long.'));
        }
        
        // Additional security: Check for common injection patterns
        if (preg_match('/[<>"\']/', $password)) {
            wp_send_json_error(array('message' => 'Password contains invalid characters.'));
        }

        if (preg_match('/[\r\n\t]/', $password)) {
            wp_send_json_error(array('message' => 'Password contains invalid whitespace characters.'));
        }        
        
        // Check if both fields are provided after validation
        if (empty($email) || empty($password)) {
            wp_send_json_error(array('message' => 'Please provide both email and password.'));
        }
        
        // Rate limiting check
        $this->check_login_rate_limit($email);
        
        // Attempt authentication
        $auth_result = $this->api->authenticate($email, $password);
        
        if (is_wp_error($auth_result)) {
            // Log failed login attempt
            wp_send_json_error(array('message' => $auth_result->get_error_message()));
        }
        
        // Check if onboarding is needed
        $onboarding_check = $this->check_onboarding_status();
        
        if (is_wp_error($onboarding_check)) {
            wp_send_json_error(array('message' => 'Login successful, but unable to verify account status.'));
        }
        
        if ($onboarding_check['needs_onboarding']) {

            $nonce = wp_create_nonce('cartshark_onboarding');

            $redirect_url = add_query_arg(array(
                'page' => 'cartshark',
                'action' => 'start-onboarding',
                'type' => $onboarding_check['onboarding_type'],
                '_wpnonce' => $nonce,
            ), admin_url('admin.php'));

            wp_send_json_success(array(
                'message' => 'Login successful! Redirecting to complete setup...',
                'redirect' => $redirect_url
            ));

            return;
        }

        // Successful login
        wp_send_json_success(array('message' => 'Login successful!'));
    }


    /**
     * Check login rate limiting to prevent brute force attacks
     *
     * @param string $email
     */
    private function check_login_rate_limit($email) {
        $transient_key = 'cartshark_login_attempts_' . md5($email);
        $attempts = get_transient($transient_key);
        
        if ($attempts && $attempts >= 5) {
            wp_send_json_error(array('message' => 'Too many login attempts. Please try again later.'));
        }
        
        // Increment attempts
        $attempts = $attempts ? $attempts + 1 : 1;
        set_transient($transient_key, $attempts, 300); // 5 minutes lockout
    }


    /**
     * Handle AJAX logout request.
     *
     * @since    1.0.0
     */
    public function ajax_logout() {

        // Check nonce is set and valid
        if ( ! isset( $_POST['nonce'] ) || 
             ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'cartshark_nonce' ) ) {
            wp_send_json_error( 'Invalid nonce', 400 );
        }

        $this->api->logout();
        wp_send_json_success(array('message' => 'Logged out successfully.'));
    }


    /**
     * Handle AJAX stats call.
     *
     * @since    1.0.0
     */
    public function ajax_get_stats() {

        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized', 403);
        }

        // Check nonce is set and valid
        if ( ! isset( $_POST['nonce'] ) || 
             ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'cartshark_nonce' ) ) {
            wp_send_json_error( 'Invalid nonce', 400 );
        }

        $stats = $this->api->fetch_account_stats();

        if (is_wp_error($stats)) {
            wp_send_json_error($stats->get_error_message());
        }

        wp_send_json_success($stats);
    }


    /**
     * Handle AJAX website pageview.
     *
     * @since    1.0.0
     */
    public function ajax_get_pageviews() {

        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized', 403);
        }

        // Check nonce is set and valid
        if ( ! isset( $_POST['nonce'] ) || 
             ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'cartshark_nonce' ) ) {
            wp_send_json_error( 'Invalid nonce', 400 );
        }


        // Get the locally stored website UUID
        $websiteUuid = get_option('cartshark_website_id');

        if (empty($websiteUuid)) {
            wp_send_json_error('Website UUID not configured', 400);
        }        

        $views = $this->api->fetch_website_pageviews($websiteUuid);

        if (is_wp_error($views)) {
            wp_send_json_error($views->get_error_message());
        }

        wp_send_json_success($views);
    }


    /**
     * Handle AJAX endpoints call.
     *
     * @since    1.0.0
     */
    public function ajax_get_endpoints() {

        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized', 403);
        }

        // Check nonce is set and valid
        if ( ! isset( $_POST['nonce'] ) || 
             ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'cartshark_nonce' ) ) {
            wp_send_json_error( 'Invalid nonce', 400 );
        }

        $stats = $this->api->fetch_account_endpoints();

        if (is_wp_error($stats)) {
            wp_send_json_error($stats->get_error_message());
        }

        wp_send_json_success($stats);
    }


    /**
     * Generate short-token and log in
     *
     * @since    1.0.0
     */
    public function handle_sso_token_redirect() {

        if (
            ! current_user_can( 'manage_options' ) ||
            ! isset( $_GET['_wpnonce'] ) ||
            ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['_wpnonce'] ) ), 'cartshark_onboarding' )
        ) {
            wp_die( 'Unauthorized request' );
        }

        $token_response = $this->api->make_authenticated_request('POST', $this->api->build_endpoint('/users/login-sso'));

        if (is_wp_error($token_response) || !isset($token_response['data']['auth']['token'])) {
            wp_die('Failed to generate login token. Please try again.');
        }

        $token = $token_response['data']['auth']['token'];
        $cartshark_url = defined('CARTSHARK_APP_URL') ? CARTSHARK_APP_URL : 'https://cartshark.rapidspike.com';
        $redirect_url = $cartshark_url . '/platform-sso?' . http_build_query([
            'token' => $token,
        ]);

        wp_redirect($redirect_url);
        exit;
    }


    /**
     * Register AJAX handlers.
     *
     * @since    1.0.0
     */
    public function register_ajax_handlers() {
        add_action('wp_ajax_cartshark_login', array($this, 'ajax_login'));
        add_action('wp_ajax_cartshark_logout', array($this, 'ajax_logout'));

        add_action('wp_ajax_cartshark_get_stats', array($this, 'ajax_get_stats'));
        add_action('wp_ajax_cartshark_get_pageviews', array($this, 'ajax_get_pageviews'));
        add_action('wp_ajax_cartshark_get_endpoints', array($this, 'ajax_get_endpoints'));
        add_action('admin_post_cartshark_generate_sso_token', array($this, 'handle_sso_token_redirect'));
    }


    /**
     * Tidy domains for matching
     *
     * @since    1.0.0
     */
    private function normalize_domain($url) {
        // Add scheme if missing
        if (!preg_match('#^https?://#i', $url)) {
            $url = 'https://' . $url;
        }

        $host = wp_parse_url($url, PHP_URL_HOST);
        if (!$host) {
            return null;
        }

        // Remove 'www.' prefix for comparison
        return preg_replace('/^www\./', '', strtolower($host));
    }
}