<?php
/**
 * Frontend handler for enqueueing checkout protection scripts.
 *
 * @package Carticy\CheckoutShield\Frontend
 */

declare(strict_types=1);

namespace Carticy\CheckoutShield\Frontend;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Handles frontend script enqueuing for checkout pages.
 */
final class FrontendHandler {

	/**
	 * Register hooks.
	 */
	public function register(): void {
		add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_checkout_scripts' ) );
	}

	/**
	 * Enqueue scripts on checkout pages.
	 */
	public function enqueue_checkout_scripts(): void {
		// Check if protection is enabled first (cheapest check).
		if ( ! $this->is_enabled() ) {
			return;
		}

		// Determine checkout type and whether to enqueue.
		$checkout_type = $this->get_checkout_type();
		if ( ! $checkout_type ) {
			return;
		}

		// For blocks checkout, we need wp-api-fetch to register our middleware.
		$deps = ( 'blocks' === $checkout_type ) ? array( 'wp-api-fetch' ) : array();

		// Enqueue the checkout shield script.
		wp_enqueue_script(
			'carticy-checkout-shield',
			CARTICY_CHECKOUT_SHIELD_URL . 'assets/js/checkout-shield.js',
			$deps,
			CARTICY_CHECKOUT_SHIELD_VERSION,
			array(
				'in_footer' => false, // Load in head for early execution.
				'strategy'  => 'defer', // Defer to not block rendering.
			)
		);

		// Add inline configuration if needed.
		wp_add_inline_script(
			'carticy-checkout-shield',
			$this->get_inline_config(),
			'before'
		);
	}

	/**
	 * Get the checkout type for the current page.
	 *
	 * Returns 'classic', 'blocks', or false if not a checkout page.
	 * Optimized to avoid expensive has_block() calls on non-singular pages.
	 *
	 * @return string|false 'classic', 'blocks', or false.
	 */
	private function get_checkout_type() {
		// Check for WooCommerce checkout page (fast - uses page ID comparison).
		if ( function_exists( 'is_checkout' ) && is_checkout() ) {
			// Could be classic or blocks - check if it has the checkout block.
			if ( is_singular() && function_exists( 'has_block' ) && has_block( 'woocommerce/checkout' ) ) {
				return 'blocks';
			}
			return 'classic';
		}

		// Also check for checkout pay page (fast - uses page ID comparison).
		if ( function_exists( 'is_checkout_pay_page' ) && is_checkout_pay_page() ) {
			return 'classic';
		}

		// Only check for checkout block on singular pages (posts/pages).
		// This avoids expensive content parsing on archive pages, home, etc.
		if ( is_singular() && function_exists( 'has_block' ) && has_block( 'woocommerce/checkout' ) ) {
			return 'blocks';
		}

		return false;
	}

	/**
	 * Check if protection is enabled.
	 *
	 * @return bool
	 */
	private function is_enabled(): bool {
		return get_option( 'carticy_checkout_shield_enabled', 'yes' ) === 'yes';
	}

	/**
	 * Get inline configuration for the script.
	 *
	 * @return string
	 */
	private function get_inline_config(): string {
		$config = array(
			'cookieName' => 'ccs_proof',
			'fieldName'  => 'ccs_proof',
			'headerName' => 'X-CCS-Proof',
		);

		return 'window.carticyCheckoutShieldConfig = ' . wp_json_encode( $config ) . ';';
	}
}
