<?php
/**
 * Orders list column for shield status display.
 *
 * @package Carticy\CheckoutShield\Admin
 */

declare(strict_types=1);

namespace Carticy\CheckoutShield\Admin;

use Carticy\CheckoutShield\Services\OrderAttributionService;
use WC_Order;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Adds a shield status column to the WooCommerce orders list.
 * Supports both HPOS and legacy post-based orders.
 */
final class OrdersColumn {

	/**
	 * Column ID.
	 */
	private const COLUMN_ID = 'ccs_status';

	/**
	 * Order attribution service.
	 *
	 * @var OrderAttributionService
	 */
	private OrderAttributionService $attribution_service;

	/**
	 * Constructor.
	 *
	 * @param OrderAttributionService $attribution_service Order attribution service instance.
	 */
	public function __construct( OrderAttributionService $attribution_service ) {
		$this->attribution_service = $attribution_service;
	}

	/**
	 * Register hooks.
	 */
	public function register(): void {
		// HPOS (High-Performance Order Storage) support.
		add_filter( 'manage_woocommerce_page_wc-orders_columns', array( $this, 'add_column' ) );
		add_action( 'manage_woocommerce_page_wc-orders_custom_column', array( $this, 'render_column' ), 10, 2 );

		// Legacy post-based orders support.
		add_filter( 'manage_edit-shop_order_columns', array( $this, 'add_column' ) );
		add_action( 'manage_shop_order_posts_custom_column', array( $this, 'render_legacy_column' ), 10, 2 );

		// Add admin styles.
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_column_styles' ) );
	}

	/**
	 * Add shield status column to orders list.
	 *
	 * @param array $columns Existing columns.
	 * @return array Modified columns.
	 */
	public function add_column( array $columns ): array {
		$new_columns = array();

		foreach ( $columns as $key => $label ) {
			$new_columns[ $key ] = $label;

			// Insert after order status column.
			if ( 'order_status' === $key ) {
				$new_columns[ self::COLUMN_ID ] = __( 'Shield', 'carticy-checkout-shield-for-woocommerce' );
			}
		}

		return $new_columns;
	}

	/**
	 * Render column content for HPOS orders.
	 *
	 * @param string   $column_id The column ID.
	 * @param WC_Order $order     The order object.
	 */
	public function render_column( string $column_id, WC_Order $order ): void {
		if ( self::COLUMN_ID !== $column_id ) {
			return;
		}

		$this->output_status_badge( $order );
	}

	/**
	 * Render column content for legacy post-based orders.
	 *
	 * @param string $column_id The column ID.
	 * @param int    $post_id   The post/order ID.
	 */
	public function render_legacy_column( string $column_id, int $post_id ): void {
		if ( self::COLUMN_ID !== $column_id ) {
			return;
		}

		$order = wc_get_order( $post_id );
		if ( ! $order ) {
			echo '—';
			return;
		}

		$this->output_status_badge( $order );
	}

	/**
	 * Output the status badge for an order.
	 *
	 * @param WC_Order $order The order object.
	 */
	private function output_status_badge( WC_Order $order ): void {
		$attribution = $this->attribution_service->get_attribution( $order );
		$status      = $attribution['status'];

		if ( empty( $status ) ) {
			echo '<span class="ccs-column-badge ccs-column-badge--none" title="' . esc_attr__( 'No shield data', 'carticy-checkout-shield-for-woocommerce' ) . '">—</span>';
			return;
		}

		$labels = array(
			'passed'   => __( 'Passed', 'carticy-checkout-shield-for-woocommerce' ),
			'blocked'  => __( 'Blocked', 'carticy-checkout-shield-for-woocommerce' ),
			'bypassed' => __( 'Bypassed', 'carticy-checkout-shield-for-woocommerce' ),
			'learning' => __( 'Learning', 'carticy-checkout-shield-for-woocommerce' ),
		);

		$icons = array(
			'passed'   => '✓',
			'blocked'  => '✕',
			'bypassed' => '↷',
			'learning' => '◐',
		);

		$label = $labels[ $status ] ?? $status;
		$icon  = $icons[ $status ] ?? '?';

		$origin_label = 'store_api' === $attribution['origin'] ? 'Block Checkout' : 'Classic Checkout';
		$title        = sprintf(
			/* translators: 1: Status label, 2: Checkout origin */
			__( '%1$s via %2$s', 'carticy-checkout-shield-for-woocommerce' ),
			$label,
			$origin_label
		);

		printf(
			'<span class="ccs-column-badge ccs-column-badge--%s" title="%s">%s</span>',
			esc_attr( $status ),
			esc_attr( $title ),
			esc_html( $icon )
		);
	}

	/**
	 * Enqueue column styles on orders pages.
	 *
	 * @param string $hook Current admin page hook.
	 */
	public function enqueue_column_styles( string $hook ): void {
		// Check if we're on a relevant orders page.
		if ( ! in_array( $hook, array( 'edit.php', 'woocommerce_page_wc-orders' ), true ) ) {
			return;
		}

		// For edit.php, verify we're on shop_order post type.
		if ( 'edit.php' === $hook ) {
			$screen = get_current_screen();
			if ( ! $screen || 'edit-shop_order' !== $screen->id ) {
				return;
			}
		}

		$css = '
			.column-ccs_status {
				width: 50px;
				text-align: center;
			}
			.ccs-column-badge {
				display: inline-block;
				width: 22px;
				height: 22px;
				line-height: 22px;
				text-align: center;
				border-radius: 50%;
				font-size: 12px;
				cursor: help;
			}
			.ccs-column-badge--passed {
				background: #d6f4e9;
				color: #00a32a;
			}
			.ccs-column-badge--blocked {
				background: #f8e8e8;
				color: #d63638;
			}
			.ccs-column-badge--bypassed {
				background: #e7f0f5;
				color: #2271b1;
			}
			.ccs-column-badge--learning {
				background: #fcf0c3;
				color: #996800;
			}
			.ccs-column-badge--none {
				color: #a7aaad;
			}
		';

		wp_add_inline_style( 'woocommerce_admin_styles', $css );
	}
}
