<?php
/**
 * Plugin Name: Carticy Checkout Shield for WooCommerce
 * Plugin URI: https://carticy.com/checkout-shield
 * Description: Protects WooCommerce stores from card testing and fake order attacks using stateless bot detection.
 * Version: 1.0.0
 * Author: Carticy
 * Author URI: https://carticy.com
 * License: GPL-2.0-or-later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: carticy-checkout-shield-for-woocommerce
 * Domain Path: /languages
 * Requires at least: 6.0
 * Requires PHP: 8.0
 * Requires Plugins: woocommerce
 * WC requires at least: 8.0
 * WC tested up to: 10.4.3
 *
  *
 * @package Carticy\CheckoutShield
 */

declare(strict_types=1);

// Prevent direct file access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Dynamic path constants (folder-agnostic for Freemius deployment).
if ( ! defined( 'CARTICY_CHECKOUT_SHIELD_PLUGIN_DIR' ) ) {
	define( 'CARTICY_CHECKOUT_SHIELD_PLUGIN_DIR', __DIR__ );
	define( 'CARTICY_CHECKOUT_SHIELD_PLUGIN_URL', plugins_url( plugin_basename( CARTICY_CHECKOUT_SHIELD_PLUGIN_DIR ) ) );
}

/**
 * Freemius integration for free/premium version handling.
 *
 * DO NOT REMOVE THIS IF, IT IS ESSENTIAL FOR THE
 * `function_exists` CALL BELOW TO PROPERLY WORK.
 *
 * This structure allows the premium version to auto-deactivate the free version.
 */
if ( function_exists( 'carticy_checkoutshield_fs' ) ) {
	carticy_checkoutshield_fs()->set_basename( false, __FILE__ );
} else {
	// Plugin constants.
	define( 'CARTICY_CHECKOUT_SHIELD_VERSION', '1.0.0' );
	define( 'CARTICY_CHECKOUT_SHIELD_FILE', __FILE__ );
	define( 'CARTICY_CHECKOUT_SHIELD_PATH', plugin_dir_path( __FILE__ ) );
	define( 'CARTICY_CHECKOUT_SHIELD_URL', plugin_dir_url( __FILE__ ) );
	define( 'CARTICY_CHECKOUT_SHIELD_BASENAME', plugin_basename( __FILE__ ) );

	/**
	 * Load Composer autoloader.
	 */
	if ( file_exists( CARTICY_CHECKOUT_SHIELD_PATH . 'vendor/autoload.php' ) ) {
		require_once CARTICY_CHECKOUT_SHIELD_PATH . 'vendor/autoload.php';
	}

	/**
	 * Load Freemius SDK and licensing functions.
	 */
	if ( file_exists( CARTICY_CHECKOUT_SHIELD_PATH . 'src/licensing.php' ) ) {
		require_once CARTICY_CHECKOUT_SHIELD_PATH . 'src/licensing.php';
	}

	/**
	 * Declare HPOS compatibility.
	 */
	add_action(
		'before_woocommerce_init',
		static function (): void {
			if ( class_exists( \Automattic\WooCommerce\Utilities\FeaturesUtil::class ) ) {
				\Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility(
					'custom_order_tables',
					__FILE__,
					true
				);
				\Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility(
					'cart_checkout_blocks',
					__FILE__,
					true
				);
			}
		}
	);

	/**
	 * Check plugin requirements.
	 *
	 * @return bool True if requirements are met.
	 */
	function carticy_checkout_shield_check_requirements(): bool {
		$errors = array();

		// Check PHP version.
		if ( version_compare( PHP_VERSION, '8.0', '<' ) ) {
			$errors[] = sprintf(
				/* translators: %s: Required PHP version */
				__( 'Carticy Checkout Shield requires PHP %s or higher.', 'carticy-checkout-shield-for-woocommerce' ),
				'8.0'
			);
		}

		// Check WordPress version.
		if ( version_compare( get_bloginfo( 'version' ), '6.0', '<' ) ) {
			$errors[] = sprintf(
				/* translators: %s: Required WordPress version */
				__( 'Carticy Checkout Shield requires WordPress %s or higher.', 'carticy-checkout-shield-for-woocommerce' ),
				'6.0'
			);
		}

		// Check if WooCommerce is active.
		if ( ! class_exists( 'WooCommerce' ) ) {
			$errors[] = __( 'Carticy Checkout Shield requires WooCommerce to be installed and active.', 'carticy-checkout-shield-for-woocommerce' );
		}

		// Check WooCommerce version.
		if ( defined( 'WC_VERSION' ) && version_compare( WC_VERSION, '8.0', '<' ) ) {
			$errors[] = sprintf(
				/* translators: %s: Required WooCommerce version */
				__( 'Carticy Checkout Shield requires WooCommerce %s or higher.', 'carticy-checkout-shield-for-woocommerce' ),
				'8.0'
			);
		}

		if ( ! empty( $errors ) ) {
			add_action(
				'admin_notices',
				static function () use ( $errors ): void {
					foreach ( $errors as $error ) {
						printf(
							'<div class="notice notice-error"><p>%s</p></div>',
							esc_html( $error )
						);
					}
				}
			);
			return false;
		}

		return true;
	}

	/**
	 * Initialize the plugin.
	 */
	function carticy_checkout_shield_init(): void {
		// Check requirements.
		if ( ! carticy_checkout_shield_check_requirements() ) {
			return;
		}

		// Initialize the plugin.
		if ( class_exists( \Carticy\CheckoutShield\Init::class ) ) {
			\Carticy\CheckoutShield\Init::get_instance();
		}
	}
	add_action( 'plugins_loaded', 'carticy_checkout_shield_init', 20 );

	/**
	 * Plugin activation hook.
	 */
	function carticy_checkout_shield_activate(): void {
		if ( class_exists( \Carticy\CheckoutShield\Core\Activator::class ) ) {
			\Carticy\CheckoutShield\Core\Activator::activate();
		}
	}
	register_activation_hook( __FILE__, 'carticy_checkout_shield_activate' );

	/**
	 * Plugin deactivation hook.
	 */
	function carticy_checkout_shield_deactivate(): void {
		if ( class_exists( \Carticy\CheckoutShield\Core\Deactivator::class ) ) {
			\Carticy\CheckoutShield\Core\Deactivator::deactivate();
		}
	}
	register_deactivation_hook( __FILE__, 'carticy_checkout_shield_deactivate' );

	/**
	 * Add plugin action links (Settings, etc.) on the plugins page.
	 *
	 * @param array $links Existing plugin action links.
	 * @return array Modified plugin action links.
	 */
	function carticy_checkout_shield_plugin_action_links( array $links ): array {
		$settings_url  = admin_url( 'admin.php?page=wc-settings&tab=advanced&section=checkout_shield' );
		$settings_link = '<a href="' . esc_url( $settings_url ) . '">' . esc_html__( 'Settings', 'carticy-checkout-shield-for-woocommerce' ) . '</a>';

		// Add settings link at the beginning.
		array_unshift( $links, $settings_link );

		return $links;
	}
	add_filter( 'plugin_action_links_' . CARTICY_CHECKOUT_SHIELD_BASENAME, 'carticy_checkout_shield_plugin_action_links' );
}
