<?php
/*
Plugin Name: Card To Card Gateway For EDD
Description: Add Card to Card gateway to EDD
Version: 1.4.0
Author: mohammadr3z
License: GPLv2 or later
License URI: https://www.gnu.org/licenses/gpl-2.0.html
Requires Plugins: easy-digital-downloads
Text Domain: card-to-card-gateway-for-edd
Domain Path: /languages
*/

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

// Define plugin version constant
if (!defined('MMZ_CTC_VERSION')) {
    define('MMZ_CTC_VERSION', '1.4.0');
}

// Register Card to Card gateway
add_filter('edd_payment_gateways', static function ($gateways) {
    $gateways['mmz_ctc'] = [
        'admin_label'    => __('Card to Card', 'card-to-card-gateway-for-edd'),
        'checkout_label' => __('Card to Card', 'card-to-card-gateway-for-edd')
    ];
    return $gateways;
});

// Get available banks from assets directory
function mmz_ctc_get_available_banks()
{
    $banks = [
        ''         => __('Select Bank', 'card-to-card-gateway-for-edd'),
        'ansar'    => __('Ansar Bank', 'card-to-card-gateway-for-edd'),
        'ba'       => __('Bank Ayandeh', 'card-to-card-gateway-for-edd'),
        'bim'      => __('Bank Melli Iran', 'card-to-card-gateway-for-edd'),
        'bki'      => __('Bank Keshavarzi', 'card-to-card-gateway-for-edd'),
        'bmi'      => __('Bank Mellat', 'card-to-card-gateway-for-edd'),
        'bpi'      => __('Bank Pasargad', 'card-to-card-gateway-for-edd'),
        'bsi'      => __('Bank Saderat', 'card-to-card-gateway-for-edd'),
        'day'      => __('Day Bank', 'card-to-card-gateway-for-edd'),
        'edbi'     => __('Export Development Bank of Iran', 'card-to-card-gateway-for-edd'),
        'en'       => __('Eghtesad Novin Bank', 'card-to-card-gateway-for-edd'),
        'ghbi'     => __('Ghavamin Bank', 'card-to-card-gateway-for-edd'),
        'hi'       => __('Hekmat Iranian Bank', 'card-to-card-gateway-for-edd'),
        'iz'       => __('Iranzamin Bank', 'card-to-card-gateway-for-edd'),
        'kar'      => __('Bank Kar Afarin', 'card-to-card-gateway-for-edd'),
        'maskan'   => __('Bank Maskan', 'card-to-card-gateway-for-edd'),
        'me'       => __('Bank Tejarat', 'card-to-card-gateway-for-edd'),
        'mehriran' => __('Mehr Iran Bank', 'card-to-card-gateway-for-edd'),
        'melal'    => __('Bank Mellal', 'card-to-card-gateway-for-edd'),
        'mellat'   => __('Bank Mellat', 'card-to-card-gateway-for-edd'),
        'parsian'  => __('Parsian Bank', 'card-to-card-gateway-for-edd'),
        'post'     => __('Post Bank', 'card-to-card-gateway-for-edd'),
        'rb'       => __('Refah Bank', 'card-to-card-gateway-for-edd'),
        'resalat'  => __('Resalat Bank', 'card-to-card-gateway-for-edd'),
        'sarmayeh' => __('Sarmayeh Bank', 'card-to-card-gateway-for-edd'),
        'sb'       => __('Saman Bank', 'card-to-card-gateway-for-edd'),
        'sepah'    => __('Bank Sepah', 'card-to-card-gateway-for-edd'),
        'shahr'    => __('Shahr Bank', 'card-to-card-gateway-for-edd'),
        'sina'     => __('Sina Bank', 'card-to-card-gateway-for-edd'),
        'tejarat'  => __('Tejarat Bank', 'card-to-card-gateway-for-edd'),
        'tourism'  => __('Tourism Bank', 'card-to-card-gateway-for-edd'),
        'tt'       => __('Tosee Taavon Bank', 'card-to-card-gateway-for-edd'),
        'custom'   => __('Custom (Upload)', 'card-to-card-gateway-for-edd'),
    ];
    return $banks;
}

// Get bank name from bank code
function mmz_ctc_get_bank_name($bank_code)
{
    $banks = mmz_ctc_get_available_banks();
    return isset($banks[$bank_code]) ? $banks[$bank_code] : '';
}

// Add gateway settings (bank icon, Telegram, etc.)
add_filter('edd_settings_gateways', static function ($gateway_settings) {
    $gateway_settings[] = [
        'id'   => 'mmz_ctc_settings',
        'name' => '<strong>' . __('Card to Card Settings', 'card-to-card-gateway-for-edd') . '</strong>',
        'desc' => '',
        'type' => 'header',
    ];
    $gateway_settings[] = [
        'id'   => 'mmz_ctc_card_number',
        'name' => __('Card Number', 'card-to-card-gateway-for-edd'),
        'desc' => __('Destination card number for card to card payment', 'card-to-card-gateway-for-edd'),
        'type' => 'text',
        'size' => 'regular',
    ];
    $gateway_settings[] = [
        'id'   => 'mmz_ctc_card_name',
        'name' => __('Card Holder Name', 'card-to-card-gateway-for-edd'),
        'desc' => __('Name of the destination card holder', 'card-to-card-gateway-for-edd'),
        'type' => 'text',
        'size' => 'regular',
    ];
    $gateway_settings[] = [
        'id'      => 'mmz_ctc_bank_icon',
        'name'    => __('Bank Icon', 'card-to-card-gateway-for-edd'),
        'desc'    => __('Select your bank from the list or choose "Custom" to upload your own.', 'card-to-card-gateway-for-edd'),
        'type'    => 'select',
        'options' => mmz_ctc_get_available_banks(),
    ];
    $gateway_settings[] = [
        'id'   => 'mmz_ctc_custom_bank_icon',
        'name' => __('Custom Bank Icon', 'card-to-card-gateway-for-edd'),
        'desc' => __('Upload your custom bank icon (only used when "Custom" is selected above).', 'card-to-card-gateway-for-edd'),
        'type' => 'upload',
        'size' => 'regular',
    ];
    $gateway_settings[] = [
        'id'   => 'mmz_ctc_telegram_enable',
        'name' => __('Send Message to Telegram', 'card-to-card-gateway-for-edd'),
        'desc' => __('If enabled, order message will be sent to Telegram bot.', 'card-to-card-gateway-for-edd') .
            '<br><div style="background:#fff3cd;border-right:4px solid #ffc107;padding:12px 15px;margin-top:10px;border-radius:4px;">' .
            '<span style="color:#856404;font-weight:600;font-size:13px;">⚠️ ' . __('Note: This feature will not work on Iranian servers due to Telegram filtering.', 'card-to-card-gateway-for-edd') . '</span>' .
            '</div>',
        'type' => 'checkbox',
    ];
    $gateway_settings[] = [
        'id'   => 'mmz_ctc_telegram_bot_token',
        'name' => __('Telegram Bot Token', 'card-to-card-gateway-for-edd'),
        'desc' => __('Enter your Telegram bot token.', 'card-to-card-gateway-for-edd'),
        'type' => 'text',
        'size' => 'regular',
        'attributes' => [
            'data-telegram-dependent' => '1'
        ]
    ];
    $gateway_settings[] = [
        'id'   => 'mmz_ctc_telegram_chat_id',
        'name' => __('Telegram Chat ID', 'card-to-card-gateway-for-edd'),
        'desc' => __('Enter your numeric Telegram Chat ID.', 'card-to-card-gateway-for-edd'),
        'type' => 'text',
        'size' => 'regular',
        'attributes' => [
            'data-telegram-dependent' => '1'
        ]
    ];
    $gateway_settings[] = [
        'id'   => 'mmz_ctc_notice_text',
        'name' => __('Instructions Notice', 'card-to-card-gateway-for-edd'),
        'desc' => __('Custom notice text shown on the checkout form. Leave empty to use the default message.', 'card-to-card-gateway-for-edd'),
        'type' => 'rich_editor',
    ];
    return $gateway_settings;
});

// Enqueue admin scripts and styles
add_action('admin_enqueue_scripts', static function ($hook_suffix) {
    // Only load on EDD settings page
    if ($hook_suffix === 'download_page_edd-settings') {
        wp_enqueue_script(
            'mmz-ctc-admin-settings',
            plugin_dir_url(__FILE__) . 'assets/js/admin-settings.js',
            ['jquery'],
            MMZ_CTC_VERSION,
            ['in_footer' => true]
        );
    }
});

// Render Card to Card form on checkout
add_action('edd_mmz_ctc_cc_form', static function () {
    // Get and sanitize gateway settings
    $card_number = sanitize_text_field(edd_get_option('mmz_ctc_card_number'));
    $card_name   = sanitize_text_field(edd_get_option('mmz_ctc_card_name'));
    $bank_code   = sanitize_text_field(edd_get_option('mmz_ctc_bank_icon'));

    // Validate card number (basic validation for display)
    if (!empty($card_number) && !preg_match('/^\d{16}$/', str_replace(['-', ' '], '', $card_number))) {
        $card_number = ''; // Clear invalid card number
    }

    // Build bank icon URL from selected bank code
    $bank_icon_url = '';
    if (!empty($bank_code)) {
        if ($bank_code === 'custom') {
            // Use custom uploaded icon
            $custom_icon = edd_get_option('mmz_ctc_custom_bank_icon');
            if (!empty($custom_icon)) {
                if (is_numeric($custom_icon)) {
                    $bank_icon_url = wp_get_attachment_url(absint($custom_icon));
                } else {
                    $bank_icon_url = esc_url($custom_icon);
                }
            }
        } else {
            // Use predefined bank icon
            $bank_icon_url = plugin_dir_url(__FILE__) . 'assets/images/banks/' . $bank_code . '.png';
        }
    }

    // Get bank name for display
    $bank_name = '';
    if (!empty($bank_code) && $bank_code !== 'custom') {
        $bank_name = mmz_ctc_get_bank_name($bank_code);
    }

?>
    <div class="mmz-ctc-container">
        <div class="mmz-ctc-bank-section">
            <?php if (!empty($bank_icon_url)): ?>
                <img src="<?php echo esc_url($bank_icon_url); ?>" alt="<?php echo esc_attr__('Bank Icon', 'card-to-card-gateway-for-edd'); ?>" class="mmz-ctc-bank-icon">
            <?php endif; ?>
            <?php if (!empty($card_number)): ?>
                <div class="mmz-ctc-card-number-display">
                    <span class="mmz-ctc-card-number"><?php echo esc_html(preg_replace('/\b(\d{4})(\d{4})(\d{4})(\d+)/', '$1-$2-$3-$4', $card_number)); ?></span>
                    <button type="button" onclick="copyToClipboard('<?php echo esc_js($card_number); ?>')" class="mmz-ctc-copy-button">
                        <svg xmlns='http://www.w3.org/2000/svg' width='18' height='18' viewBox='0 0 24 24' fill='none' stroke='#1976d2' stroke-width='2' stroke-linecap='round' stroke-linejoin='round'>
                            <rect x='9' y='9' width='13' height='13' rx='2' ry='2'></rect>
                            <path d='M5 15H4a2 2 0 0 1-2-2V4a2 2 0 0 1 2-2h9a2 2 0 0 1 2 2v1'></path>
                        </svg>
                    </button>
                </div>
            <?php endif; ?>
            <?php if (!empty($card_name)): ?>
                <div class="mmz-ctc-card-holder"><?php echo esc_html(sprintf(__('Account Holder: %s', 'card-to-card-gateway-for-edd'), $card_name)); ?></div>
            <?php endif; ?>
            <?php if (!empty($bank_name)): ?>
                <div class="mmz-ctc-bank-name"><?php echo esc_html(sprintf(__('%s', 'card-to-card-gateway-for-edd'), $bank_name)); ?></div>
            <?php endif; ?>
        </div>
        <fieldset class="mmz-ctc-fieldset">
            <div class="mmz-ctc-form-fields">
                <p class="mmz-ctc-field">
                    <label class="mmz-ctc-label">
                        <?php esc_html_e('Tracking Number:', 'card-to-card-gateway-for-edd'); ?>
                        <span class="edd-required-indicator mmz-ctc-required" title="<?php echo esc_attr__('Required', 'card-to-card-gateway-for-edd'); ?>">*</span>
                    </label>
                    <input type="text" name="mmz_ctc_tracking_code" required class="edd-input mmz-ctc-input">
                </p>
                <p class="mmz-ctc-field">
                    <label class="mmz-ctc-label">
                        <?php esc_html_e('Reference Number:', 'card-to-card-gateway-for-edd'); ?>
                        <span class="edd-required-indicator mmz-ctc-required" title="<?php echo esc_attr__('Required', 'card-to-card-gateway-for-edd'); ?>">*</span>
                    </label>
                    <input type="text" name="mmz_ctc_reference_code" required class="edd-input mmz-ctc-input">
                </p>
            </div>
        </fieldset>
        <?php wp_nonce_field('mmz_ctc_payment', 'mmz_ctc_nonce'); ?>
        <?php $custom_notice = edd_get_option('mmz_ctc_notice_text'); ?>
        <div class="mmz-ctc-notice">
            <?php if (!empty($custom_notice)): ?>
                <?php echo wp_kses_post(wpautop($custom_notice)); ?>
            <?php else: ?>
                <?php esc_html_e('Your payment will be confirmed by our specialists as soon as possible and your access will be activated.', 'card-to-card-gateway-for-edd'); ?>
            <?php endif; ?>
        </div>
    </div>
    <?php
});



// Add card-to-card info to payment args
add_filter('edd_insert_payment_args', static function ($payment_data) {
    // Only process if this is a card-to-card payment
    if (!isset($payment_data['gateway']) || $payment_data['gateway'] !== 'mmz_ctc') {
        return $payment_data;
    }

    // Verify nonce for security - check nonce first before accessing other $_POST data
    if (!isset($_POST['mmz_ctc_nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['mmz_ctc_nonce'])), 'mmz_ctc_payment')) {
        return $payment_data;
    }

    // Sanitize and validate tracking codes
    $tracking_code = isset($_POST['mmz_ctc_tracking_code']) ? sanitize_text_field(wp_unslash($_POST['mmz_ctc_tracking_code'])) : '';
    $reference_code = isset($_POST['mmz_ctc_reference_code']) ? sanitize_text_field(wp_unslash($_POST['mmz_ctc_reference_code'])) : '';

    if (!empty($tracking_code) && !empty($reference_code)) {
        $payment_data['mmz_ctc'] = [
            'tracking_code'  => $tracking_code,
            'reference_code' => $reference_code,
        ];
    }
    return $payment_data;
});

// Keep payment status as pending for this gateway
add_filter('edd_payment_status', static function ($status, $payment_id, $new_status, $old_status) {
    if (edd_get_payment_gateway($payment_id) === 'mmz_ctc' && $old_status === 'pending') {
        return 'pending';
    }
    return $status;
}, 10, 4);

// Show card-to-card info in personal details list (admin)
add_action('edd_payment_personal_details_list', static function ($payment_id) {
    $payment = edd_get_payment($payment_id);
    $meta = $payment?->get_meta('mmz_ctc');
    if ($meta) {
    ?>
        <li>
            <strong><?php echo esc_html__('Tracking Number:', 'card-to-card-gateway-for-edd'); ?></strong>
            <span><?php echo esc_html($meta['tracking_code']); ?></span>
        </li>
        <li>
            <strong><?php echo esc_html__('Reference Number:', 'card-to-card-gateway-for-edd'); ?></strong>
            <span><?php echo esc_html($meta['reference_code']); ?></span>
        </li>
    <?php
    }
});

// Add confirm button to order actions in admin if payment is pending
add_action('edd_order_details_actions', static function ($payment_id) {
    $payment = edd_get_payment($payment_id);
    if ($payment?->gateway === 'mmz_ctc' && $payment?->status === 'pending') {
        $url = wp_nonce_url(
            add_query_arg([
                'edd_action' => 'mmz_ctc_confirm',
                'payment_id' => absint($payment_id),
            ], admin_url('edit.php?post_type=download&page=edd-payment-history')),
            'mmz_ctc_confirm_' . absint($payment_id)
        );
        echo '<a href="' . esc_url($url) . '" class="button button-primary">' . esc_html__('Confirm Card to Card Payment', 'card-to-card-gateway-for-edd') . '</a>';
    }
});

// Handle confirm action in admin (set payment to publish)
add_action('admin_init', static function () {
    // Early permission check to prevent unnecessary processing
    if (!current_user_can('manage_shop_payments')) {
        return; // Silent return for unauthorized users
    }

    // Basic check for required parameters without processing values
    if (!isset($_GET['edd_action'], $_GET['payment_id'], $_GET['_wpnonce'])) {
        return;
    }

    // Sanitize and validate payment ID first
    $payment_id = absint($_GET['payment_id']);

    // Validate payment ID early
    if ($payment_id <= 0) {
        wp_die(__('Invalid payment ID.', 'card-to-card-gateway-for-edd'), __('Invalid Request', 'card-to-card-gateway-for-edd'), ['response' => 400]);
    }

    // Verify nonce FIRST - this is critical for security
    if (!wp_verify_nonce(sanitize_text_field(wp_unslash($_GET['_wpnonce'])), 'mmz_ctc_confirm_' . $payment_id)) {
        wp_die(__('Security check failed. Please try again.', 'card-to-card-gateway-for-edd'), __('Security Error', 'card-to-card-gateway-for-edd'), ['response' => 403]);
    }

    // Now safely check the action after nonce verification
    if (sanitize_text_field(wp_unslash($_GET['edd_action'])) !== 'mmz_ctc_confirm') {
        return;
    }

    // Final permission check after nonce verification
    if (!current_user_can('manage_shop_payments')) {
        wp_die(__('You do not have permission to perform this action.', 'card-to-card-gateway-for-edd'), __('Permission Error', 'card-to-card-gateway-for-edd'), ['response' => 403]);
    }

    // Verify payment exists and belongs to card-to-card gateway
    $payment = edd_get_payment($payment_id);
    if (!$payment || edd_get_payment_gateway($payment_id) !== 'mmz_ctc') {
        wp_die(__('Invalid payment or payment method.', 'card-to-card-gateway-for-edd'), __('Invalid Request', 'card-to-card-gateway-for-edd'), ['response' => 400]);
    }

    // All checks passed, update payment status
    edd_update_payment_status($payment_id, 'publish');
    wp_redirect(remove_query_arg(['edd_action', 'payment_id', '_wpnonce']));
    exit;
});

// Handle Card to Card gateway payment processing
add_action('edd_gateway_mmz_ctc', static function ($purchase_data) {
    // Verify this is the correct gateway
    if (!isset($purchase_data['gateway']) || $purchase_data['gateway'] !== 'mmz_ctc') {
        edd_set_error('mmz_ctc_gateway_mismatch', __('Payment gateway mismatch. Please try again.', 'card-to-card-gateway-for-edd'));
        edd_send_back_to_checkout('?payment-mode=mmz_ctc');
        return;
    }

    // Verify nonce first - this is critical for security
    if (!isset($_POST['mmz_ctc_nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['mmz_ctc_nonce'])), 'mmz_ctc_payment')) {
        edd_set_error('nonce_error', __('Security validation failed. Please try again.', 'card-to-card-gateway-for-edd'));
        edd_send_back_to_checkout('?payment-mode=mmz_ctc');
        return;
    }

    // Sanitize and validate required fields
    $tracking_code = isset($_POST['mmz_ctc_tracking_code']) ? sanitize_text_field(wp_unslash($_POST['mmz_ctc_tracking_code'])) : '';
    $reference_code = isset($_POST['mmz_ctc_reference_code']) ? sanitize_text_field(wp_unslash($_POST['mmz_ctc_reference_code'])) : '';

    if (empty($tracking_code) || empty($reference_code)) {
        edd_set_error('missing_info', __('Please fill in all card to card fields.', 'card-to-card-gateway-for-edd'));
        edd_send_back_to_checkout('?payment-mode=mmz_ctc');
        return;
    }

    if (edd_get_errors()) {
        edd_send_back_to_checkout('?payment-mode=mmz_ctc');
        return;
    }

    // Prepare meta data with sanitized inputs
    $mmz_ctc_meta = [
        'tracking_code'  => $tracking_code,
        'reference_code' => $reference_code,
        'date'           => date_i18n('j F Y', current_time('timestamp')),
        'time'           => date_i18n('H:i', current_time('timestamp')),
    ];

    // Prepare payment data
    $payment_data = [
        'price'        => $purchase_data['price'],
        'date'         => $purchase_data['date'],
        'user_email'   => $purchase_data['user_email'],
        'purchase_key' => $purchase_data['purchase_key'],
        'currency'     => edd_get_currency(),
        'downloads'    => $purchase_data['downloads'],
        'cart_details' => $purchase_data['cart_details'],
        'user_info'    => $purchase_data['user_info'],
        'status'       => 'pending',
        'gateway'      => 'mmz_ctc',
        'meta'         => [
            'mmz_ctc' => $mmz_ctc_meta
        ]
    ];

    $payment_id = edd_insert_payment($payment_data);

    // Save card-to-card meta for admin compatibility
    if ($payment_id && !empty($mmz_ctc_meta)) {
        edd_update_payment_meta($payment_id, 'mmz_ctc', $mmz_ctc_meta);
    }

    // Telegram notification settings
    $telegram_enabled = edd_get_option('mmz_ctc_telegram_enable') === '1';
    $telegram_token   = sanitize_text_field(edd_get_option('mmz_ctc_telegram_bot_token'));
    $telegram_chat_id = sanitize_text_field(edd_get_option('mmz_ctc_telegram_chat_id'));

    // Build Telegram message if payment was successful
    $message = null;
    if ($payment_id) {
        $user_info = $purchase_data['user_info'];
        $first_name = isset($user_info['first_name']) ? sanitize_text_field($user_info['first_name']) : '';
        $last_name = isset($user_info['last_name']) ? sanitize_text_field($user_info['last_name']) : '';
        $user_email = sanitize_email($purchase_data['user_email']);

        $message = "<b>" . __('New Card to Card Order', 'card-to-card-gateway-for-edd') . "</b>\n";
        $message .= __('Name:', 'card-to-card-gateway-for-edd') . " " . esc_html($first_name . ' ' . $last_name) . "\n";
        $message .= __('Email:', 'card-to-card-gateway-for-edd') . " " . esc_html($user_email) . "\n";
        $message .= __('Amount:', 'card-to-card-gateway-for-edd') . " " . esc_html(edd_currency_filter(edd_format_amount($purchase_data['price']))) . "\n";
        $message .= __('Tracking Number:', 'card-to-card-gateway-for-edd') . " " . esc_html($mmz_ctc_meta['tracking_code']) . "\n";
        $message .= __('Reference Number:', 'card-to-card-gateway-for-edd') . " " . esc_html($mmz_ctc_meta['reference_code']) . "\n";
        $message .= __('Date:', 'card-to-card-gateway-for-edd') . " " . esc_html($mmz_ctc_meta['date']) . "\n";
        $message .= __('Time:', 'card-to-card-gateway-for-edd') . " " . esc_html($mmz_ctc_meta['time']) . "\n";
    }

    // Send Telegram notification if enabled
    if ($payment_id && $telegram_enabled && !empty($telegram_token) && !empty($telegram_chat_id) && !empty($message)) {
        mmz_ctc_send_telegram_message($message, absint($payment_id));
    }

    if ($payment_id) {
        edd_empty_cart();
        wp_redirect(edd_get_success_page_uri());
        exit;
    } else {
        edd_set_error('payment_error', __('Error in order registration. Please try again.', 'card-to-card-gateway-for-edd'));
        edd_send_back_to_checkout('?payment-mode=mmz_ctc');
        return;
    }
});

// Show card-to-card info in admin order sidebar
function mmz_ctc_show_info_in_admin($payment_id)
{
    if (edd_get_payment_gateway($payment_id) !== 'mmz_ctc') {
        return;
    }
    // Try to get card-to-card info from payment meta
    $payment = edd_get_payment($payment_id);
    $meta = null;
    if (is_object($payment) && method_exists($payment, 'get_meta')) {
        $meta = $payment->get_meta('mmz_ctc');
    }
    if (!$meta) {
        $meta = edd_get_payment_meta($payment_id, 'mmz_ctc', true);
    }
    ?>
    <div id="edd-order-mmz-ctc" class="postbox edd-order-logs">
        <h3 class="hndle">
            <span><?php echo esc_html__('Card to Card Information', 'card-to-card-gateway-for-edd'); ?></span>
        </h3>
        <div class="inside">
            <div class="edd-admin-box">
                <div class="edd-admin-box-inside">
                    <?php if ($meta && is_array($meta)) : ?>
                        <p><strong><?php echo esc_html__('Tracking Number:', 'card-to-card-gateway-for-edd'); ?></strong> <?php echo esc_html($meta['tracking_code'] ?? '---'); ?></p>
                        <p><strong><?php echo esc_html__('Reference Number:', 'card-to-card-gateway-for-edd'); ?></strong> <?php echo esc_html($meta['reference_code'] ?? '---'); ?></p>
                        <p><strong><?php echo esc_html__('Date:', 'card-to-card-gateway-for-edd'); ?></strong> <?php echo esc_html($meta['date'] ?? '---'); ?></p>
                        <p><strong><?php echo esc_html__('Time:', 'card-to-card-gateway-for-edd'); ?></strong> <?php echo esc_html($meta['time'] ?? '---'); ?></p>
                    <?php else: ?>
                        <p><?php echo esc_html__('No information recorded.', 'card-to-card-gateway-for-edd'); ?></p>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
<?php
}
add_action('edd_view_order_details_sidebar_before', 'mmz_ctc_show_info_in_admin');

// Enqueue frontend styles and scripts
add_action('wp_enqueue_scripts', static function () {
    // Only enqueue on checkout and success pages where EDD is active
    if (function_exists('edd_is_checkout') && (edd_is_checkout() || edd_is_success_page())) {
        wp_enqueue_style(
            'mmz-ctc-frontend',
            plugin_dir_url(__FILE__) . 'assets/css/frontend.css',
            [],
            MMZ_CTC_VERSION
        );

        wp_enqueue_script(
            'mmz-ctc-frontend',
            plugin_dir_url(__FILE__) . 'assets/js/frontend.js',
            ['jquery'],
            MMZ_CTC_VERSION,
            ['in_footer' => true]
        );
    }
});

// Send Telegram message helper
function mmz_ctc_send_telegram_message(string $message, ?int $order_id = null): array|WP_Error
{
    $bot_token = sanitize_text_field(edd_get_option('mmz_ctc_telegram_bot_token'));
    $chat_id   = sanitize_text_field(edd_get_option('mmz_ctc_telegram_chat_id'));

    // Check for required Telegram settings
    if (empty($bot_token) || empty($chat_id)) {
        return new WP_Error('telegram_settings_missing', __('Telegram bot token or chat_id not set.', 'card-to-card-gateway-for-edd'));
    }

    // Validate and sanitize message
    if (empty($message) || !is_string($message)) {
        return new WP_Error('invalid_message', __('Invalid message provided.', 'card-to-card-gateway-for-edd'));
    }

    $args = [
        'body' => [
            'chat_id'    => $chat_id,
            'text'       => $message, // Keep HTML tags for Telegram formatting
            'parse_mode' => 'HTML',
        ],
        'timeout' => 10,
    ];

    // Add order link button if order_id is provided
    if ($order_id && is_int($order_id) && $order_id > 0) {
        $order_link = esc_url_raw(site_url('/wp-admin/edit.php?post_type=download&page=edd-payment-history&view=view-order-details&id=' . absint($order_id)));
        $button = [
            'inline_keyboard' => [
                [
                    [
                        'text' => sprintf(__('View Order #%d', 'card-to-card-gateway-for-edd'), absint($order_id)),
                        'url'  => $order_link,
                    ]
                ]
            ]
        ];
        $args['body']['reply_markup'] = wp_json_encode($button, JSON_UNESCAPED_UNICODE);
    }

    $url = esc_url_raw("https://api.telegram.org/bot{$bot_token}/sendMessage");
    return wp_remote_post($url, $args);
}
