<?php

/**
 * Widget class for Bytesweavers AI Chat Master.
 *
 * Handles the front-end chat widget display, AJAX message processing,
 * and enqueues necessary scripts and styles.
 *
 * @package Bytesweavers_AI_Chat_Master
 */

namespace AICW;

class Widget {

    /**
     * Constructor.
     */
    public function __construct() {
        // Render widget in the footer.
        add_action('wp_footer', array($this, 'render_widget'));
        // AJAX endpoints for sending messages.
        add_action('wp_ajax_aicw_send_message', array($this, 'handle_message'));
        add_action('wp_ajax_nopriv_aicw_send_message', array($this, 'handle_message'));
        // Enqueue front-end assets.
        add_action('wp_enqueue_scripts', array($this, 'enqueue_scripts'));
    }

    /**
     * Enqueue front-end styles and scripts.
     */
    public function enqueue_scripts() {
        // Enqueue chat widget CSS.
        wp_enqueue_style(
            'aicw-chat-widget',
            AICW_PLUGIN_URL . 'assets/css/chat-widget.css',
            array(),
            AICW_VERSION
        );
        // Enqueue the Markdown parser (marked) from a local file.
        wp_enqueue_script(
            'marked',
            AICW_PLUGIN_URL . 'assets/js/marked.min.js',
            array(),
            '5.0.0',
            true
        );
        // Enqueue chat widget JavaScript.
        wp_enqueue_script(
            'aicw-chat-widget',
            AICW_PLUGIN_URL . 'assets/js/chat-widget.js',
            array('jquery', 'marked'),
            AICW_VERSION,
            true
        );
        // Localize script with AJAX URL and nonce.
        wp_localize_script(
            'aicw-chat-widget',
            'aicwSettings',
            array(
                'ajaxurl' => admin_url('admin-ajax.php'),
                'nonce'   => wp_create_nonce('aicw-chat-nonce')
            )
        );
    }

    /**
     * Render the front-end chat widget.
     */
    public function render_widget() {
        $enabled = get_option('aicw_enable_widget', 0);
        if ( ! $enabled ) {
            return;
        }

        $model = get_option('aicw_selected_model', 'gpt-3.5-turbo');
        $model_family = $this->get_model_family($model);
        $api_key = '';
        // Retrieve the appropriate API key based on the model family.
        switch ($model_family) {
            case 'openai':
                $api_key = get_option('aicw_openai_key');
                break;
            case 'anthropic':
                $api_key = get_option('aicw_anthropic_key');
                break;
            case 'gemini':
                $api_key = get_option('aicw_gemini_key');
                break;
        }
        // If API key is missing, show an admin error (only for users with manage_options).
        if (empty($api_key)) {
            if (current_user_can('manage_options')) {
                echo '<div class="notice notice-error"><p>' . esc_html__('Error: Please set up an API key in the Bytesweavers AI Chat Master settings before enabling the widget.', 'bytesweavers-ai-chat-master') . '</p></div>';
            }
            return;
        }

        // Determine display options.
        $display_option = get_option('aicw_display_option', 'all');
        $specific_pages = get_option('aicw_specific_pages', '');
        if ($display_option === 'some') {
            $page_ids = array_map('trim', explode(',', $specific_pages));
            $page_ids = array_filter($page_ids);
            $current_id = get_queried_object_id();
            if (!in_array($current_id, $page_ids)) {
                return;
            }
        }

        // Get widget appearance settings.
        $settings = array(
            'title'         => get_option('aicw_widget_title', __('AI Assistant', 'bytesweavers-ai-chat-master')),
            'subtitle'      => get_option('aicw_widget_subtitle', __('How can I help you today?', 'bytesweavers-ai-chat-master')),
            'primary_color' => get_option('aicw_primary_color', '#007bff'),
            'width'         => get_option('aicw_chat_width', '380'),
            'height'        => get_option('aicw_chat_height', '600'),
        );

        $position = get_option('aicw_chat_position', 'bottom-left');
        $position_class = 'aicw-position-' . $position;

        // In the free version, force the default chat icon.
        $icon_type = 'chat';

        // Include the widget template.
        include AICW_PLUGIN_DIR . 'templates/chat-widget.php';
    }

    /**
     * Handle the AJAX request to send a chat message.
     */
    public function handle_message() {
        $nonce = isset($_POST['nonce']) ? sanitize_text_field( wp_unslash($_POST['nonce']) ) : '';
        if ( ! wp_verify_nonce( $nonce, 'aicw-chat-nonce' ) ) {
            wp_send_json_error( __('Invalid nonce', 'bytesweavers-ai-chat-master') );
            exit;
        }
        $message = isset($_POST['message']) ? sanitize_text_field( wp_unslash($_POST['message']) ) : '';
        $session_id = isset($_POST['session_id']) ? sanitize_text_field( wp_unslash($_POST['session_id']) ) : '';

        $chat_processor = new \AICW\Chat_Processor();
        $response = $chat_processor->process_message($message, $session_id);

        wp_send_json($response);
    }

    /**
     * Retrieve the model family based on the model string.
     *
     * @param string $model The selected model.
     * @return string|null Returns 'openai', 'anthropic', or 'gemini' if found; otherwise null.
     */
    private function get_model_family($model) {
        if (strpos($model, 'gpt') !== false) {
            return 'openai';
        } elseif (strpos($model, 'claude') !== false) {
            return 'anthropic';
        } elseif (strpos($model, 'gemini') !== false) {
            return 'gemini';
        }
        return null;
    }
}