<?php
/**
 * API Handler class for Bytesweavers AI Chat Master.
 *
 * Handles communication with external AI APIs.
 *
 * @package Bytesweavers_AI_Chat_Master
 */

namespace AICW;

class API_Handler {
    private $openai_key;
    private $anthropic_key;
    private $gemini_key;
    private $selected_model;
    private $temperature;
    private $max_tokens;
    private $top_p;
    private $frequency_penalty;
    private $presence_penalty;

    /**
     * Constructor - retrieve plugin options.
     */
    public function __construct() {
        $this->openai_key = get_option('aicw_openai_key');
        $this->anthropic_key = get_option('aicw_anthropic_key');
        $this->gemini_key = get_option('aicw_gemini_key');
        $this->selected_model = get_option('aicw_selected_model', 'gpt-3.5-turbo');
        $this->temperature = floatval(get_option('aicw_temperature', 0.7));
        $this->max_tokens = intval(get_option('aicw_max_tokens', 1000));
        $this->top_p = floatval(get_option('aicw_top_p', 1.0));
        $this->frequency_penalty = floatval(get_option('aicw_frequency_penalty', 0.0));
        $this->presence_penalty = floatval(get_option('aicw_presence_penalty', 0.0));
    }

    /**
     * Determine the model family.
     *
     * @param string $model AI model name.
     * @return string|null
     */
    private function get_model_family($model) {
        if (strpos($model, 'gpt') === 0 || strpos($model, 'chatgpt') === 0) return 'openai';
        if (strpos($model, 'claude') === 0) return 'anthropic';
        if (strpos($model, 'gemini') === 0) return 'gemini';
        return null;
    }

    /**
     * Test connection to the selected API.
     *
     * @param string $model Model.
     * @param string $api_key API key.
     * @return array
     */
    public function test_connection($model, $api_key) {
        $model_family = $this->get_model_family($model);
        switch ($model_family) {
            case 'openai':
                return $this->test_openai_connection($api_key);
            case 'anthropic':
                return $this->test_anthropic_connection($api_key);
            case 'gemini':
                return $this->test_gemini_connection($api_key);
            default:
                return array('error' => __('Invalid model family', 'bytesweavers-ai-chat-master'));
        }
    }

    /**
     * Process the user message using the selected AI model.
     *
     * @param string $message
     * @param string $context
     * @return mixed
     */
    public function process_message($message, $context = '') {
        $model_family = $this->get_model_family($this->selected_model);
        switch ($model_family) {
            case 'openai':
                return $this->call_openai($message, $context);
            case 'anthropic':
                return $this->call_anthropic($message, $context);
            case 'gemini':
                return $this->call_gemini($message, $context);
            default:
                return array('error' => __('Invalid model selected', 'bytesweavers-ai-chat-master'));
        }
    }

    /**
     * Call the OpenAI API.
     *
     * @param string $message
     * @param string $context
     * @return mixed
     */
    private function call_openai($message, $context) {
        $url = 'https://api.openai.com/v1/chat/completions';
        $headers = array(
            'Authorization' => 'Bearer ' . $this->openai_key,
            'Content-Type'  => 'application/json'
        );

        // Format messages array.
        $messages = array();
        if (!empty($context)) {
            $messages[] = array('role' => 'system', 'content' => $context);
        }
        $messages[] = array('role' => 'user', 'content' => $message);

        $body = array(
            'model' => $this->selected_model,
            'messages' => $messages,
            'temperature' => $this->temperature,
            'max_tokens' => $this->max_tokens,
            'top_p' => $this->top_p,
            'frequency_penalty' => $this->frequency_penalty,
            'presence_penalty' => $this->presence_penalty
        );

        $response = wp_remote_post($url, array(
            'headers' => $headers,
            'body'    => json_encode($body),
            'timeout' => 60
        ));

        if (is_wp_error($response)) {
            return array('error' => $response->get_error_message());
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);
        if (isset($body['error'])) {
            return array('error' => $body['error']['message']);
        }
        return $body['choices'][0]['message']['content'] ?? __('Error processing request', 'bytesweavers-ai-chat-master');
    }

    /**
     * Call the Anthropic API.
     *
     * @param string $message
     * @param string $context
     * @return mixed
     */
    private function call_anthropic($message, $context) {
        $url = 'https://api.anthropic.com/v1/messages';
        $headers = array(
            'anthropic-version' => '2024-01-01',
            'x-api-key' => $this->anthropic_key,
            'Content-Type' => 'application/json'
        );

        $full_content = ($context ? $context . "\n\n" : '') . $message;

        $body = array(
            'model' => $this->selected_model,
            'messages' => [
                array('role' => 'user', 'content' => $full_content)
            ],
            'max_tokens' => $this->max_tokens,
            'temperature' => $this->temperature
        );

        $response = wp_remote_post($url, array(
            'headers' => $headers,
            'body' => json_encode($body),
            'timeout' => 60
        ));

        if (is_wp_error($response)) {
            return array('error' => $response->get_error_message());
        }
        $body = json_decode(wp_remote_retrieve_body($response), true);
        if (isset($body['error'])) {
            return array('error' => $body['error']['message']);
        }
        return $body['content'][0]['text'] ?? __('Error processing request', 'bytesweavers-ai-chat-master');
    }

    /**
     * Call the Gemini API (Google).
     *
     * @param string $message
     * @param string $context
     * @return mixed
     */
    private function call_gemini($message, $context) {
        $url = 'https://generativelanguage.googleapis.com/v1beta/models/' . $this->selected_model . ':generateContent?key=' . $this->gemini_key;
        $headers = array(
            'Content-Type' => 'application/json'
        );

        $full_content = ($context ? $context . "\n\n" : '') . $message;

        $body = array(
            'contents' => array(
                array(
                    'parts' => array(
                        array(
                            'text' => $full_content
                        )
                    )
                )
            ),
            'generationConfig' => array(
                'temperature' => $this->temperature,
                'maxOutputTokens' => $this->max_tokens,
                'topP' => $this->top_p,
                'topK' => 40
            ),
            'safetySettings' => array(
                array(
                    'category' => 'HARM_CATEGORY_HARASSMENT',
                    'threshold' => 'BLOCK_MEDIUM_AND_ABOVE'
                ),
                array(
                    'category' => 'HARM_CATEGORY_HATE_SPEECH',
                    'threshold' => 'BLOCK_MEDIUM_AND_ABOVE'
                ),
                array(
                    'category' => 'HARM_CATEGORY_SEXUALLY_EXPLICIT',
                    'threshold' => 'BLOCK_MEDIUM_AND_ABOVE'
                ),
                array(
                    'category' => 'HARM_CATEGORY_DANGEROUS_CONTENT',
                    'threshold' => 'BLOCK_MEDIUM_AND_ABOVE'
                )
            )
        );

        $response = wp_remote_post($url, array(
            'headers' => $headers,
            'body' => json_encode($body),
            'timeout' => 60
        ));

        if (is_wp_error($response)) {
            return array('error' => $response->get_error_message());
        }
        $body = json_decode(wp_remote_retrieve_body($response), true);
        if (isset($body['error'])) {
            return array('error' => $body['error']['message']);
        }
        return $body['candidates'][0]['content']['parts'][0]['text'] ?? __('Error processing request', 'bytesweavers-ai-chat-master');
    }

    /**
     * Test connection to OpenAI.
     *
     * @param string $api_key
     * @return mixed
     */
    private function test_openai_connection($api_key) {
        $url = 'https://api.openai.com/v1/models';
        $response = wp_remote_get($url, array(
            'headers' => array(
                'Authorization' => 'Bearer ' . $api_key
            )
        ));
        if (is_wp_error($response)) {
            return array('error' => $response->get_error_message());
        }
        $body = json_decode(wp_remote_retrieve_body($response), true);
        return isset($body['data']) ? true : array('error' => $body['error']['message'] ?? __('Unknown error', 'bytesweavers-ai-chat-master'));
    }

    /**
     * Test connection to Anthropic.
     *
     * @param string $api_key
     * @return mixed
     */
    private function test_anthropic_connection($api_key) {
        $url = 'https://api.anthropic.com/v1/messages';
        $response = wp_remote_post($url, array(
            'headers' => array(
                'anthropic-version' => '2024-01-01',
                'x-api-key' => $api_key,
                'Content-Type' => 'application/json'
            ),
            'body' => json_encode(array(
                'model' => 'claude-3-haiku-20240307',
                'messages' => [array('role' => 'user', 'content' => 'Test')]
            ))
        ));
        if (is_wp_error($response)) {
            return array('error' => $response->get_error_message());
        }
        $body = json_decode(wp_remote_retrieve_body($response), true);
        return isset($body['content']) ? true : array('error' => $body['error']['message'] ?? __('Unknown error', 'bytesweavers-ai-chat-master'));
    }

    /**
     * Test connection to Gemini.
     *
     * @param string $api_key
     * @return mixed
     */
    private function test_gemini_connection($api_key) {
        $url = 'https://generativelanguage.googleapis.com/v1beta/models?key=' . $api_key;
        $response = wp_remote_get($url);
        if (is_wp_error($response)) {
            return array('error' => $response->get_error_message());
        }
        $body = json_decode(wp_remote_retrieve_body($response), true);
        return isset($body['models']) ? true : array('error' => $body['error']['message'] ?? __('Unknown error', 'bytesweavers-ai-chat-master'));
    }
}