<?php

/**
 * Admin Settings class for Bytesweavers AI Chat Master.
 *
 * This class registers the admin menu, settings, and handles AJAX requests.
 *
 * @package Bytesweavers_AI_Chat_Master
 */

namespace AICW;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! defined( 'AICW_DEFAULT_OPENAI_KEY' ) ) {
	define( 'AICW_DEFAULT_OPENAI_KEY', '' );
}
if ( ! defined( 'AICW_DEFAULT_ANTHROPIC_KEY' ) ) {
	define( 'AICW_DEFAULT_ANTHROPIC_KEY', '' );
}
if ( ! defined( 'AICW_DEFAULT_GEMINI_KEY' ) ) {
	define( 'AICW_DEFAULT_GEMINI_KEY', '' );
}
if ( ! defined( 'AICW_DEFAULT_SELECTED_MODEL' ) ) {
	define( 'AICW_DEFAULT_SELECTED_MODEL', 'gpt-3.5-turbo' );
}
if ( ! defined( 'AICW_DEFAULT_WIDGET_TITLE' ) ) {
	define( 'AICW_DEFAULT_WIDGET_TITLE', 'AI Assistant' );
}
if ( ! defined( 'AICW_DEFAULT_WIDGET_SUBTITLE' ) ) {
	define( 'AICW_DEFAULT_WIDGET_SUBTITLE', 'How can I help you today?' );
}
if ( ! defined( 'AICW_DEFAULT_PRIMARY_COLOR' ) ) {
	define( 'AICW_DEFAULT_PRIMARY_COLOR', '#007bff' );
}
if ( ! defined( 'AICW_DEFAULT_CHAT_POSITION' ) ) {
	define( 'AICW_DEFAULT_CHAT_POSITION', 'bottom-right' );
}
if ( ! defined( 'AICW_DEFAULT_TEMPERATURE' ) ) {
	define( 'AICW_DEFAULT_TEMPERATURE', 0.7 );
}
if ( ! defined( 'AICW_DEFAULT_MAX_TOKENS' ) ) {
	define( 'AICW_DEFAULT_MAX_TOKENS', 1000 );
}
if ( ! defined( 'AICW_DEFAULT_ENABLE_WIDGET' ) ) {
	define( 'AICW_DEFAULT_ENABLE_WIDGET', false );
}
if ( ! defined( 'AICW_DEFAULT_INSTRUCTIONS' ) ) {
	define( 'AICW_DEFAULT_INSTRUCTIONS', "You are a professional customer support assistant. Your role is to:\n1. Be friendly, professional, and helpful at all times\n2. Help users with questions about our products, services, and website\n3. Provide accurate information based on the website content\n4. If unsure about something, acknowledge it and suggest contacting support directly\n5. Keep responses concise but informative" );
}
if ( ! defined( 'AICW_DEFAULT_KNOWLEDGE_BASE' ) ) {
	define( 'AICW_DEFAULT_KNOWLEDGE_BASE', '' );
}
if ( ! defined( 'AICW_DEFAULT_DISPLAY_OPTION' ) ) {
	define( 'AICW_DEFAULT_DISPLAY_OPTION', 'all' );
}
if ( ! defined( 'AICW_DEFAULT_SPECIFIC_PAGES' ) ) {
	define( 'AICW_DEFAULT_SPECIFIC_PAGES', '' );
}
if ( ! defined( 'AICW_DEFAULT_CHAT_WIDTH' ) ) {
	define( 'AICW_DEFAULT_CHAT_WIDTH', 380 );
}
if ( ! defined( 'AICW_DEFAULT_CHAT_HEIGHT' ) ) {
	define( 'AICW_DEFAULT_CHAT_HEIGHT', 600 );
}
if ( ! defined( 'AICW_DEFAULT_TYPING_INDICATOR_TEXT' ) ) {
	define( 'AICW_DEFAULT_TYPING_INDICATOR_TEXT', 'AI is typing...' );
}
if ( ! defined( 'AICW_DEFAULT_ENABLE_ICON_ANIMATION' ) ) {
	define( 'AICW_DEFAULT_ENABLE_ICON_ANIMATION', true );
}
if ( ! defined( 'AICW_DEFAULT_WIDGET_ICON' ) ) {
	define( 'AICW_DEFAULT_WIDGET_ICON', 'chat' );
}
if ( ! defined( 'AICW_DEFAULT_CUSTOM_ICON_SVG' ) ) {
	define( 'AICW_DEFAULT_CUSTOM_ICON_SVG', '' );
}

class Admin_Settings {

	/**
	 * Constructor.
	 */
	public function __construct() {
		add_action( 'admin_menu', array( $this, 'add_menu_page' ) );
		add_action( 'admin_init', array( $this, 'register_settings' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_scripts' ) );
		add_action( 'wp_ajax_aicw_test_connection', array( $this, 'test_connection' ) );
		add_action( 'wp_ajax_aicw_manual_scrape', array( $this, 'manual_scrape' ) );
		add_action( 'admin_notices', array( $this, 'api_key_notice' ) );
	}

	/**
	 * Add the settings menu page.
	 */
	public function add_menu_page() {
		add_menu_page(
			__( 'Bytesweavers AI Chat Master Settings', 'bytesweavers-ai-chat-master' ),
			__( 'Bytesweavers AI Chat Master', 'bytesweavers-ai-chat-master' ),
			'manage_options',
			'bytesweavers-ai-chat-master',
			array( $this, 'render_settings_page' ),
			'dashicons-format-chat'
		);
	}

	/**
	 * Register plugin settings.
	 */
	public function register_settings() {
		// Apply filters for sanitization.
		add_filter( 'pre_update_option_aicw_openai_key', array( $this, 'sanitize_api_key' ) );
		add_filter( 'pre_update_option_aicw_anthropic_key', array( $this, 'sanitize_api_key' ) );
		add_filter( 'pre_update_option_aicw_gemini_key', array( $this, 'sanitize_api_key' ) );
		add_filter( 'pre_update_option_aicw_selected_model', array( $this, 'validate_model' ) );
		add_filter( 'pre_update_option_aicw_widget_title', 'sanitize_text_field' );
		add_filter( 'pre_update_option_aicw_widget_subtitle', 'sanitize_text_field' );
		add_filter( 'pre_update_option_aicw_primary_color', 'sanitize_hex_color' );
		add_filter( 'pre_update_option_aicw_chat_position', array( $this, 'validate_position' ) );
		add_filter( 'pre_update_option_aicw_temperature', array( $this, 'validate_temperature' ) );
		add_filter( 'pre_update_option_aicw_max_tokens', array( $this, 'validate_max_tokens' ) );
		add_filter( 'pre_update_option_aicw_enable_widget', array( $this, 'validate_enable_widget' ) );
		add_filter( 'pre_update_option_aicw_custom_instructions', 'wp_kses_post' );
		add_filter( 'pre_update_option_aicw_knowledge_base', 'wp_kses_post' );

		// Register new settings for display and appearance options.
		$args = array(
			'type'              => 'string',
			'sanitize_callback' => 'sanitize_text_field',
			'default'           => AICW_DEFAULT_DISPLAY_OPTION,
		);
		register_setting( 'aicw_options', 'aicw_display_option', $args );

		$args = array(
			'type'              => 'string',
			'sanitize_callback' => 'sanitize_text_field',
			'default'           => AICW_DEFAULT_SPECIFIC_PAGES,
		);
		register_setting( 'aicw_options', 'aicw_specific_pages', $args );

		$args = array(
			'type'              => 'integer',
			'sanitize_callback' => 'absint',
			'default'           => AICW_DEFAULT_CHAT_WIDTH,
		);
		register_setting( 'aicw_options', 'aicw_chat_width', $args );

		$args = array(
			'type'              => 'integer',
			'sanitize_callback' => 'absint',
			'default'           => AICW_DEFAULT_CHAT_HEIGHT,
		);
		register_setting( 'aicw_options', 'aicw_chat_height', $args );

		$args = array(
			'type'              => 'string',
			'sanitize_callback' => 'sanitize_text_field',
			'default'           => AICW_DEFAULT_TYPING_INDICATOR_TEXT,
		);
		register_setting( 'aicw_options', 'aicw_typing_indicator_text', $args );

		$args = array(
			'type'              => 'boolean',
			'sanitize_callback' => array( $this, 'validate_boolean' ),
			'default'           => AICW_DEFAULT_ENABLE_ICON_ANIMATION,
		);
		register_setting( 'aicw_options', 'aicw_enable_icon_animation', $args );

		$args = array(
			'type'              => 'string',
			'sanitize_callback' => 'sanitize_text_field',
			'default'           => AICW_DEFAULT_WIDGET_ICON,
		);
		register_setting( 'aicw_options', 'aicw_widget_icon', $args );

		$args = array(
			'type'              => 'string',
			'sanitize_callback' => 'wp_kses_post',
			'default'           => AICW_DEFAULT_CUSTOM_ICON_SVG,
		);
		register_setting( 'aicw_options', 'aicw_custom_icon_svg', $args );

		// Register existing settings.
		$args = array(
			'type'              => 'string',
			'sanitize_callback' => array( $this, 'sanitize_api_key' ),
			'default'           => AICW_DEFAULT_OPENAI_KEY,
		);
		register_setting( 'aicw_options', 'aicw_openai_key', $args );

		$args = array(
			'type'              => 'string',
			'sanitize_callback' => array( $this, 'sanitize_api_key' ),
			'default'           => AICW_DEFAULT_ANTHROPIC_KEY,
		);
		register_setting( 'aicw_options', 'aicw_anthropic_key', $args );

		$args = array(
			'type'              => 'string',
			'sanitize_callback' => array( $this, 'sanitize_api_key' ),
			'default'           => AICW_DEFAULT_GEMINI_KEY,
		);
		register_setting( 'aicw_options', 'aicw_gemini_key', $args );

		$args = array(
			'type'              => 'string',
			'sanitize_callback' => array( $this, 'validate_model' ),
			'default'           => AICW_DEFAULT_SELECTED_MODEL,
		);
		register_setting( 'aicw_options', 'aicw_selected_model', $args );

		$args = array(
			'type'              => 'string',
			'sanitize_callback' => 'sanitize_text_field',
			'default'           => AICW_DEFAULT_WIDGET_TITLE,
		);
		register_setting( 'aicw_options', 'aicw_widget_title', $args );

		$args = array(
			'type'              => 'string',
			'sanitize_callback' => 'sanitize_text_field',
			'default'           => AICW_DEFAULT_WIDGET_SUBTITLE,
		);
		register_setting( 'aicw_options', 'aicw_widget_subtitle', $args );

		$args = array(
			'type'              => 'string',
			'sanitize_callback' => 'sanitize_hex_color',
			'default'           => AICW_DEFAULT_PRIMARY_COLOR,
		);
		register_setting( 'aicw_options', 'aicw_primary_color', $args );

		$args = array(
			'type'              => 'string',
			'sanitize_callback' => array( $this, 'validate_position' ),
			'default'           => AICW_DEFAULT_CHAT_POSITION,
		);
		register_setting( 'aicw_options', 'aicw_chat_position', $args );

		$args = array(
			'type'              => 'number',
			'sanitize_callback' => array( $this, 'validate_temperature' ),
			'default'           => AICW_DEFAULT_TEMPERATURE,
		);
		register_setting( 'aicw_options', 'aicw_temperature', $args );

		$args = array(
			'type'              => 'integer',
			'sanitize_callback' => array( $this, 'validate_max_tokens' ),
			'default'           => AICW_DEFAULT_MAX_TOKENS,
		);
		register_setting( 'aicw_options', 'aicw_max_tokens', $args );

		$args = array(
			'type'              => 'boolean',
			'sanitize_callback' => array( $this, 'validate_enable_widget' ),
			'default'           => AICW_DEFAULT_ENABLE_WIDGET,
		);
		register_setting( 'aicw_options', 'aicw_enable_widget', $args );

		$args = array(
			'type'              => 'string',
			'sanitize_callback' => 'wp_kses_post',
			'default'           => AICW_DEFAULT_INSTRUCTIONS,
		);
		register_setting( 'aicw_options', 'aicw_custom_instructions', $args );

		$args = array(
			'type'              => 'string',
			'sanitize_callback' => 'wp_kses_post',
			'default'           => AICW_DEFAULT_KNOWLEDGE_BASE,
		);
		register_setting( 'aicw_options', 'aicw_knowledge_base', $args );

		// Set default options on first run.
		$this->set_default_options();
	}

	/**
	 * Sanitize chat widget position.
	 *
	 * @param string $position
	 * @return string
	 */
	public function sanitize_position( $position ) {
		$allowed = array( 'bottom-right', 'bottom-left', 'top-right', 'top-left' );
		return in_array( $position, $allowed, true ) ? $position : 'bottom-right';
	}

	/**
	 * Set default options.
	 */
	private function set_default_options() {
		$defaults = array(
			'aicw_primary_color'       => AICW_DEFAULT_PRIMARY_COLOR,
			'aicw_chat_position'       => AICW_DEFAULT_CHAT_POSITION,
			'aicw_temperature'         => AICW_DEFAULT_TEMPERATURE,
			'aicw_max_tokens'          => AICW_DEFAULT_MAX_TOKENS,
			'aicw_top_p'               => '1.0',
			'aicw_frequency_penalty'   => '0.0',
			'aicw_presence_penalty'    => '0.0',
			'aicw_custom_instructions' => AICW_DEFAULT_INSTRUCTIONS,
		);

		foreach ( $defaults as $key => $value ) {
			if ( get_option( $key ) === false ) {
				update_option( $key, $value );
			}
		}
	}

	/**
	 * Sanitize checkbox values.
	 *
	 * @param mixed $input
	 * @return string
	 */
	public function sanitize_checkbox_value( $input ) {
		return ! empty( $input ) ? '1' : '0';
	}

	/**
	 * Validate boolean input.
	 *
	 * @param mixed $input
	 * @return bool
	 */
	public function validate_boolean( $input ) {
		return (bool) $input;
	}

	/**
	 * Show an API key error notice if necessary.
	 */
	public function api_key_notice() {
		if ( ! isset( $_GET['settings-updated'] ) || ! isset( $_GET['page'] ) ||
			! wp_verify_nonce( wp_create_nonce( 'aicw_settings_nonce' ), 'aicw_settings' ) ||
			$_GET['page'] !== 'bytesweavers-ai-chat-master'
		) {
			return;
		}

		if ( isset( $_GET['settings-updated'] ) && ( isset( $_GET['page'] ) && $_GET['page'] === 'bytesweavers-ai-chat-master' ) ) {
			$model        = get_option( 'aicw_selected_model', 'gpt-3.5-turbo' );
			$model_family = $this->get_model_family( $model );
			$api_key      = '';
			switch ( $model_family ) {
				case 'openai':
					$api_key = get_option( 'aicw_openai_key' );
					break;
				case 'anthropic':
					$api_key = get_option( 'aicw_anthropic_key' );
					break;
				case 'gemini':
					$api_key = get_option( 'aicw_gemini_key' );
					break;
			}
			if ( empty( $api_key ) && get_option( 'aicw_enable_widget' ) == 1 ) {
				echo '<div class="notice notice-error is-dismissible">
					<p><strong>' . esc_html__( 'Error:', 'bytesweavers-ai-chat-master' ) . '</strong> ' . esc_html__( 'Please set up an API key in the API settings before enabling the AI Chat Widget.', 'bytesweavers-ai-chat-master' ) . ' <a href="' . esc_url( admin_url( 'admin.php?page=bytesweavers-ai-chat-master&api-settings=1' ) ) . '">' . esc_html__( 'Go to API settings', 'bytesweavers-ai-chat-master' ) . '</a></p>
				</div>';
				update_option( 'aicw_enable_widget', 0 );
			}
		}
	}

	/**
	 * Enqueue admin scripts and styles.
	 *
	 * @param string $hook
	 */
	public function enqueue_admin_scripts( $hook ) {
		if ( 'toplevel_page_bytesweavers-ai-chat-master' !== $hook ) {
			return;
		}
		wp_enqueue_media();
		wp_enqueue_style( 'wp-color-picker' );
		wp_enqueue_script( 'wp-color-picker' );
		wp_enqueue_style( 'aicw-admin', AICW_PLUGIN_URL . 'assets/css/admin.css', array(), AICW_VERSION );
		wp_enqueue_script( 'aicw-admin', AICW_PLUGIN_URL . 'assets/js/admin.js', array( 'jquery', 'wp-color-picker' ), AICW_VERSION, true );
		wp_localize_script( 'aicw-admin', 'aicwSettings', array(
			'ajaxurl' => admin_url( 'admin-ajax.php' ),
			'nonce'   => wp_create_nonce( 'aicw-admin-nonce' )
		) );
		
		// Add inline script properly via wp_add_inline_script 
		wp_add_inline_script( 'aicw-admin', "document.addEventListener('DOMContentLoaded', function() { 
			var urlParams = new URLSearchParams(window.location.search); 
			if (urlParams.has('api-settings')) { 
				jQuery('.nav-tab-wrapper a[href=\"#api\"]').trigger('click'); 
			} 
		});" );
	}

	/**
	 * Handle AJAX request to test the API connection.
	 */
	public function test_connection() {
		check_ajax_referer( 'aicw-admin-nonce', 'nonce' );
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( __( 'Unauthorized', 'bytesweavers-ai-chat-master' ) );
		}
		$model   = isset( $_POST['model'] ) ? sanitize_text_field( wp_unslash( $_POST['model'] ) ) : '';
		$api_key = isset( $_POST['api_key'] ) ? sanitize_text_field( wp_unslash( $_POST['api_key'] ) ) : '';
		$api_handler = new API_Handler();
		$result      = $api_handler->test_connection( $model, $api_key );
		if ( isset( $result['error'] ) ) {
			wp_send_json_error( $result['error'] );
		}
		wp_send_json_success( __( 'Connection successful!', 'bytesweavers-ai-chat-master' ) );
	}

	/**
	 * Handle AJAX request to manually scrape website content.
	 */
	public function manual_scrape() {
		check_ajax_referer( 'aicw-admin-nonce', 'nonce' );
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( __( 'Unauthorized', 'bytesweavers-ai-chat-master' ) );
		}
		$model = get_option( 'aicw_selected_model', 'gpt-3.5-turbo' );
		$api_handler = new API_Handler();
		$model_family = $this->get_model_family( $model );
		$api_key = '';
		switch ( $model_family ) {
			case 'openai':
				$api_key = get_option( 'aicw_openai_key' );
				break;
			case 'anthropic':
				$api_key = get_option( 'aicw_anthropic_key' );
				break;
			case 'gemini':
				$api_key = get_option( 'aicw_gemini_key' );
				break;
		}
		if ( empty( $api_key ) ) {
			wp_send_json_error( __( 'Please set an API key first before scraping.', 'bytesweavers-ai-chat-master' ) );
		}
		$scraper = new Data_Scraper();
		$result  = $scraper->scrape_website();
		if ( is_wp_error( $result ) ) {
			wp_send_json_error( $result->get_error_message() );
		}
		wp_send_json_success( array(
			'message' => __( 'Website content scraped successfully!', 'bytesweavers-ai-chat-master' ),
			'summary' => $result['summaries']
		) );
	}

	/**
	 * Retrieve the model family based on the model.
	 *
	 * @param string $model
	 * @return string|null
	 */
	private function get_model_family( $model ) {
		if ( strpos( $model, 'gpt' ) !== false ) {
			return 'openai';
		} elseif ( strpos( $model, 'claude' ) !== false ) {
			return 'anthropic';
		} elseif ( strpos( $model, 'gemini' ) !== false ) {
			return 'gemini';
		}
		return null;
	}

	/**
	 * Render the admin settings page.
	 */
	public function render_settings_page() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}
		$available_models = $this->get_available_models();
		include AICW_PLUGIN_DIR . 'templates/admin-settings.php';
	}

	/**
	 * Return available AI models.
	 *
	 * @return array
	 */
	public function get_available_models() {
		return array(
			'openai'   => array(
				'label'  => 'OpenAI Models',
				'models' => array(
					'gpt-4o-2024-08-06'     => 'GPT-4o (Ultra-fast, high accuracy)',
					'gpt-4o-mini-2024-07-18'  => 'GPT-4o Mini (Budget-friendly, efficient)',
					'chatgpt-4o-latest'      => 'ChatGPT-4o Latest (Enhanced reasoning)',
					'gpt-4-turbo-preview'    => 'GPT-4 Turbo (128K context)',
					'gpt-4'                  => 'GPT-4 (8K context)',
					'gpt-3.5-turbo'          => 'GPT-3.5 Turbo (16K context)',
				),
			),
			'anthropic' => array(
				'label'  => 'Anthropic Models',
				'models' => array(
					'claude-3-7-sonnet-20250219' => 'Claude 3.7 Sonnet (Latest & most advanced)',
					'claude-3-5-sonnet-20241022' => 'Claude 3.5 Sonnet (High intelligence)',
					'claude-3-5-haiku-20241022'  => 'Claude 3.5 Haiku (Fast & cost-effective)',
					'claude-3-opus-20240229'     => 'Claude 3 Opus (200K context)',
					'claude-3-sonnet-20240229'   => 'Claude 3 Sonnet (200K context)',
					'claude-3-haiku-20240307'    => 'Claude 3 Haiku (200K context)',
				),
			),
			'gemini'   => array(
				'label'  => 'Google Models',
				'models' => array(
					'gemini-2.0-pro-exp-02-05'          => 'Gemini 2.0 Pro (2M context)',
					'gemini-2.0-flash'                  => 'Gemini 2.0 Flash (Balanced performance)',
					'gemini-2.0-flash-lite-preview-02-05' => 'Gemini 2.0 Flash-Lite (Cost-effective)',
					'gemini-1.5-pro'                    => 'Gemini 1.5 Pro (1M context)',
					'gemini-1.5-flash'                  => 'Gemini 1.5 Flash (1M context)',
				),
			),
		);
	}

	/**
	 * Sanitize API keys - ensures only valid characters are allowed.
	 *
	 * @param string $key
	 * @return string
	 */
	public function sanitize_api_key( $key ) {
		$sanitized = preg_replace( '/[^a-zA-Z0-9_\-]/', '', $key );
		if ( $this->validate_api_key( $sanitized ) instanceof \WP_Error ) {
			return '';
		}
		return $sanitized;
	}

	/**
	 * Validate API key.
	 *
	 * @param mixed $value The value to validate.
	 * @return mixed The validated value or WP_Error.
	 */
	public function validate_api_key( $value ) {
		if ( empty( $value ) ) {
			return new \WP_Error(
				'invalid_api_key',
				__( 'API key cannot be empty.', 'bytesweavers-ai-chat-master' )
			);
		}
		if ( ! preg_match( '/^[a-zA-Z0-9_\-]+$/', $value ) ) {
			return new \WP_Error(
				'invalid_api_key_format',
				__( 'API key contains invalid characters.', 'bytesweavers-ai-chat-master' )
			);
		}
		return $value;
	}

	/**
	 * Validate text field input.
	 *
	 * @param string $input
	 * @return string
	 */
	public function validate_text_field( $input ) {
		return sanitize_text_field( $input );
	}

	/**
	 * Validate color input.
	 *
	 * @param string $input
	 * @return string
	 */
	public function validate_color( $input ) {
		$color = sanitize_hex_color( $input );
		return $color ? $color : '#007bff';
	}

	/**
	 * Validate position.
	 *
	 * @param string $input
	 * @return string
	 */
	public function validate_position( $input ) {
		$allowed = array( 'bottom-right', 'bottom-left', 'top-right', 'top-left' );
		return in_array( $input, $allowed, true ) ? $input : 'bottom-right';
	}

	/**
	 * Validate temperature.
	 *
	 * @param mixed $input
	 * @return float
	 */
	public function validate_temperature( $input ) {
		$temp = (float) $input;
		return min( max( $temp, 0 ), 1 );
	}

	/**
	 * Validate max tokens.
	 *
	 * @param mixed $input
	 * @return int
	 */
	public function validate_max_tokens( $input ) {
		$tokens = (int) $input;
		return min( max( $tokens, 100 ), 4000 );
	}

	/**
	 * Validate instructions.
	 *
	 * @param string $input
	 * @return string
	 */
	public function validate_instructions( $input ) {
		return wp_kses_post( $input );
	}

	/**
	 * Validate knowledge base.
	 *
	 * @param string $input
	 * @return string
	 */
	public function validate_knowledge_base( $input ) {
		return wp_kses_post( $input );
	}

	/**
	 * Validate model selection.
	 *
	 * @param string $input
	 * @return string
	 */
	public function validate_model( $input ) {
		$available_models = $this->get_available_models();
		$all_models       = array();
		foreach ( $available_models as $family ) {
			$all_models = array_merge( $all_models, array_keys( $family['models'] ) );
		}
		return in_array( $input, $all_models, true ) ? $input : 'gpt-3.5-turbo';
	}
	
	/**
	 * Validate enable widget.
	 *
	 * @param mixed $input
	 * @return int
	 */
	public function validate_enable_widget( $input ) {
		return ! empty( $input ) ? 1 : 0;
	}
}