/**
 * Marked - A markdown parser and compiler.
 *
 * This library takes markdown source text and converts it into HTML.
 * It provides a lexer to tokenize the input and a parser/renderer to
 * compile the tokens into HTML.
 */

(function () {
    /**
     * Lexer constructor.
     * Responsible for converting markdown text into an array of tokens.
     *
     * @param {Object} options - Configuration options.
     */
    function Lexer(options) {
      this.tokens = [];
      this.tokens.links = {}; // Stores reference links found in the text.
      this.options = options || marked.defaults;
      this.rules = block.normal; // Default block-level rules.
      if (this.options.gfm) {
        // Use GitHub-flavored markdown rules if specified.
        this.rules = this.options.tables ? block.tables : block.gfm;
      }
    }
  
    /**
     * InlineLexer constructor.
     * Responsible for parsing inline markdown syntax (e.g. emphasis, links, code spans).
     *
     * @param {Object} links - The links object from the lexer tokens.
     * @param {Object} options - Configuration options.
     */
    function InlineLexer(links, options) {
      this.options = options || marked.defaults;
      this.links = links;
      this.rules = inline.normal; // Default inline rules.
      // Set renderer if not provided.
      this.renderer = this.options.renderer || new Renderer();
      this.renderer.options = this.options;
      if (!this.links)
        throw new Error("Tokens array requires a `links` property.");
      if (this.options.gfm) {
        this.rules = this.options.breaks ? inline.breaks : inline.gfm;
      } else if (this.options.pedantic) {
        this.rules = inline.pedantic;
      }
    }
  
    /**
     * Renderer constructor.
     * Defines methods for producing HTML from tokens.
     *
     * @param {Object} options - Configuration options.
     */
    function Renderer(options) {
      this.options = options || {};
    }
  
    /**
     * Parser constructor.
     * Responsible for taking an array of tokens and converting them into HTML.
     *
     * @param {Object} options - Configuration options.
     */
    function Parser(options) {
      this.tokens = [];
      this.token = null;
      this.options = options || marked.defaults;
      // Use a default renderer if one is not provided.
      this.options.renderer = this.options.renderer || new Renderer();
      this.renderer = this.options.renderer;
      this.renderer.options = this.options;
    }
  
    /**
     * Escape HTML characters in the given string.
     *
     * @param {string} src - Source string.
     * @param {boolean} encode - If true, also encode existing HTML entities.
     * @returns {string} - Escaped HTML.
     */
    function escapeHtml(src, encode) {
      return src
        .replace(encode ? /&/g : /&(?!#?\w+;)/g, "&amp;")
        .replace(/</g, "&lt;")
        .replace(/>/g, "&gt;")
        .replace(/"/g, "&quot;")
        .replace(/'/g, "&#39;");
    }
  
    /**
     * Unescape HTML entities in a string.
     *
     * @param {string} src - The string with HTML entities.
     * @returns {string} - The unescaped string.
     */
    function unescapeHtml(src) {
      return src.replace(/&([#\w]+);/g, function (match, n) {
        n = n.toLowerCase();
        if (n === "colon") return ":";
        if (n.charAt(0) === "#")
          return String.fromCharCode(n.charAt(1) === "x" ? parseInt(n.substring(2), 16) : +n.substring(1));
        return "";
      });
    }
  
    /**
     * Helper function to create a new regular expression by replacing parts
     * of an existing regular expression.
     *
     * @param {RegExp} regex - Original regex.
     * @param {string} opt - Additional options/flags.
     * @returns {Function} - A function accepting replacements and returns a new RegExp.
     */
    function replace(regex, opt) {
      regex = regex.source;
      opt = opt || "";
      return function () {
        var i = 0,
          args = arguments,
          len = args.length;
        while (i < len) {
          // Replace each placeholder by the provided argument.
          regex = regex.replace(args[i], args[++i]);
        }
        return new RegExp(regex, opt);
      };
    }
  
    // No-operation function (an empty placeholder).
    function noop() {}
  
    /**
     * Merge multiple objects into one.
     *
     * @param {Object} target - The target object.
     * @returns {Object} - The extended object.
     */
    function merge(target) {
      for (var i = 1, obj; i < arguments.length; i++) {
        obj = arguments[i];
        for (var key in obj) {
          if (Object.prototype.hasOwnProperty.call(obj, key)) {
            target[key] = obj[key];
          }
        }
      }
      return target;
    }
  
    /**
     * The main marked function.
     * Converts a markdown string (src) to HTML by lexing and parsing.
     *
     * @param {string} src - Markdown source string.
     * @param {Object|function} [opt] - Options or callback.
     * @param {function} [callback] - Callback function if using async highlighting.
     * @returns {string|undefined} - Returns HTML string if synchronous.
     */
    function marked(src, opt, callback) {
      if (callback || typeof opt === "function") {
        callback = callback || opt;
        opt = merge({}, marked.defaults, opt || {});
        var highlight = opt.highlight,
          tokens,
          pending = 0;
        try {
          tokens = Lexer.lex(src, opt);
        } catch (err) {
          return callback(err);
        }
        var done = function (err) {
          if (err) return callback(err);
          callback(null, Parser.parse(tokens, opt));
        };
        if (!highlight || highlight.length < 3) return done();
        if (!tokens.length) return done();
        for (var i = 0; i < tokens.length; i++) {
          (function (token) {
            if (token.type !== "code") {
              pending--;
              if (!pending) done();
              return;
            }
            highlight(token.text, token.lang, function (err, code) {
              if (err) return done(err);
              if (code != null && code !== token.text) {
                token.text = code;
                token.escaped = true;
              }
              pending--;
              if (!pending) done();
            });
          })(tokens[i]),
            pending++;
        }
      } else {
        try {
          opt = merge({}, marked.defaults, opt);
          return Parser.parse(Lexer.lex(src, opt), opt);
        } catch (e) {
          e.message += "\nPlease report this to https://github.com/chjj/marked.";
          if (opt.silent)
            return "<p>An error occured:</p><pre>" + escapeHtml(e.message, true) + "</pre>";
          throw e;
        }
      }
    }
  
    // ----------------------------
    // Block-Level Grammar Rules
    // ----------------------------
    var block = {
      newline: /^\n+/,
      code: /^( {4}[^\n]+\n*)+/,
      fences: noop, // Fenced code blocks will be defined later.
      hr: /^( *[-*_]){3,} *(?:\n+|$)/,
      heading: /^ *(#{1,6}) *([^\n]+?) *#* *(?:\n+|$)/,
      nptable: noop, // Nottables (GFM) defined later.
      lheading: /^([^\n]+)\n *(=|-){2,} *(?:\n+|$)/,
      blockquote: /^( *>[^\n]+(\n(?!def)[^\n]+)*\n*)+/,
      list: /^( *)(bull) [\s\S]+?(?:hr|def|\n{2,}(?! )(?!\1bull )\n*|\s*$)/,
      html: /^ *(?:comment *(?:\n|\s*$)|closed *(?:\n{2,}|\s*$)|closing *(?:\n{2,}|\s*$))/,
      def: /^ *\[([^\]]+)\]: *<?([^\s>]+)>?(?: +["(]([^\n]+)[")])? *(?:\n+|$)/,
      table: noop, // Table support defined later.
      paragraph: /^((?:[^\n]+\n?(?!hr|heading|lheading|blockquote|tag|def))+)\n*/,
      text: /^[^\n]+/
    };
  
    // Additional properties for block rules.
    block.bullet = /(?:[*+-]|\d+\.)/;
    block.item = /^( *)(bull) [^\n]*(?:\n(?!\1bull )[^\n]*)*/;
    block.item = replace(block.item, "gm")(/bull/g, block.bullet)();
    block.list = replace(block.list)(/bull/g, block.bullet)(
      "hr",
      "\\n+(?=\\1?(?:[-*_] *){3,}(?:\\n+|$))"
    )("def", "\\n+(?=" + block.def.source + ")")();
    block.blockquote = replace(block.blockquote)("def", block.def)();
    block._tag =
      "(?!(?:a|em|strong|small|s|cite|q|dfn|abbr|data|time|code|var|samp|kbd|sub|sup|i|b|u|mark|ruby|rt|rp|bdi|bdo|span|br|wbr|ins|del|img)\\b)\\w+(?!:/|[^\\w\\s@]*@)\\b";
    block.html = replace(block.html)("comment", /<!--[\s\S]*?-->/)(
      "closed",
      /<(tag)[\s\S]+?<\/\1>/
    )("closing", /<tag(?:"[^"]*"|'[^']*'|[^'">])*?>/)(/tag/g, block._tag)();
    block.paragraph = replace(block.paragraph)(
      "hr",
      block.hr
    )("heading", block.heading)("lheading", block.lheading)(
      "blockquote",
      block.blockquote
    )("tag", "<" + block._tag)("def", block.def)();
  
    block.normal = merge({}, block);
    block.gfm = merge({}, block.normal, {
      fences: /^ *(`{3,}|~{3,}) *(\S+)? *\n([\s\S]+?)\s*\1 *(?:\n+|$)/,
      paragraph: /^/
    });
    block.gfm.paragraph = replace(block.paragraph)(
      "(?!",
      "(?!" +
        block.gfm.fences.source.replace("\\1", "\\2") +
        "|" +
        block.list.source.replace("\\1", "\\3") +
        "|"
    )();
    block.tables = merge({}, block.gfm, {
      nptable:
        /^ *(\S.*\|.*)\n *([-:]+ *\|[-| :]*)\n((?:.*\|.*(?:\n|$))*)\n*/,
      table:
        /^ *\|(.+)\n *\|( *[-:]+[-| :]*)\n((?: *\|.*(?:\n|$))*)\n*/
    });
  
    // Attach the block rules to the Lexer.
    Lexer.rules = block;
    Lexer.lex = function (src, options) {
      var lexer = new Lexer(options);
      return lexer.lex(src);
    };
    Lexer.prototype.lex = function (src) {
      src = src
        .replace(/\r\n|\r/g, "\n")
        .replace(/\t/g, "    ")
        .replace(/\u00a0/g, " ")
        .replace(/\u2424/g, "\n");
      return this.token(src, true);
    };
    Lexer.prototype.token = function (src, top, bqlist) {
      var cap, loose, next, bull, b, item, space, i, l;
      src = src.replace(/^ +$/gm, "");
      while (src) {
        // Newline.
        if ((cap = this.rules.newline.exec(src))) {
          src = src.substring(cap[0].length);
          if (cap[0].length > 1) {
            this.tokens.push({ type: "space" });
          }
        }
        // Code block.
        else if ((cap = this.rules.code.exec(src))) {
          src = src.substring(cap[0].length);
          cap = cap[0].replace(/^ {4}/gm, "");
          this.tokens.push({
            type: "code",
            text: this.options.pedantic ? cap : cap.replace(/\n+$/, "")
          });
        }
        // Fenced code block.
        else if ((cap = this.rules.fences.exec(src))) {
          src = src.substring(cap[0].length);
          this.tokens.push({
            type: "code",
            lang: cap[2],
            text: cap[3]
          });
        }
        // Heading.
        else if ((cap = this.rules.heading.exec(src))) {
          src = src.substring(cap[0].length);
          this.tokens.push({
            type: "heading",
            depth: cap[1].length,
            text: cap[2]
          });
        }
        // GFM table (nptable).
        else if (top && (cap = this.rules.nptable.exec(src))) {
          src = src.substring(cap[0].length);
          var table = {
            type: "table",
            header: cap[1].replace(/^ *| *\| *$/g, "").split(/ *\| */),
            align: cap[2].replace(/^ *|\| *$/g, "").split(/ *\| */),
            cells: cap[3].replace(/\n$/, "").split("\n")
          };
          for (i = 0; i < table.align.length; i++) {
            if (/^ *-+: *$/.test(table.align[i])) {
              table.align[i] = "right";
            } else if (/^ *:-+: *$/.test(table.align[i])) {
              table.align[i] = "center";
            } else if (/^ *:-+ *$/.test(table.align[i])) {
              table.align[i] = "left";
            } else {
              table.align[i] = null;
            }
          }
          for (i = 0; i < table.cells.length; i++) {
            table.cells[i] = table.cells[i].split(/ *\| */);
          }
          this.tokens.push(table);
        }
        // Setext heading.
        else if ((cap = this.rules.lheading.exec(src))) {
          src = src.substring(cap[0].length);
          this.tokens.push({
            type: "heading",
            depth: cap[2] === "=" ? 1 : 2,
            text: cap[1]
          });
        }
        // Horizontal rule.
        else if ((cap = this.rules.hr.exec(src))) {
          src = src.substring(cap[0].length);
          this.tokens.push({ type: "hr" });
        }
        // Blockquote.
        else if ((cap = this.rules.blockquote.exec(src))) {
          src = src.substring(cap[0].length);
          this.tokens.push({ type: "blockquote_start" });
          cap = cap[0].replace(/^ *> ?/gm, "");
          this.token(cap, top, true);
          this.tokens.push({ type: "blockquote_end" });
        }
        // List.
        else if ((cap = this.rules.list.exec(src))) {
          src = src.substring(cap[0].length);
          bull = cap[2];
          this.tokens.push({ type: "list_start", ordered: bull.length > 1 });
          cap = cap[0].match(this.rules.item);
          loose = !/\n\n(?!\s*$)/.test(cap[0]);
          for (i = 0, l = cap.length; i < l; i++) {
            item = cap[i];
            space = item.length;
            item = item.replace(/^ *([*+-]|\d+\.) +/, "");
            if (~item.indexOf("\n ")) {
              space -= item.length;
              item = this.options.pedantic
                ? item.replace(/^ {1,4}/gm, "")
                : item.replace(new RegExp("^ {1," + space + "}", "gm"), "");
            }
            if (this.options.smartLists && i !== l - 1) {
              next = block.bullet.exec(cap[i + 1])[0];
              if (bull !== next && !(bull.length > 1 && next.length > 1)) {
                src = cap.slice(i + 1).join("\n") + src;
                l = i + 1;
              }
            }
            loose = loose || /\n\n(?!\s*$)/.test(item);
            this.tokens.push({ type: loose ? "loose_item_start" : "list_item_start" });
            this.token(item, false, false);
            this.tokens.push({ type: "list_item_end" });
          }
          this.tokens.push({ type: "list_end" });
        }
        // HTML block.
        else if ((cap = this.rules.html.exec(src))) {
          src = src.substring(cap[0].length);
          this.tokens.push({
            type: this.options.sanitize ? "paragraph" : "html",
            pre: cap[1] === "pre" || cap[1] === "script" || cap[1] === "style",
            text: cap[0]
          });
        }
        // Link definition.
        else if (!bqlist && top && (cap = this.rules.def.exec(src))) {
          src = src.substring(cap[0].length);
          this.tokens.links[cap[1].toLowerCase()] = {
            href: cap[2],
            title: cap[3]
          };
        }
        // Table block.
        else if (top && (cap = this.rules.table.exec(src))) {
          src = src.substring(cap[0].length);
          var tbl = {
            type: "table",
            header: cap[1].replace(/^ *| *\| *$/g, "").split(/ *\| */),
            align: cap[2].replace(/^ *|\| *$/g, "").split(/ *\| */),
            cells: cap[3].replace(/(?: *\| *)?\n$/, "").split("\n")
          };
          for (i = 0; i < tbl.align.length; i++) {
            if (/^ *-+: *$/.test(tbl.align[i])) {
              tbl.align[i] = "right";
            } else if (/^ *:-+: *$/.test(tbl.align[i])) {
              tbl.align[i] = "center";
            } else if (/^ *:-+ *$/.test(tbl.align[i])) {
              tbl.align[i] = "left";
            } else {
              tbl.align[i] = null;
            }
          }
          for (i = 0; i < tbl.cells.length; i++) {
            tbl.cells[i] = tbl.cells[i]
              .replace(/^ *\| *| *\| *$/g, "")
              .split(/ *\| */);
          }
          this.tokens.push(tbl);
        }
        // Paragraph.
        else if (top && (cap = this.rules.paragraph.exec(src))) {
          src = src.substring(cap[0].length);
          this.tokens.push({
            type: "paragraph",
            text: cap[1].charAt(cap[1].length - 1) === "\n" ? cap[1].slice(0, -1) : cap[1]
          });
        }
        // Text.
        else if ((cap = this.rules.text.exec(src))) {
          src = src.substring(cap[0].length);
          this.tokens.push({ type: "text", text: cap[0] });
        }
        // Prevent enternal loop.
        else if (src) {
          throw new Error("Infinite loop on byte: " + src.charCodeAt(0));
        }
      }
      return this.tokens;
    };
  
    // ----------------------------
    // Inline-Level Grammar Rules
    // ----------------------------
    var inline = {
      escape: /^\\([\\`*{}\[\]()#+\-.!_>])/,
      autolink: /^<([^ >]+(@|:\/)[^ >]+)>/,
      url: noop,
      tag: /^<!--[\s\S]*?-->|^<\/?\w+(?:"[^"]*"|'[^']*'|[^'">])*?>/,
      link: /^!?\[(inside)\]\(href\)/,
      reflink: /^!?\[(inside)\]\s*\[([^\]]*)\]/,
      nolink: /^!?\[((?:\[[^\]]*\]|[^\[\]])*)\]/,
      strong: /^__([\s\S]+?)__(?!_)|^\*\*([\s\S]+?)\*\*(?!\*)/,
      em: /^\b_((?:__|[\s\S])+?)_\b|^\*((?:\*\*|[\s\S])+?)\*(?!\*)/,
      code: /^(`+)\s*([\s\S]*?[^`])\s*\1(?!`)/,
      br: /^ {2,}\n(?!\s*$)/,
      del: noop,
      text: /^[\s\S]+?(?=[\\<!\[_*`]| {2,}\n|$)/
    };
  
    // Internal subexpressions.
    inline._inside = /(?:\[[^\]]*\]|[^\[\]]|\](?=[^\[]*\]))*/;
    inline._href = /\s*<?([\s\S]*?)>?(?:\s+['"]([\s\S]*?)['"])?\s*/;
    inline.link = replace(inline.link)("inside", inline._inside)("href", inline._href)();
    inline.reflink = replace(inline.reflink)("inside", inline._inside)();
    inline.normal = merge({}, inline);
    inline.pedantic = merge({}, inline.normal, {
      strong: /^__(?=\S)([\s\S]*?\S)__(?!_)|^\*\*(?=\S)([\s\S]*?\S)\*\*(?!\*)/,
      em: /^_(?=\S)([\s\S]*?\S)_(?!_)|^\*(?=\S)([\s\S]*?\S)\*(?!\*)/
    });
    inline.gfm = merge({}, inline.normal, {
      escape: replace(inline.escape)("])", "~|])")(),
      url: /^(https?:\/\/[^\s<]+[^<.,:;"')\]\s])/
    });
    inline.gfm.del = /^~~(?=\S)([\s\S]*?\S)~~/;
    inline.gfm.text = replace(inline.text)("]|", "~]|")("|", "|https?://|")();
    inline.breaks = merge({}, inline.gfm, {
      br: replace(inline.br)("{2,}", "*")(),
      text: replace(inline.gfm.text)("{2,}", "*")()
    });
  
    // Attach the inline rules to the InlineLexer.
    InlineLexer.rules = inline;
    InlineLexer.output = function (src) {
      var inlineLexer = new InlineLexer(this.links, this.options, this.renderer);
      return inlineLexer.output(src);
    };
  
    /**
     * Process inline markdown and output HTML.
     *
     * @param {string} src - Inline markdown text.
     * @returns {string} - HTML output.
     */
    InlineLexer.prototype.output = function (src) {
      var out = "";
      var cap, text;
      while (src) {
        // Escape characters.
        if ((cap = this.rules.escape.exec(src))) {
          src = src.substring(cap[0].length);
          out += cap[1];
          continue;
        }
        // Autolink.
        if ((cap = this.rules.autolink.exec(src))) {
          src = src.substring(cap[0].length);
          if (cap[2] === "@") {
            text = this.mangle(":" === cap[1].charAt(6) ? cap[1].substring(7) : cap[1]);
            out += this.renderer.link(text, null, text);
          } else {
            text = escapeHtml(cap[1]);
            out += this.renderer.link(text, null, text);
          }
          continue;
        }
        // URL (if not already inside a link).
        if (!this.inLink && !(cap = this.rules.url.exec(src))) {
          if ((cap = this.rules.tag.exec(src))) {
            if (!this.inLink && /^<a /i.test(cap[0])) {
              this.inLink = true;
            } else if (this.inLink && /^<\/a>/i.test(cap[0])) {
              this.inLink = false;
            }
            src = src.substring(cap[0].length);
            out += this.options.sanitize ? escapeHtml(cap[0]) : cap[0];
            continue;
          }
          if ((cap = this.rules.link.exec(src))) {
            src = src.substring(cap[0].length);
            this.inLink = true;
            out += this.outputLink(cap, { href: cap[2], title: cap[3] });
            this.inLink = false;
            continue;
          }
          if ((cap = this.rules.reflink.exec(src)) || (cap = this.rules.nolink.exec(src))) {
            src = src.substring(cap[0].length);
            text = (cap[2] || cap[1]).replace(/\s+/g, " ");
            text = this.links[text.toLowerCase()];
            if (!text || !text.href) {
              out += cap[0].charAt(0);
              src = cap[0].substring(1) + src;
              continue;
            }
            this.inLink = true;
            out += this.outputLink(cap, text);
            this.inLink = false;
            continue;
          }
          if ((cap = this.rules.strong.exec(src))) {
            src = src.substring(cap[0].length);
            out += this.renderer.strong(this.output(cap[2] || cap[1]));
            continue;
          }
          if ((cap = this.rules.em.exec(src))) {
            src = src.substring(cap[0].length);
            out += this.renderer.em(this.output(cap[2] || cap[1]));
            continue;
          }
          if ((cap = this.rules.code.exec(src))) {
            src = src.substring(cap[0].length);
            out += this.renderer.codespan(escapeHtml(cap[2], true));
            continue;
          }
          if ((cap = this.rules.br.exec(src))) {
            src = src.substring(cap[0].length);
            out += this.renderer.br();
            continue;
          }
          if ((cap = this.rules.del.exec(src))) {
            src = src.substring(cap[0].length);
            out += this.renderer.del(this.output(cap[1]));
            continue;
          }
          if ((cap = this.rules.text.exec(src))) {
            src = src.substring(cap[0].length);
            out += escapeHtml(this.smartypants(cap[0]));
            continue;
          }
          if (src) {
            throw new Error("Infinite loop on byte: " + src.charCodeAt(0));
          }
        } else {
          src = src.substring(cap[0].length);
          text = escapeHtml(cap[1]);
          out += this.renderer.link(text, null, text);
        }
      }
      return out;
    };
  
    /**
     * Apply smart punctuation transformations.
     *
     * @param {string} text - Input text.
     * @returns {string} - Transformed text.
     */
    InlineLexer.prototype.smartypants = function (text) {
      if (this.options.smartypants) {
        text = text
          .replace(/--/g, "—")
          .replace(/(^|[-\u2014\/(\[{"\s])'/g, "$1‘")
          .replace(/'/g, "’")
          .replace(/(^|[-\u2014\/(\[{\u2018\s])"/g, "$1“")
          .replace(/"/g, "”")
          .replace(/\.{3}/g, "…");
      }
      return text;
    };
  
    /**
     * Mangle (randomize) a string to help obfuscate email addresses.
     *
     * @param {string} text - Text to mangle.
     * @returns {string} - Mangled text.
     */
    InlineLexer.prototype.mangle = function (text) {
      var out = "";
      for (var i = 0, len = text.length; i < len; i++) {
        var charCode = text.charCodeAt(i);
        if (Math.random() > 0.5) {
          charCode = "x" + charCode.toString(16);
        }
        out += "&#" + charCode + ";";
      }
      return out;
    };
  
    /**
     * Outputs an inline link or image.
     *
     * @param {Array} cap - Captured regex groups.
     * @param {Object} link - Object containing href and title.
     * @returns {string} - HTML for the link or image.
     */
    InlineLexer.prototype.outputLink = function (cap, link) {
      var href = escapeHtml(link.href),
        title = link.title ? escapeHtml(link.title) : null;
      if (cap[0].charAt(0) !== "!") {
        return this.renderer.link(href, title, this.output(cap[1]));
      } else {
        return this.renderer.image(href, title, escapeHtml(cap[1]));
      }
    };
  
    // Expose the InlineLexer to marked.
    marked.InlineLexer = InlineLexer;
  
    // ----------------------------
    // Parser Methods
    // ----------------------------
  
    /**
     * Static parse function.
     * @param {Array} tokens - Array of tokens produced by the lexer.
     * @param {Object} options - Configuration options.
     * @param {Renderer} renderer - Instance of the renderer.
     * @returns {string} - HTML output.
     */
    Parser.parse = function (tokens, options, renderer) {
      var parser = new Parser(options, renderer);
      return parser.parse(tokens);
    };
  
    Parser.prototype.next = function () {
      this.token = this.tokens.pop();
      return this.token;
    };
  
    Parser.prototype.peek = function () {
      return this.tokens[this.tokens.length - 1] || 0;
    };
  
    /**
     * Combine adjacent text tokens.
     *
     * @returns {string} - Combined text.
     */
    Parser.prototype.parseText = function () {
      var body = this.token.text;
      while (this.peek().type === "text") {
        body += "\n" + this.next().text;
      }
      return this.inline.output(body);
    };
  
    /**
     * Process a single token and return its HTML output.
     *
     * @returns {string} - HTML string.
     */
    Parser.prototype.tok = function () {
      switch (this.token.type) {
        case "space":
          return "";
        case "hr":
          return this.renderer.hr();
        case "heading":
          return this.renderer.heading(this.inline.output(this.token.text), this.token.depth, this.token.text);
        case "code":
          return this.renderer.code(this.token.text, this.token.lang, this.token.escaped);
        case "table":
          var header = "",
            body = "",
            i,
            row;
          for (i = 0; i < this.token.header.length; i++) {
            header += this.renderer.tablecell(this.inline.output(this.token.header[i]), {
              header: true,
              align: this.token.align[i]
            });
          }
          header = this.renderer.tablerow(header);
          for (i = 0; i < this.token.cells.length; i++) {
            row = "";
            for (var j = 0; j < this.token.cells[i].length; j++) {
              row += this.renderer.tablecell(this.inline.output(this.token.cells[i][j]), {
                header: false,
                align: this.token.align[j]
              });
            }
            body += this.renderer.tablerow(row);
          }
          return this.renderer.table(header, body);
        case "blockquote_start":
          var body = "";
          while (this.next().type !== "blockquote_end") {
            body += this.tok();
          }
          return this.renderer.blockquote(body);
        case "list_start":
          var body = "",
            ordered = this.token.ordered;
          while (this.next().type !== "list_end") {
            body += this.tok();
          }
          return this.renderer.list(body, ordered);
        case "list_item_start":
          var body = "";
          while (this.next().type !== "list_item_end") {
            body += this.token.type === "text" ? this.parseText() : this.tok();
          }
          return this.renderer.listitem(body);
        case "loose_item_start":
          var body = "";
          while (this.next().type !== "list_item_end") {
            body += this.tok();
          }
          return this.renderer.listitem(body);
        case "html":
          var html = this.token.pre || this.options.pedantic ? this.token.text : this.inline.output(this.token.text);
          return this.renderer.html(html);
        case "paragraph":
          return this.renderer.paragraph(this.inline.output(this.token.text));
        case "text":
          return this.renderer.paragraph(this.parseText());
      }
    };
  
    /**
     * Parse an array of tokens and produce HTML.
     *
     * @param {Array} tokens - Array of tokens.
     * @returns {string} - Final HTML output.
     */
    Parser.prototype.parse = function (tokens) {
      this.inline = new InlineLexer(tokens.links, this.options, this.renderer);
      this.tokens = tokens.reverse();
      var out = "";
      while (this.next()) {
        out += this.tok();
      }
      return out;
    };
  
    // Expose Parser, Renderer, Lexer, and InlineLexer.
    marked.Parser = Parser;
    marked.parser = Parser.parse;
    marked.Renderer = Renderer;
    marked.Lexer = Lexer;
    marked.lexer = Lexer.lex;
    marked.InlineLexer = InlineLexer;
    marked.inlineLexer = InlineLexer.output;
    marked.parse = marked;
  
    // Export marked in various module systems.
    if (typeof module !== "undefined" && typeof exports === "object") {
      module.exports = marked;
    } else if (typeof define === "function" && define.amd) {
      define(function () {
        return marked;
      });
    } else {
      this.marked = marked;
    }
  }).call(function () {
    return this || (typeof window !== "undefined" ? window : global);
  });
  
  //# sourceMappingURL=./marked.min.js.map