/*
 * Chat Widget JavaScript for Bytesweavers AI Chat Master plugin.
 * Handles user interactions in the front-end chat widget.
 */

document.addEventListener('DOMContentLoaded', function() {
    const widget = {
        init: function() {
            this.isProcessing = false;
            this.sessionId = this.generateSessionId();
            this.bindElements();
            this.bindEvents();
        },
        // Generate a unique session ID.
        generateSessionId: function() {
            return 'aicw_' + Date.now();
        },
        // Cache essential DOM elements.
        bindElements: function() {
            this.container = document.querySelector('.aicw-widget');
            this.toggle = document.querySelector('.aicw-toggle');
            this.messages = document.querySelector('.aicw-messages');
            this.form = document.querySelector('.aicw-input-form');
            this.input = document.querySelector('.aicw-input');
            this.typing = document.querySelector('.aicw-typing');
            this.closeBtn = document.querySelector('.aicw-close');

            // Append any initial messages for display.
            if (window.aicwInitialMessages && this.messages) {
                window.aicwInitialMessages.forEach(msg => {
                    const messageDiv = document.createElement('div');
                    messageDiv.classList.add('aicw-message', msg.type);
                    if (msg.isWelcome) {
                        messageDiv.setAttribute('data-is-welcome', 'true');
                        messageDiv.innerHTML = `
                            <div class="aicw-message-content">
                                <p>${msg.content}</p>
                            </div>
                            <div class="aicw-message-time">${msg.timestamp}</div>
                        `;
                    } else {
                        messageDiv.innerHTML = `
                            <div class="aicw-message-content">${msg.content}</div>
                            <div class="aicw-message-time">${msg.timestamp}</div>
                        `;
                    }
                    this.messages.appendChild(messageDiv);
                });
            }
        },
        // Bind event listeners.
        bindEvents: function() {
            if (this.toggle) {
                this.toggle.addEventListener('click', () => {
                    this.container.classList.add('active');
                });
            }
            if (this.form) {
                this.form.addEventListener('submit', (e) => this.handleSubmit(e));
            }
            if (this.closeBtn) {
                this.closeBtn.addEventListener('click', () => this.closeWidget());
            }
        },
        // Open or close the widget.
        toggleWidget: function() {
            if (!this.container) return;
            if (this.container.classList.contains('active')) {
                this.closeWidget();
            } else {
                this.container.classList.add('active');
                if (this.input) {
                    this.input.focus();
                }
            }
        },
        // Close the widget.
        closeWidget: function() {
            if (!this.container) return;
            this.container.classList.remove('active');
        },
        // Append a message to the chat.
        addMessage: function(content, type) {
            if (!this.messages || !content) return;
            
            const messageDiv = document.createElement('div');
            messageDiv.classList.add('aicw-message', type);

            // Format the content based on message type
            let formattedContent = content;
            if (type === 'assistant') {
                // Sanitize and format the content using Markdown if available
                formattedContent = this.formatContent(content);
            }

            messageDiv.innerHTML = `
                <div class="aicw-message-content">${formattedContent}</div>
                <div class="aicw-message-time">${this.getTime()}</div>
            `;

            this.messages.appendChild(messageDiv);
            this.scrollToBottom();
        },
        // Updated formatContent method to use a Markdown parser if available.
        formatContent: function(content) {
            if (typeof marked !== 'undefined') {
                return marked.parse(content);
            }
            // Fallback: Basic formatting below.
            let sanitized = content
                .replace(/&/g, '&amp;')
                .replace(/</g, '&lt;')
                .replace(/>/g, '&gt;');
            // Code blocks with language
            sanitized = sanitized.replace(/```(\w+)?\n([\s\S]+?)\n```/g, '<pre><code class="language-$1">$2</code></pre>');
            // Inline code
            sanitized = sanitized.replace(/`([^`]+)`/g, '<code>$1</code>');
            // Headers
            sanitized = sanitized.replace(/^### (.*$)/gm, '<h3>$1</h3>');
            sanitized = sanitized.replace(/^## (.*$)/gm, '<h2>$1</h2>');
            sanitized = sanitized.replace(/^# (.*$)/gm, '<h1>$1</h1>');
            // Bullet Lists: Convert lines starting with * into <li> and wrap consecutive ones in <ul>
            sanitized = sanitized.replace(/^\* (.*$)/gm, '<li>$1</li>');
            sanitized = sanitized.replace(/(<li>.*<\/li>)(\s*<li>.*<\/li>)+/g, function(match) {
                return '<ul>' + match + '</ul>';
            });
            // Numbered Lists
            sanitized = sanitized.replace(/^\d+\. (.*$)/gm, '<li>$1</li>');
            sanitized = sanitized.replace(/(<li>.*<\/li>)(\s*<li>.*<\/li>)+/g, function(match) {
                return '<ol>' + match + '</ol>';
            });
            // Bold and Italic
            sanitized = sanitized.replace(/\*\*([^*]+)\*\*/g, '<strong>$1</strong>');
            sanitized = sanitized.replace(/\*([^*]+)\*/g, '<em>$1</em>');
            sanitized = sanitized.replace(/_([^_]+)_/g, '<em>$1</em>');
            // Links
            sanitized = sanitized.replace(/\[([^\]]+)\]\(([^)]+)\)/g, '<a href="$2" target="_blank" rel="noopener noreferrer">$1</a>');
            // Convert double newlines to paragraph breaks and wrap content in paragraphs
            sanitized = sanitized.replace(/\n\n+/g, '</p><p>');
            sanitized = '<p>' + sanitized + '</p>';
            sanitized = sanitized.replace(/\n/g, '<br>');
            return sanitized;
        },
        // Send the message to the server.
        sendMessage: async function(message) {
            const formData = new FormData();
            formData.append('action', 'aicw_send_message');
            formData.append('nonce', aicwSettings.nonce);
            formData.append('message', message);
            formData.append('session_id', this.sessionId);

            const response = await fetch(aicwSettings.ajaxurl, {
                method: 'POST',
                body: formData
            });
            return await response.text().then(text => {
                try {
                    return JSON.parse(text);
                } catch (e) {
                    console.error('Invalid JSON response:', text);
                    return {
                        success: false,
                        response: 'Invalid response from server'
                    };
                }
            });
        },
        // Show the typing indicator.
        showTyping: function() {
            if (this.typing) {
                this.typing.style.display = 'flex';
                const textElement = this.typing.querySelector('.aicw-typing-text');
                if (textElement) {
                    textElement.style.display = 'inline';
                }
            }
            this.scrollToBottom();
        },
        // Hide the typing indicator.
        hideTyping: function() {
            if (this.typing) {
                this.typing.style.display = 'none';
            }
        },
        // Scroll chat to the bottom.
        scrollToBottom: function() {
            if (this.messages) {
                this.messages.scrollTop = this.messages.scrollHeight;
            }
        },
        // Get current time string.
        getTime: function() {
            return new Date().toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' });
        },
        // Enable or disable the submit button.
        toggleSubmitButton: function(disabled) {
            if (this.form) {
                const submitButton = this.form.querySelector('button[type="submit"]');
                if (submitButton) {
                    submitButton.disabled = disabled;
                    if (disabled) {
                        submitButton.classList.add('aicw-button-disabled');
                    } else {
                        submitButton.classList.remove('aicw-button-disabled');
                    }
                }
            }
        },
        // Handler for message submitting.
        handleSubmit: async function(e) {
            e.preventDefault();
            if (this.isProcessing) return;

            const message = this.input.value.trim();
            if (!message) return;

            this.toggleSubmitButton(true);
            this.isProcessing = true;

            try {
                this.addMessage(message, 'user');
                this.input.value = '';
                this.showTyping();

                const response = await this.sendMessage(message);
                this.hideTyping();

                if (response.success && response.response) {
                    this.addMessage(response.response, 'assistant');
                } else {
                    const errorMessage = response.error || "Sorry, something went wrong. Please try again.";
                    this.addMessage(errorMessage, 'assistant');
                }
            } catch (error) {
                console.error("Error in chat submission:", error);
                this.hideTyping();
                this.addMessage("Sorry, there was an error processing your request. Please try again.", 'assistant');
            } finally {
                this.isProcessing = false;
                this.toggleSubmitButton(false);
                this.scrollToBottom();
            }
        }
    };

    widget.init();
});