/*
 * Admin Settings JavaScript for Bytesweavers AI Chat Master plugin.
 * Handles settings panel interactions, API testing, color pickers,
 * and dynamic UI updates in the WordPress admin area.
 */

jQuery(document).ready(function($) {

    // Initialize the color picker.
    $('.aicw-color-picker').wpColorPicker();

    // Preload upgrade tab content.
    let upgradeTab = document.getElementById('upgrade');
    if (upgradeTab) {
        upgradeTab.style.display = 'none';
        upgradeTab.style.opacity = '1';
    }

    // Tab Navigation.
    $('.nav-tab-wrapper a').on('click', function(e) {
        e.preventDefault();

        // Cache DOM elements.
        const $this = $(this);
        const targetId = $this.attr('href').replace('#', '');
        const $targetContent = $('#' + targetId);

        // Update active tab classes.
        $('.nav-tab').removeClass('nav-tab-active');
        $this.addClass('nav-tab-active');

        // Hide all tabs and show the target tab.
        $('.tab-content').hide();
        // Use simple show() for Upgrade and About tabs to reduce animation load.
        if (targetId === 'upgrade' || targetId === 'about') {
            $targetContent.show();
        } else {
            $targetContent.fadeIn(200);
        }
    });

    // Trigger the first tab.
    $('.nav-tab-wrapper a:first').trigger('click');

    // Model Selection Handler.
    $('#aicw_selected_model').on('change', function() {
        const model = $(this).val();
        const modelFamily = getModelFamily(model);

        // Hide all API key fields.
        $('.api-key-field').hide();

        // Show the relevant API key field.
        switch(modelFamily) {
            case 'openai':
                $('#aicw_openai_key_field').show();
                break;
            case 'anthropic':
                $('#aicw_anthropic_key_field').show();
                break;
            case 'gemini':
                $('#aicw_gemini_key_field').show();
                break;
        }

        // Update model context information.
        updateModelInfo(model);
    });

    // Trigger model selection change on page load.
    $('#aicw_selected_model').trigger('change');

    // Temperature Slider.
    const temperatureSlider = $('input[name="aicw_temperature"]');
    const temperatureValue = $('.temperature-value');

    temperatureSlider.on('input', function() {
        temperatureValue.text($(this).val());
    });
    temperatureValue.text(temperatureSlider.val());

    // Enable/disable widget checkbox handler.
    $('input[name="aicw_enable_widget"]').on('change', function() {
        if ($(this).is(':checked')) {
            const model = $('#aicw_selected_model').val();
            const modelFamily = getModelFamily(model);
            let apiKey = '';

            switch(modelFamily) {
                case 'openai':
                    apiKey = $('#aicw_openai_key').val();
                    break;
                case 'anthropic':
                    apiKey = $('#aicw_anthropic_key').val();
                    break;
                case 'gemini':
                    apiKey = $('#aicw_gemini_key').val();
                    break;
            }

            if (!apiKey) {
                alert("Please set up an API key in the API settings tab before enabling the AI Chat Widget.");
                $(this).prop('checked', false);
                // Switch to API tab.
                $('.nav-tab-wrapper a[href="#api"]').trigger('click');
                return false;
            }
        }
    });

    // Test API Connection.
    $('.test-api-connection').on('click', function(e) {
        e.preventDefault();
        const button = $(this);
        const originalText = button.text();
        const model = $('#aicw_selected_model').val();
        const modelFamily = getModelFamily(model);
        let apiKey = '';

        switch(modelFamily) {
            case 'openai':
                apiKey = $('#aicw_openai_key').val();
                break;
            case 'anthropic':
                apiKey = $('#aicw_anthropic_key').val();
                break;
            case 'gemini':
                apiKey = $('#aicw_gemini_key').val();
                break;
        }

        if (!apiKey) {
            alert("Please enter an API key first.");
            return;
        }

        button.text('Testing...').prop('disabled', true);

        $.ajax({
            url: aicwSettings.ajaxurl,
            type: 'POST',
            data: {
                action: 'aicw_test_connection',
                nonce: aicwSettings.nonce,
                model: model,
                api_key: apiKey
            },
            success: function(response) {
                if (response.success) {
                    alert("Connection successful! The API key is valid.");
                } else {
                    alert("Connection failed: " + response.data);
                }
            },
            error: function() {
                alert("Connection test failed. Please check your internet connection and try again.");
            },
            complete: function() {
                button.text(originalText).prop('disabled', false);
            }
        });
    });

    // Bind click event specifically for the "Upgrade to Pro" links.
    $(document).on('click', '.upgrade-pro-link', function(e) {
        e.preventDefault();
        // Switch to the Upgrade tab.
        $('.nav-tab-wrapper a[href="#upgrade"]').trigger('click');
        $('html, body').animate({
            scrollTop: $('#upgrade').offset().top - 50
        }, 500);
    });

    // Manual Scrape Handler.
    $(document).on('click', '#trigger-scrape', function(e) {
        e.preventDefault();
        const button = $(this);
        const originalText = button.text();

        button.text('Scraping...').prop('disabled', true);

        $.ajax({
            url: aicwSettings.ajaxurl,
            type: 'POST',
            data: {
                action: 'aicw_manual_scrape',
                nonce: aicwSettings.nonce
            },
            success: function(response) {
                if (response.success) {
                    $('#aicw_knowledge_base').val(response.data.summary);
                    alert('Website content scraped successfully!');
                } else {
                    alert('Error: ' + response.data);
                }
            },
            error: function() {
                alert('Network error occurred while scraping content.');
            },
            complete: function() {
                button.text(originalText).prop('disabled', false);
            }
        });
    });

    // Helper Functions.
    function getModelFamily(model) {
        if (model.indexOf('gpt') !== -1 || model.indexOf('chatgpt') !== -1) return 'openai';
        if (model.indexOf('claude') !== -1) return 'anthropic';
        if (model.indexOf('gemini') !== -1) return 'gemini';
        return null;
    }

    function updateModelInfo(model) {
        let contextWindow = '4K tokens';
        let features = [];

        if (model.indexOf('gpt-4o') !== -1 || model === 'chatgpt-4o-latest') {
            contextWindow = '128K tokens';
            features = ['Text input/output', 'Advanced reasoning', 'Ultra-fast processing'];
        } else if (model === 'gpt-4-turbo-preview') {
            contextWindow = '128K tokens';
            features = ['Text input/output', 'Advanced reasoning', 'Latest knowledge'];
        } else if (model === 'gpt-4') {
            contextWindow = '8K tokens';
            features = ['Text input/output', 'High accuracy'];
        } else if (model === 'gpt-3.5-turbo') {
            contextWindow = '16K tokens';
            features = ['Text input/output', 'Fast responses'];
        } else if (model.indexOf('claude-3-7') !== -1 || model.indexOf('claude-3-5') !== -1) {
            contextWindow = '200K tokens';
            features = ['Text and image input', 'High intelligence'];
        } else if (model.indexOf('claude-3') !== -1) {
            contextWindow = '200K tokens';
            features = ['Text and image input', 'High intelligence'];
        } else if (model.indexOf('gemini-2.0-pro') !== -1) {
            contextWindow = '2M tokens';
            features = ['Text and image input', 'Advanced reasoning', 'Strong coding capabilities'];
        } else if (model.indexOf('gemini-2.0') !== -1) {
            contextWindow = '1M tokens';
            features = ['Text and image input', 'Balanced performance'];
        } else if (model.indexOf('gemini-1.5') !== -1) {
            contextWindow = '1M tokens';
            features = ['Text and image input', 'Advanced reasoning'];
        }

        $('#model_context_info').html(`
            <strong>Context Window:</strong> ${contextWindow}<br>
            <strong>Features:</strong> ${features.join(', ')}<br>
        `);

        const maxTokensInput = $('input[name="aicw_max_tokens"]');
        const maxTokens = parseInt(contextWindow.replace(/[^0-9]/g, '')) * 1000;
        maxTokensInput.attr('max', maxTokens);
        $('.max-tokens-info').text(`Maximum allowed: ${maxTokens.toLocaleString()} tokens`);
    }

    const displayOption = $('#aicw_display_option');
    const specificPagesRow = $('#aicw_specific_pages_row');

    function toggleSpecificPagesRow() {
        if (displayOption.val() === 'some') {
            specificPagesRow.show();
        } else {
            specificPagesRow.hide();
        }
    }
    displayOption.on('change', toggleSpecificPagesRow);
    toggleSpecificPagesRow();
});