<?php
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly.

// Define product types in desired order.
$product_types = array( 'simple', 'external', 'grouped', 'variable' );

foreach ( $product_types as $product_type ) {
	// Loop through posts.
	foreach ( $posts_to_process as $post_data ) {
		// Check if post with same slug already exists.
		$existing_post = get_page_by_path( $post_data['slug'], OBJECT, $post_data['post_type'] );

		if ( 1 === $count ) {
			if ( ! empty( $post_data['custom_taxonomies'] ) ) {
				foreach ( $post_data['custom_taxonomies'] as $taxonomy_data ) {
					$taxonomy_val = sanitize_key( $taxonomy_data['taxonomy'] );

					// Check if the taxonomy exists, if not, create it.
					$taxonomy_exists = taxonomy_exists( $taxonomy_val );
					if ( ! $taxonomy_exists ) {
						// Display error message and exit.
						$import_messages[] = sprintf(
							/* translators: %s: name of the taxonomy */
							esc_html__(
								'Failed to import %s: Invalid taxonomy.',
								'bytes-import-export-lite'
							),
							esc_html( $taxonomy_val )
						);
					}

					// Loop through each term data and add it.
					if ( $taxonomy_exists ) {
						if ( ! empty( $taxonomy_data['terms'] ) ) {
							foreach ( $taxonomy_data['terms'] as $term_data ) {
								// Check if the term already exists.
								$existing_term = term_exists( sanitize_text_field( $term_data['name'] ), $taxonomy_val );
								if ( ! $existing_term ) {
									// Insert the term.
									$inserted_term = wp_insert_term(
										sanitize_text_field( $term_data['name'] ),
										$taxonomy_val,
										array(
											'slug'   => sanitize_title( $term_data['slug'] ),
											'parent' => 0, // Set parent ID if it's a child term.
										)
									);

									// Check if the term was inserted successfully.
									if ( ! is_wp_error( $inserted_term ) ) {
										// Recursively add child terms.
										if ( ! empty( $term_data['children'] ) ) {
											foreach ( $term_data['children'] as $child ) {
												// Check if the child term already exists.
												$existing_child = term_exists( sanitize_title( $child['slug'] ), $taxonomy_val );
												if ( ! $existing_child ) {
													// Insert the child term.
													$inserted_child = wp_insert_term(
														sanitize_text_field( $child['name'] ),
														$taxonomy_val,
														array(
															'parent' => $inserted_term['term_id'],
															'slug' => sanitize_title( $child['slug'] ),
														)
													);
												}
											}
										}
									}
								}
							}
						}
					}
				}
			}

			// Set product attributes global.
			if ( ! empty( $post_data['custom_attributes'] ) ) {
				// Loop through each attribute data.
				foreach ( $post_data['custom_attributes'] as $attribute_data ) {
					$attribute_name = $attribute_data['taxonomy']; // Attribute taxonomy name.

					// These are exported as labels, so convert the label to a name if possible first.
					$attribute_labels = wp_list_pluck( wc_get_attribute_taxonomies(), 'attribute_label', 'attribute_name' );
					$attribute_name   = array_search( $attribute_data['label'], $attribute_labels, true );

					if ( ! $attribute_name ) {
						$attribute_name = wc_sanitize_taxonomy_name( $attribute_data['label'] );
					}

					$attribute_id = wc_attribute_taxonomy_id_by_name( $attribute_name );

					if ( ! $attribute_id ) {
						$beautify_attr_name = ucfirst( str_replace( '-', ' ', $attribute_data['label'] ) );
						// If the attribute does not exist, create it.
						$attribute_id = wc_create_attribute(
							array(
								'name'         => $beautify_attr_name,
								'slug'         => $attribute_name,
								'type'         => 'select',
								'order_by'     => 'menu_order',
								'has_archives' => false,
							)
						);

						if ( is_wp_error( $attribute_id ) ) {
							$import_messages[] = esc_html( $attribute_id->get_error_message() );
						}

						// Register as taxonomy while importing.
						$taxonomy_name = wc_attribute_taxonomy_name( $attribute_name );
						register_taxonomy(
							$taxonomy_name,
							array( $post_data['post_type'] ),
							array(
								'labels'       => array(
									'name' => $attribute_data['label'],
								),
								'hierarchical' => true,
								'show_ui'      => false,
								'query_var'    => true,
								'rewrite'      => false,
							)
						);

						// Loop through each term data and add it to the attribute.
						if ( ! empty( $attribute_data['terms'] ) ) {
							foreach ( $attribute_data['terms'] as $term_data ) {
								$term_name = sanitize_text_field( $term_data['name'] ); // Term name.
								$term_slug = sanitize_title( $term_data['slug'] ); // Term slug.

								// Check if the term already exists.
								$existing_term = term_exists( $term_slug, $taxonomy_name );

								if ( ! $existing_term ) {
									// Insert the term.
									wp_insert_term(
										$term_name,
										$taxonomy_name,
										array(
											'slug' => $term_slug,
										)
									);
								}
							}
						}
					}
				}
			}
		}
		++$count;

		// Check product type.
		if ( ! ( $post_data['product_type'] === $product_type ) ) {
			continue;
		}

		if ( $existing_post ) {
			// Post with same slug already exists, skip it.
			$escaped_post_type = esc_html( ucfirst( $post_data['post_type'] ) );
			$escaped_title     = esc_html( $post_data['title'] );
			$import_messages[] = '<li>' . esc_html( "{$escaped_post_type} - {$escaped_title} already exists." ) . '</li>';
			continue;
		}

		// Download and update images in post content.
		$updated_content = ! empty( $post_data['content'] ) ? $post_data['content'] : ''; // Initialize updated content.

		// Find and replace image URLs in the post content.
		if ( ! empty( $updated_content ) ) {
			preg_match_all( '/<img[^>]+src="([^"]+)"[^>]*>/', $post_data['content'], $matches ); // Find image tags.
			foreach ( $matches[1] as $img_url ) {
				// Check if image already exists.
				$attachment_id = attachment_url_to_postid( $img_url );
				if ( $attachment_id ) {
					// Image already exists, use existing attachment URL.
					$updated_content = str_replace( $img_url, wp_get_attachment_url( $attachment_id ), $updated_content );
					continue;
				}

				// Download image.
                $image_data = file_get_contents($img_url); // phpcs:ignore
				$filename   = basename( $img_url ); // Get image filename.
				$upload_dir = wp_upload_dir(); // Get upload directory.
				$file_path  = $upload_dir['path'] . '/' . $filename; // Set file path.
				// Save image file.
                file_put_contents($file_path, $image_data); // phpcs:ignore

				// Check file type.
				$wp_filetype = wp_check_filetype( $filename, null ); // Check file type.
				$attachment  = array(
					'post_mime_type' => $wp_filetype['type'],
					'post_title'     => sanitize_file_name( $filename ),
					'post_content'   => '',
					'post_status'    => 'inherit',
				);

				// Insert image as attachment.
				$attachment_id = wp_insert_attachment( $attachment, $file_path, 0 );
				if ( ! is_wp_error( $attachment_id ) ) {
					// Generate attachment metadata.
					require_once ABSPATH . 'wp-admin/includes/image.php';
					$attachment_data = wp_generate_attachment_metadata( $attachment_id, $file_path );
					wp_update_attachment_metadata( $attachment_id, $attachment_data );

					// Replace image URL in content with new attachment URL.
					$updated_content = str_replace( $img_url, wp_get_attachment_url( $attachment_id ), $updated_content );
				} else {
					// Image not imported.
					$import_messages[] = esc_html( 'Image not imported for post ' . $post_data['title'] );
				}
			}
		}

		// Get post data.
		$post_title   = ( ! empty( $post_data['title'] ) ) ? sanitize_text_field( $post_data['title'] ) : 'No title';
		$post_excerpt = ( ! empty( $post_data['excerpt'] ) ) ? sanitize_text_field( $post_data['excerpt'] ) : '';
		$post_date    = ( ! empty( $post_data['post_date'] ) ) ? $post_data['post_date'] : '';

		// map the post author.
		$author = isset( $post_data['post_author'] ) ? sanitize_user( $post_data['post_author'], true ) : '';
		if ( isset( $author_mapping[ $author ] ) ) {
			$author = $author_mapping[ $author ];
		} else {
			$author = (int) get_current_user_id();
		}

		// Insert post.
		$post_args = array(
			'post_title'   => $post_title,
			'post_name'    => ! empty( $post_data['slug'] ) ? urldecode( sanitize_title ( $post_data['slug'] ) ) : '', // Add post slug.
			'post_excerpt' => $post_excerpt, // Add post excerpt.
			'post_content' => $updated_content,
			'post_type'    => ! empty( $post_data['post_type'] ) ? sanitize_text_field( $post_data['post_type'] ) : '',
			'post_status'  => ! empty( $post_data['post_status'] ) ? sanitize_text_field( $post_data['post_status'] ) : '',
			'post_date'    => $post_date,
			'post_author'  => $author,
		);

		// Insert the post.
		$custom_post_id = wp_insert_post( $post_args );

		// Remove "Uncategorized" category from the inserted post.
		wp_remove_object_terms( $custom_post_id, 'uncategorized', 'category' );

		// Handle post meta data import.
		if ( ! empty( $post_data['post_meta'] ) && is_array( $post_data['post_meta'] ) ) {
			foreach ( $post_data['post_meta'] as $meta_key => $meta_value ) {
				update_post_meta( $custom_post_id, sanitize_key( $meta_key ), $meta_value );
			}
		}

		// Assign term values to the post.
		if ( ! empty( $post_data['selected_post_terms'] ) ) {
			foreach ( $post_data['selected_post_terms'] as $taxonomy_val => $terms ) {
				foreach ( $terms as $term_name ) {
					// Sanitize term name.
					$term_name = sanitize_text_field( $term_name );

					// Check if term exists, if not, insert it.
					$existing_term = term_exists( $term_name, $taxonomy_val );
					if ( ! $existing_term ) {
						wp_insert_term( $term_name, $taxonomy_val );
					}
				}
				// Assign terms to the post.
				wp_set_object_terms( $custom_post_id, $terms, $taxonomy_val, true );
			}
		}

		// Set featured image.
		if ( ! empty( $post_data['featured_image'] ) ) {
			$image_url  = esc_url( $post_data['featured_image'] );
			$upload_dir = wp_upload_dir();
			$filename   = basename( $image_url );

			// Check if the image exists.
			// Get the attachment object by title.
			global $wpdb;
            $image_exists = $wpdb->get_row($wpdb->prepare("SELECT * FROM $wpdb->posts WHERE post_title = %s AND post_type = 'attachment'", $filename)); // phpcs:ignore

			if ( ! $image_exists ) {
				// Image doesn't exist, download and insert it.
                $image_data = file_get_contents($image_url); // phpcs:ignore
				if ( wp_mkdir_p( $upload_dir['path'] ) ) {
					$file = $upload_dir['path'] . '/' . $filename;
				} else {
					$file = $upload_dir['basedir'] . '/' . $filename;
				}
                file_put_contents($file, $image_data); // phpcs:ignore
				$wp_filetype   = wp_check_filetype( $filename, null );
				$attachment    = array(
					'post_mime_type' => $wp_filetype['type'],
					'post_title'     => sanitize_file_name( $filename ),
					'post_content'   => '',
					'post_status'    => 'inherit',
				);
				$attachment_id = wp_insert_attachment( $attachment, $file, $custom_post_id );
				if ( ! is_wp_error( $attachment_id ) ) {
					require_once ABSPATH . 'wp-admin/includes/image.php';
					$attachment_data = wp_generate_attachment_metadata( $attachment_id, $file );
					wp_update_attachment_metadata( $attachment_id, $attachment_data );
					set_post_thumbnail( $custom_post_id, $attachment_id );
				} else {
					// Error inserting featured image.
					$import_messages[] = esc_html( 'Error: inserting featured image for post ' . $post_data['title'] );
				}
			} else {
				// Image already exists, set it as the featured image.
				set_post_thumbnail( $custom_post_id, $image_exists->ID );
			}
		}

		// Set product gallery images for WooCommerce products.
		if ( 'product' === $post_data['post_type'] && class_exists( 'WC_Product' ) ) {
			$product_id = absint( $custom_post_id ); // Validate and sanitize product ID.

			// Check if product exists.
			$product = wc_get_product( $product_id );
			if ( ! $product ) {
				$import_messages[] = esc_html__( 'Error: Product not found with ID ', 'bytes-import-export-lite' ) . esc_html( $product_id );
				return;
			}

			// Array to store attachment IDs for new gallery images.
			$new_gallery_attachment_ids = array();

			// Process each product gallery image data.
			if ( ! empty( $post_data['product_gallery_images'] ) && is_array( $post_data['product_gallery_images'] ) ) {
				foreach ( $post_data['product_gallery_images'] as $image_data ) {
					$image_url = isset( $image_data['url'] ) ? esc_url_raw( $image_data['url'] ) : null;
					$alt       = isset( $image_data['alt'] ) ? sanitize_text_field( $image_data['alt'] ) : null;

					if ( ! $image_url ) {
						$import_messages[] = esc_html__( 'Missing image URL for product gallery image ', 'bytes-import-export-lite' ) . esc_html( $product_id );
						continue;
					}

					// Download image content.
                    $image_data = file_get_contents($image_url); // phpcs:ignore

					// Generate unique filename and create upload directory if needed.
					$filename = wp_basename( $image_url );
					$file     = wp_upload_bits( $filename, null, $image_data );
					// Check if the image exists.
					// Get the attachment object by title.
					global $wpdb;
                    $image_exists = $wpdb->get_row($wpdb->prepare("SELECT * FROM $wpdb->posts WHERE post_title = %s AND post_type = 'attachment'", $filename)); // phpcs:ignore

					// Check if file was successfully uploaded.
					if ( $image_exists && ! empty( $image_exists ) ) {
						$new_gallery_attachment_ids[] = $image_exists->ID;
					} elseif ( ! $file['error'] ) {
						// Set up attachment data.
						$attachment_data = array(
							'post_mime_type' => $file['type'],
							'post_title'     => sanitize_file_name( $filename ),
							'post_content'   => '',
							'post_status'    => 'inherit',
						);

						// Insert attachment.
						$attachment_id = wp_insert_attachment( $attachment_data, $file['file'], $product_id );

						// Handle attachment errors.
						if ( ! is_wp_error( $attachment_id ) ) {
							// Set alt text.
							update_post_meta( $attachment_id, '_wp_attachment_image_alt', $alt );

							$new_gallery_attachment_ids[] = $attachment_id;
						} else {
							$import_messages[] = esc_html__( 'Error inserting attachment: ', 'bytes-import-export-lite' ) . esc_html( $attachment_id->get_error_message() );
						}
					} else {
						$import_messages[] = esc_html__( 'Error uploading image: ', 'bytes-import-export-lite' ) . esc_html( $file['error'] );
					}
				}

				// For non-variable product types, set gallery images.
				if ( ! $product->is_type( 'variable' ) && ! empty( $new_gallery_attachment_ids ) ) {
					// Set new gallery image IDs.
					$product->set_gallery_image_ids( $new_gallery_attachment_ids );
					// Save product.
					$product->save();
				}
			}

			// Set downloadable files data.
			if ( isset( $post_data['downloadable_files'] ) && ( ! empty( $post_data['downloadable_files'] ) ) ) {
				if ( $product && $product->is_downloadable() ) {
					$downloadable_files = $post_data['downloadable_files'];
					$downloadable_data  = array();
					foreach ( $downloadable_files as $file_data ) {
						$file_name = isset( $file_data['name'] ) ? sanitize_text_field( $file_data['name'] ) : '';
						$file_path = isset( $file_data['file'] ) ? esc_url_raw( $file_data['file'] ) : '';
						if ( ! empty( $file_name ) && ! empty( $file_path ) ) {
							$downloadable_data[] = array(
								'name' => sanitize_text_field( $file_name ),
								'file' => sanitize_text_field( $file_path ),
							);
						}
					}
					// Set downloadable files data for the product.
					$product->set_downloads( $downloadable_data );
					$product->save();
				}
			}

			// Set grouped product and child product data.
			if ( ! empty( $post_data['grouped_data'] ) ) {
				$grouped_data         = wp_unslash( $post_data['grouped_data'] ); // Sanitize grouped product data.
				$grouped_product_name = isset( $grouped_data['grouped_product_name'] ) ? sanitize_text_field( $grouped_data['grouped_product_name'] ) : '';
				$children_data        = isset( $grouped_data['children'] ) ? $grouped_data['children'] : array();

				// Check if grouped product exists.
				if ( ! $product ) {
					// Grouped product does not exist, create new grouped product.
					$product = new WC_Product_Grouped();
					$product->set_name( $grouped_product_name );
					// Save the grouped product.
					$product_id = $product->save();
				} else {
					// Grouped product already exists, update its data.
					$product->set_name( $grouped_product_name );
					// Save the grouped product.
					$product->save();
				}

				// Loop through each child product data.
				$grouped_product_children = array();
				foreach ( $children_data as $child_data ) {
					// Extract child product data.
					$child_product_id   = isset( $child_data['product_id'] ) ? absint( $child_data['product_id'] ) : 0;
					$child_product_name = isset( $child_data['name'] ) ? sanitize_text_field( $child_data['name'] ) : '';
					$child_product_sku  = isset( $child_data['sku'] ) ? sanitize_text_field( $child_data['sku'] ) : '';

					// add child product id in array.
					$grouped_product_children[] = $child_product_sku;
				}

				// Add child product to grouped product.
				if ( $product_id ) {
					$grouped_product = new WC_Product_Grouped( $product_id );
					if ( ! empty( $grouped_product_children ) ) {
						$linked_product_id = array();
						foreach ( $grouped_product_children as $child_product_sku ) {
							global $wpdb;
                            $linked_product = $wpdb->get_var( $wpdb->prepare( "SELECT post_id FROM {$wpdb->postmeta} WHERE meta_key='_sku' AND meta_value='%s'", $child_product_sku ) ); // phpcs:ignore

							if ( $linked_product ) {
								$linked_product_id[] = $linked_product;
							} else {
								$import_messages[] = esc_html__( 'Error: Linked product with SKU ', 'bytes-import-export-lite' ) . esc_html( $child_product_sku ) . esc_html__( ' not found.', 'bytes-import-export-lite' );
							}
						}

						// Set properties for grouped product.
						$grouped_product->set_props(
							array(
								'type'               => 'grouped',
								'status'             => 'publish',
								'catalog_visibility' => 'visible',
								'grouped_products'   => ( ! empty( $linked_product_id ) ) ? $grouped_product->set_children( $linked_product_id ) : '',
							)
						);

						$grouped_product->save(); // Save grouped product.
					}
				}
			}

			// Extract variable product data.
			if ( ! empty( $post_data['variable_data'] ) ) {
				$product_data = array();
				$product_data = wp_unslash( $post_data['variable_data'] ); // Sanitize variable product data.
				$product      = new WC_Product_Variable( $product_id );

				// Set product name.
				$product->set_name( sanitize_text_field( $product_data['name'] ) );

				// Get all existing attribute taxonomies in WooCommerce.
				$existing_attribute_taxonomies = wc_get_attribute_taxonomies( array( 'global' => true ) ); // Include global attributes.

				// Iterate through attributes and add them to the product.
				$attr_pos = 0;
				$atts     = array();
				if ( ! empty( $product_data['attributes'] ) ) {
					foreach ( $product_data['attributes'] as $attribute_name => $attribute_data ) {
						$attribute_taxonomy_name = sanitize_title( $attribute_name );
						// Check if attribute name contains "pa_" prefix.
						if ( strpos( $attribute_name, 'pa_' ) === 0 ) {
							// Extract attribute name from taxonomy name and capitalize it.
							$attribute_name = str_replace( 'pa_', '', $attribute_name );
						}

						// Check if the attribute name exists in the list of existing attribute taxonomies.
						$attribute_exists = false;
						foreach ( $existing_attribute_taxonomies as $taxonomy_val ) {
							if ( $taxonomy_val->attribute_name === $attribute_name ) {
								$attribute_exists = true;
								break;
							}
						}

						if ( $attribute_exists ) {
							$attribute = new WC_Product_Attribute( $attribute_name );
							$attribute->set_id( wc_attribute_taxonomy_id_by_name( ucfirst( $attribute_name ) ) ); // Use existing ID.
						} else {
							// Create a new attribute if it doesn't exist.
							$attribute               = new WC_Product_Attribute();
							$attribute_taxonomy_name = ucfirst( $attribute_name );
							$attribute->set_id( 0 );
						}

						// Creating Attributes.
						$attribute->set_name( $attribute_taxonomy_name ); // Set attribute name.
						$options = array();
						foreach ( $attribute_data['terms'] as $term_val ) {
							$options[] = $term_val['name'];
						}
						$attribute->set_options( $options );
						$attribute->set_visible( true );
						$attribute->set_variation( true );
						++$attr_pos;

						$atts[] = $attribute;
					}
				}
				// //Adding attributes to the created product.
				$product->set_attributes( $atts );

				// save the changes and go on.
				$product->save();

				// Iterate through variations and add them to the product.
				if ( ! empty( $product_data['variations'] ) ) {
					foreach ( $product_data['variations'] as $variation_data ) {
						// Remove "pa_" prefix from attribute keys.
						$cleaned_attributes = array();
						if ( ! empty( $variation_data['attributes'] ) ) {
							foreach ( $variation_data['attributes'] as $attribute_key => $attribute_value ) {
								if ( strpos( $attribute_key, 'pa_' ) === 0 ) {
									$cleaned_attributes[ $attribute_key ] = strtolower( $attribute_value );
								} else {
									$cleaned_attributes[ $attribute_key ] = ucfirst( $attribute_value );
								}
							}
						}
						// Create variation and set attributes.
						$variation = new WC_Product_Variation();
						$variation->set_parent_id( $product->get_id() );
						$variation->set_attributes( $cleaned_attributes );
						$variation->set_status( 'publish' );
						$variation->set_sku( sanitize_text_field( $variation_data['sku'] ) );
						$variation->set_price( floatval( $variation_data['price'] ) );
						$variation->set_regular_price( floatval( $variation_data['price'] ) );
						$variation->set_stock_status();
						$variation->save();
						$product = wc_get_product( $product->get_id() );
						$product->save();
					}
				}
			}

			// Import product attribute data for simple, external and grouped product.
			$product_attribute_data = array();
			if ( ! empty( $post_data['simple_attribute_data'] ) ) {
				$product_attribute_data = $post_data['simple_attribute_data'];
				$product                = new WC_Product_Simple( $product_id );
			} elseif ( ! empty( $post_data['grouped_attribute_data'] ) ) {
				$product_attribute_data = $post_data['grouped_attribute_data'];
				$product                = new WC_Product_Grouped( $product_id );
			} elseif ( ! empty( $post_data['external_attribute_data'] ) ) {
				$product_attribute_data = $post_data['external_attribute_data'];
				$product                = new WC_Product_External( $product_id );
			}

			// Process attributes.
			if ( ( ! empty( $post_data['simple_attribute_data'] ) ) || ( ! empty( $post_data['grouped_attribute_data'] ) ) || ( ! empty( $post_data['external_attribute_data'] ) ) ) {
				if ( ! empty( $product_attribute_data['attributes'] ) ) {
					$existing_attribute_taxonomies = wc_get_attribute_taxonomies( array( 'global' => true ) ); // Include global attributes.
					$atts                          = array();
					foreach ( $product_attribute_data['attributes'] as $attribute_name => $attribute_data ) {
						$attribute_taxonomy_name = sanitize_key( $attribute_name );
						// Check if attribute name contains "pa_" prefix.
						if ( strpos( $attribute_name, 'pa_' ) === 0 ) {
							// Extract attribute name from taxonomy name and capitalize it.
							$attribute_name = str_replace( 'pa_', '', $attribute_name );
						}

						// Check if the attribute name exists in the list of existing attribute taxonomies.
						$attribute_exists = false;
						foreach ( $existing_attribute_taxonomies as $taxonomy_val ) {
							if ( $taxonomy_val->attribute_name === $attribute_name ) {
								$attribute_exists = true;
								break;
							}
						}

						if ( $attribute_exists ) {
							$attribute = new WC_Product_Attribute( $attribute_name );
							$attribute->set_id( wc_attribute_taxonomy_id_by_name( ucfirst( $attribute_name ) ) ); // Use existing ID.
						} else {
							// Create a new attribute if it doesn't exist.
							$attribute               = new WC_Product_Attribute();
							$attribute_taxonomy_name = ucfirst( $attribute_name );
							$attribute->set_id( 0 );
						}

						// Creating Attributes.
						$attribute->set_name( $attribute_taxonomy_name ); // Set attribute name.
						$options = array();
						foreach ( $attribute_data['terms'] as $term_val ) {
							$options[] = $term_val['name'];
						}
						$attribute->set_options( $options );
						$attribute->set_visible( true );

						$atts[] = $attribute;
					}
					// Adding attributes to the created product.
					$product->set_attributes( $atts );

					// Save product.
					$product->save();
				}
			}
		}
		$import_messages[] = '<li>' . esc_html( ucfirst( $post_data['post_type'] ) . ' - ' . $post_data['title'] . ' imported.' ) . '</li>';
	}
}

if ( isset( $_SESSION['import_post_count'] ) && $total_posts <= $_SESSION['import_post_count'] ) {
	$import_messages[] = esc_html__( 'All done. ', 'bytes-import-export-lite' ) . '<a href="' . esc_url( admin_url() ) . '">' . esc_html__( 'Have fun!', 'bytes-import-export-lite' );
	$import_messages[] = esc_html__( 'Remember to update the passwords and roles of imported users.', 'bytes-import-export-lite' );

	// Clear session variable after import is complete.
	unset( $_SESSION['import_post_count'] );
	unset( $_SESSION['author_map_arr'] );
}
