<?php
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly.

// Loop through posts.
foreach ( $posts_to_process as $post_data ) {
	// Check if post with same slug already exists.
	$existing_post = get_page_by_path( $post_data['slug'], OBJECT, $post_data['post_type'] );

	if ( 1 === $count ) {
		if ( ! empty( $post_data['custom_taxonomies'] ) ) {
			foreach ( $post_data['custom_taxonomies'] as $taxonomy_data ) {
				$taxonomy_val = sanitize_key( $taxonomy_data['taxonomy'] );

				// Check if the taxonomy exists, if not, create it.
				$taxonomy_exists = taxonomy_exists( $taxonomy_val );
				if ( ! $taxonomy_exists ) {
					// Display error message and exit.
					$import_messages[] = sprintf(
						/* translators: %s: name of the taxonomy */
						esc_html__(
							'Failed to import %s: Invalid taxonomy.',
							'bytes-import-export-lite'
						),
						esc_html( $taxonomy_val )
					);
				}

				// Loop through each term data and add it.
				if ( $taxonomy_exists ) {
					if ( ! empty( $taxonomy_data['terms'] ) ) {
						foreach ( $taxonomy_data['terms'] as $term_data ) {
							// Check if the term already exists.
							$existing_term = term_exists( sanitize_text_field( $term_data['name'] ), $taxonomy_val );
							if ( ! $existing_term ) {
								// Insert the term.
								$inserted_term = wp_insert_term(
									sanitize_text_field( $term_data['name'] ),
									$taxonomy_val,
									array(
										'slug'   => sanitize_title( $term_data['slug'] ),
										'parent' => 0, // Set parent ID if it's a child term.
									)
								);

								// Check if the term was inserted successfully.
								if ( ! is_wp_error( $inserted_term ) ) {
									// Recursively add child terms.
									if ( ! empty( $term_data['children'] ) ) {
										foreach ( $term_data['children'] as $child ) {
											// Check if the child term already exists.
											$existing_child = term_exists( sanitize_title( $child['slug'] ), $taxonomy_val );
											if ( ! $existing_child ) {
												// Insert the child term.
												$inserted_child = wp_insert_term(
													sanitize_text_field( $child['name'] ),
													$taxonomy_val,
													array(
														'parent' => $inserted_term['term_id'],
														'slug' => sanitize_title( $child['slug'] ),
													)
												);
											}
										}
									}
								}
							}
						}
					}
				}
			}
		}
	}

	if ( $existing_post ) {
		// Post with same slug already exists, skip it.
		$import_messages[] = '<li>' . esc_html( ucfirst( $post_data['post_type'] ) . ' - ' . $post_data['title'] . ' already exists.' ) . '</li>';
		continue;
	}

	// Download and update images in post content.
	$updated_content = ! empty( $post_data['content'] ) ? $post_data['content'] : ''; // Initialize updated content.

	// Find and replace image URLs in the post content.
	if ( ! empty( $updated_content ) ) {
		preg_match_all( '/<img[^>]+src="([^"]+)"[^>]*>/', $post_data['content'], $matches ); // Find image tags.
		foreach ( $matches[1] as $img_url ) {
			// Check if image already exists.
			$attachment_id = attachment_url_to_postid( $img_url );
			if ( $attachment_id ) {
				// Image already exists, use existing attachment URL.
				$updated_content = str_replace( $img_url, wp_get_attachment_url( $attachment_id ), $updated_content );
				continue;
			}

			// Download image.
            $image_data = file_get_contents($img_url); // phpcs:ignore
			$filename   = basename( $img_url ); // Get image filename.
			$upload_dir = wp_upload_dir(); // Get upload directory.
			$file_path  = $upload_dir['path'] . '/' . $filename; // Set file path.
            file_put_contents($file_path, $image_data); // phpcs:ignore

			// Check file type.
			$wp_filetype = wp_check_filetype( $filename, null ); // Check file type.
			$attachment  = array(
				'post_mime_type' => $wp_filetype['type'],
				'post_title'     => sanitize_file_name( $filename ),
				'post_content'   => '',
				'post_status'    => 'inherit',
			);

			// Insert image as attachment.
			$attachment_id = wp_insert_attachment( $attachment, $file_path, 0 );
			if ( ! is_wp_error( $attachment_id ) ) {
				// Generate attachment metadata.
				require_once ABSPATH . 'wp-admin/includes/image.php';
				$attachment_data = wp_generate_attachment_metadata( $attachment_id, $file_path );
				wp_update_attachment_metadata( $attachment_id, $attachment_data );

				// Replace image URL in content with new attachment URL.
				$updated_content = str_replace( $img_url, wp_get_attachment_url( $attachment_id ), $updated_content );
			} else {
				// Image not imported.
				$import_messages[] = '<li>' . esc_html( 'Image not imported for post' ) . ' - ' . $post_data['title'] . '</li>';
			}
		}
	}

	// Get post data.
	$post_title   = ( ! empty( $post_data['title'] ) ) ? sanitize_text_field( $post_data['title'] ) : 'No title';
	$post_excerpt = ( ! empty( $post_data['excerpt'] ) ) ? sanitize_text_field( $post_data['excerpt'] ) : '';
	$post_date    = ( ! empty( $post_data['post_date'] ) ) ? $post_data['post_date'] : '';

	// map the post author.
	$author = isset( $post_data['post_author'] ) ? sanitize_user( $post_data['post_author'], true ) : '';
	if ( isset( $author_mapping[ $author ] ) ) {
		$author = $author_mapping[ $author ];
	} else {
		$author = (int) get_current_user_id();
	}

	// Insert post.
	$post_args = array(
		'post_title'   => $post_title,
		'post_name'    => ! empty( $post_data['slug'] ) ? urldecode( sanitize_title ( $post_data['slug'] ) ) : '', // Add post slug.
		'post_excerpt' => $post_excerpt, // Add post excerpt.
		'post_content' => $updated_content,
		'post_type'    => ! empty( $post_data['post_type'] ) ? sanitize_text_field( $post_data['post_type'] ) : '',
		'post_status'  => ! empty( $post_data['post_status'] ) ? sanitize_text_field( $post_data['post_status'] ) : '',
		'post_date'    => $post_date,
		'post_author'  => $author,
	);

	// Insert the post.
	$custom_post_id = wp_insert_post( $post_args ); // get inserted post id.

	// Remove "Uncategorized" category from the inserted post.
	wp_remove_object_terms( $custom_post_id, 'uncategorized', 'category' );

	// Handle post meta data import.
	if ( ! empty( $post_data['post_meta'] ) && is_array( $post_data['post_meta'] ) ) {
		foreach ( $post_data['post_meta'] as $meta_key => $meta_value ) {
			update_post_meta( $custom_post_id, sanitize_key( $meta_key ), $meta_value );
		}
	}

	// Assign term values to the post.
	if ( ! empty( $post_data['selected_post_terms'] ) ) {
		foreach ( $post_data['selected_post_terms'] as $taxonomy_val => $terms ) {
			foreach ( $terms as $term_name ) {
				// Sanitize term name.
				$term_name = sanitize_text_field( $term_name );

				// Check if term exists, if not, insert it.
				$existing_term = term_exists( $term_name, $taxonomy_val );
				if ( ! $existing_term ) {
					wp_insert_term( $term_name, $taxonomy_val );
				}
			}
			// Assign terms to the post.
			wp_set_object_terms( $custom_post_id, $terms, $taxonomy_val, true );
		}
	}

	// Set featured image.
	if ( ! empty( $post_data['featured_image'] ) ) {
		$image_url  = esc_url( $post_data['featured_image'] );
		$upload_dir = wp_upload_dir();
		$filename   = basename( $image_url );

		// Check if the image exists.
		// Get the attachment object by title.
		global $wpdb;
        $image_exists = $wpdb->get_row($wpdb->prepare("SELECT * FROM $wpdb->posts WHERE post_title = %s AND post_type = 'attachment'", $filename)); // phpcs:ignore

		if ( ! $image_exists ) {
			// Image doesn't exist, download and insert it.
            $image_data = file_get_contents($image_url); // phpcs:ignore
			if ( wp_mkdir_p( $upload_dir['path'] ) ) {
				$file = $upload_dir['path'] . '/' . $filename;
			} else {
				$file = $upload_dir['basedir'] . '/' . $filename;
			}
            file_put_contents($file, $image_data); // phpcs:ignore
			$wp_filetype   = wp_check_filetype( $filename, null );
			$attachment    = array(
				'post_mime_type' => $wp_filetype['type'],
				'post_title'     => sanitize_file_name( $filename ),
				'post_content'   => '',
				'post_status'    => 'inherit',
			);
			$attachment_id = wp_insert_attachment( $attachment, $file, $custom_post_id );
			if ( ! is_wp_error( $attachment_id ) ) {
				require_once ABSPATH . 'wp-admin/includes/image.php';
				$attachment_data = wp_generate_attachment_metadata( $attachment_id, $file );
				wp_update_attachment_metadata( $attachment_id, $attachment_data );
				set_post_thumbnail( $custom_post_id, $attachment_id );
			} else {
				// Error inserting featured image.
				$import_messages[] = '<li>' . esc_html( 'Error: inserting featured image for post' ) . ' - ' . $post_data['title'] . '</li>';
			}
		} else {
			// Image already exists, set it as the featured image.
			set_post_thumbnail( $custom_post_id, $image_exists->ID );
		}
	}
	$import_messages[] = '<li>' . esc_html( ucfirst( $post_data['post_type'] ) . ' - ' . $post_data['title'] . ' imported.' ) . '</li>';
}

if ( isset( $_SESSION['import_post_count'] ) && $total_posts <= $_SESSION['import_post_count'] ) {
	$import_messages[] = esc_html__( 'All done. ', 'bytes-import-export-lite' ) . '<a href="' . esc_url( admin_url() ) . '">' . esc_html__( 'Have fun!', 'bytes-import-export-lite' );
	$import_messages[] = esc_html__( 'Remember to update the passwords and roles of imported users.', 'bytes-import-export-lite' );

	// Clear session variable after import is complete.
	unset( $_SESSION['import_post_count'] );
	unset( $_SESSION['author_map_arr'] );
}
