<?php
// Export posts.
add_action( 'wp_ajax_biel_export_posts', 'biel_export_posts_file' );
add_action( 'wp_ajax_nopriv_biel_export_posts', 'biel_export_posts_file' );

if ( ! function_exists( 'biel_export_posts_file' ) ) {
	// Export file function.
	function biel_export_posts_file() {
		// Verify nonce, action.
		if ( isset( $_POST['action'] ) && 'biel_export_posts' === $_POST['action'] && isset( $_POST['security'] ) && wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['security'] ) ), 'biel_export_posts_nonce' ) && current_user_can( 'manage_options' ) ) {
			// get exported post status.
			$biel_bytes_import_export_posts_status = get_option( 'biel_bytes_import_export_posts_status' );

			// Get the selected post type from the AJAX request.
			$post_type  = isset( $_POST['post_type'] ) ? sanitize_text_field( wp_unslash( $_POST['post_type'] ) ) : '';
			$start_date = isset( $_POST['start_date'] ) ? sanitize_text_field( wp_unslash( $_POST['start_date'] ) ) : '';
			$end_date   = isset( $_POST['end_date'] ) ? sanitize_text_field( wp_unslash( $_POST['end_date'] ) ) : '';
			$author     = isset( $_POST['author'] ) ? sanitize_text_field( wp_unslash( $_POST['author'] ) ) : '';

			$author_data = array();

			if ( $author ) { // Single author selected.
				$author_info = get_user_by( 'id', $author );
				if ( $author_info ) {
						$author_data[ $author_info->user_login ] = array(
							'author_id'           => $author_info->ID,
							'author_login'        => $author_info->user_login,
							'author_email'        => $author_info->user_email,
							'author_display_name' => $author_info->display_name,
							'author_first_name'   => $author_info->first_name,
							'author_last_name'    => $author_info->last_name,
						);
				}
			} else { // All authors for the post type.
				global $wpdb;
				$query = "SELECT DISTINCT post_author FROM {$wpdb->posts} WHERE post_type = %s AND post_status = ";
				if ( '0' === $biel_bytes_import_export_posts_status ) {
					$query       .= '%d';
					$status_param = intval( $biel_bytes_import_export_posts_status );
				} else {
					$query       .= '%s';
					$status_param = $biel_bytes_import_export_posts_status;
				}
				$author_ids = $wpdb->get_col( $wpdb->prepare( $query, $post_type, $status_param ) ); // phpcs:ignore

				foreach ( $author_ids as $author_id ) {
					$author_info = get_userdata( $author_id );

					if ( $author_info ) {
						$author_data[ $author_info->user_login ] = array(
							'author_id'           => $author_info->ID,
							'author_login'        => $author_info->user_login,
							'author_email'        => $author_info->user_email,
							'author_display_name' => $author_info->display_name,
							'author_first_name'   => $author_info->first_name,
							'author_last_name'    => $author_info->last_name,
						);
					}
				}
			}

			// get exported fields.
			$biel_bytes_import_export_posts_fields = get_option( 'biel_bytes_import_export_posts_fields' );

			// get exported post status.
			$biel_bytes_import_export_posts_status = get_option( 'biel_bytes_import_export_posts_status' );

			// Set up query arguments based on the selected post type.
			$args = array(
				'post_type'      => $post_type,
				'posts_per_page' => -1, // Get all posts.
				'post_status'    => $biel_bytes_import_export_posts_status,
			);

			// Add date parameters.
			if ( $start_date && $end_date ) {
				$args['date_query'] = array(
					'after'     => $start_date,
					'before'    => $end_date,
					'inclusive' => true,
				);
			}

			// Add author parameter.
			if ( $author ) {
				$args['author'] = $author;
			}

			// if post type is product.
			if ( 'product' === $post_type ) {
                // phpcs:ignore
                $args['tax_query'] = array(
					array(
						'taxonomy' => 'product_type',
						'field'    => 'slug',
						'terms'    => array( 'simple', 'external', 'grouped', 'variable' ), // Specify product types in the desired order.
						'operator' => 'IN',
					),
				);
			}

			$count = 1; // Initialize count variable.

			// Query posts.
			$posts_query = new WP_Query( $args );

			// Check if there are any posts.
			if ( $posts_query->have_posts() ) {
				$export_data = array();

				// Loop through posts.
				while ( $posts_query->have_posts() ) {
					$posts_query->the_post();

					// Get post data.
					$post_id             = get_the_ID();
					$post_title          = ( '1' === $biel_bytes_import_export_posts_fields['title'] ) ? get_the_title() : '';
					$post_slug           = get_post_field( 'post_name' );
					$post_content        = ( '1' === $biel_bytes_import_export_posts_fields['content'] ) ? get_the_content() : '';
					$post_excerpt        = ( '1' === $biel_bytes_import_export_posts_fields['excerpt'] ) ? get_the_excerpt() : '';
					$post_status         = get_post_status();
					$post_author         = get_the_author_meta( 'user_login' );
					$post_date           = ( '1' === $biel_bytes_import_export_posts_fields['date'] ) ? get_the_date( 'Y-m-d H:i:s' ) : '';
					$post_featured_image = ( '1' === $biel_bytes_import_export_posts_fields['featured_image'] ) ? get_the_post_thumbnail_url() : '';

					// Initialize empty array.
					$post_data = array();

					// Retrieve postmeta data.
					$postmeta = ( '1' === $biel_bytes_import_export_posts_fields['post_meta_fields'] ) ? get_post_meta( $post_id ) : '';
					if ( ! empty( $postmeta ) ) {
						// Convert arrays to strings for each postmeta value.
						foreach ( $postmeta as $key => $value ) {
							if ( '_edit_lock' === $key ) {
								unset( $postmeta[ $key ] );
								continue;
							}
							if ( is_array( $value ) ) {
								$postmeta[ $key ] = implode( ', ', $value );
							}
						}
					}

					if ( 'product' === $post_type ) {
						$product_of_type = wc_get_product( $post_id );
						$product_type    = $product_of_type->get_type();
					} else {
						$product_type = '';
					}

					$post_data = array(
						'id'             => $post_id,
						'title'          => $post_title,
						'slug'           => $post_slug,
						'content'        => $post_content,
						'excerpt'        => $post_excerpt,
						'featured_image' => $post_featured_image,
						'post_meta'      => $postmeta,
						'post_date'      => $post_date,
						'post_type'      => $post_type,
						'product_type'   => $product_type,
						'post_status'    => $post_status,
						'post_author'    => $post_author,
					);

					if ( 1 === $count ) {
						// Add author data.
						$post_data['author_data'] = $author_data;

						// Get all taxonomies associated with the post type.
						$taxonomies = get_object_taxonomies( $post_type );

						// Initialize an array to store exported attribute data.
						$post_data['custom_taxonomies'] = array();

						if ( '1' === $biel_bytes_import_export_posts_fields['taxonomies'] ) {
							if ( ! empty( $taxonomies ) ) {
								// Loop through each taxonomy.
								foreach ( $taxonomies as $taxonomy ) {
									// Skip taxonomies with the prefix "pa_".
									if ( 0 === strpos( $taxonomy, 'pa_' ) ) {
										continue;
									}

									// Get all terms for the taxonomy.
									$terms = get_terms(
										array(
											'taxonomy'   => $taxonomy,
											'hide_empty' => false,
											'parent'     => 0, // Get top-level terms.
										)
									);

									// Initialize an array to store terms data for the taxonomy.
									$taxonomy_data = array(
										'taxonomy' => $taxonomy,
										'terms'    => array(),
									);

									// Loop through each top-level term and recursively add its children.
									if ( ! empty( $terms ) ) {
										foreach ( $terms as $term ) {
											// Get term children.
											$children    = get_term_children( $term->term_id, $taxonomy );
											$child_terms = array();
											if ( ! empty( $children ) ) {
												foreach ( $children as $child_id ) {
													$child_term = get_term_by( 'id', $child_id, $taxonomy );
													if ( $child_term ) {
														$child_terms[] = array(
															'name' => $child_term->name,
															'slug' => $child_term->slug,
														);
													}
												}
											}

											$term_data                = array(
												'name'     => $term->name,
												'slug'     => $term->slug,
												'children' => $child_terms, // Get children term slugs recursively.
											);
											$taxonomy_data['terms'][] = $term_data;
										}
									}

									// Add the taxonomy data to the exported attributes array.
									$post_data['custom_taxonomies'][ $taxonomy ] = $taxonomy_data;
								}
							}
						}

						// Get all product attributes.
						$post_data['custom_attributes'] = array();

						// If post type is product.
						if ( 'product' === $post_type && class_exists( 'WC_Product' ) ) {
							$attributes = wc_get_attribute_taxonomies();
							if ( ! empty( $attributes ) ) {
								// Loop through each attribute.
								foreach ( $attributes as $attribute ) {
									$attribute_name = 'pa_' . $attribute->attribute_name; // Attribute taxonomy name.
									$terms          = get_terms(
										array(
											'taxonomy'   => $attribute_name,
											'hide_empty' => false,
										)
									);

									// Initialize an array to store terms data for the attribute.
									$terms_data = array();
									if ( ! empty( $terms ) ) {
										foreach ( $terms as $term ) {
											// Add term data to the array.
											$terms_data[] = array(
												'name' => $term->name,
												'slug' => $term->slug,
											);
										}
									}

									// Add attribute data to the main array.
									$attribute_data = array(
										'name'     => $attribute->attribute_name, // Attribute label.
										'label'    => $attribute->attribute_label, // Attribute label.
										'taxonomy' => $attribute_name, // Attribute taxonomy name.
										'terms'    => $terms_data, // Array of terms data.
									);

									// Add the taxonomy data to the exported attributes array.
									$post_data['custom_attributes'][ $attribute->attribute_name ] = $attribute_data;
								}
							}
						}
					}
					++$count;

					// Assign taxonomy's term values to post.
					if ( '1' === $biel_bytes_import_export_posts_fields['taxonomies'] ) {
						$taxonomies = get_object_taxonomies( $post_type );
						if ( ! empty( $taxonomies ) ) {
							foreach ( $taxonomies as $taxonomy ) {
								$terms      = wp_get_object_terms( $post_id, $taxonomy );
								$term_names = array();
								if ( ! empty( $terms ) ) {
									foreach ( (array) $terms as $term ) {
										$term_names[] = $term->name;
									}
								}

								// Now check for "Uncategorized" specifically.
								$uncategorized_term = get_term_by( 'slug', 'uncategorized', $taxonomy );
								if ( $uncategorized_term && has_term( $uncategorized_term->term_id, $post_id ) ) {
									// "Uncategorized" is selected, ensure it's included.
									$term_names = array_merge( $term_names, array( $uncategorized_term->name ) );
								}

								$post_data['selected_post_terms'][ $taxonomy ] = $term_names;
							}
						}
					}

					// If post type is product.
					if ( 'product' === $post_type && class_exists( 'WC_Product' ) ) {
						$product = wc_get_product( $post_id );
						if ( $product ) {
							// Get the product gallery image IDs.
							$product_gallery_ids = $product->get_gallery_image_ids();

							// Initialize an array to store image data.
							$image_data = array();

							// Loop through the gallery image IDs to retrieve image data.
							if ( ! empty( $product_gallery_ids ) ) {
								foreach ( $product_gallery_ids as $attachment_id ) {
									// Retrieve URL and alt text of the attachment.
									$url = wp_get_attachment_url( $attachment_id );
									$alt = get_post_meta( $attachment_id, '_wp_attachment_alt', true );

									// Add image data to the array.
									$image_data[] = array(
										'url' => $url,
										'alt' => $alt,
									);
								}
							}

							// Assign the image data to the post data.
							$post_data['product_gallery_images'] = $image_data;
						}

						// Get downloadable files data.
						if ( $product && $product->is_downloadable() ) {
							$downloadable_files = $product->get_downloads();
							$downloadable_data  = array();
							if ( ! empty( $downloadable_files ) ) {
								foreach ( $downloadable_files as $key => $file ) {
									$downloadable_data[ $key ] = array(
										'name' => $file['name'],
										'file' => $file['file'],
									);
								}
							}
							// Update the downloadable files meta key.
							$post_data['downloadable_files'] = $downloadable_data;
						}

						// Check if the product is a grouped product.
						if ( $product->is_type( 'grouped' ) ) {
							// Extract grouped product data.
							$grouped_product_name     = $product->get_name();
							$grouped_product_children = $product->get_children();

							// Initialize an array to store child product data.
							$child_product_data = array();

							// Loop through each child product.
							if ( ! empty( $grouped_product_children ) ) {
								foreach ( $grouped_product_children as $child_product_id ) {
									// Get the child product object.
									$child_product = wc_get_product( $child_product_id );

									// Extract child product data.
									if ( ! empty( $child_product ) ) {
										$child_product_name = $child_product->get_name();

										// Add child product data to array.
										$child_product_data[] = array(
											'product_id' => $child_product_id,
											'name'       => $child_product_name,
											'sku'        => $child_product->get_sku(),
										);
									}
								}
							}

							// Add grouped product and child product data to exported array.
							$post_data['grouped_data'] = array(
								'grouped_product_id'   => $post_id,
								'grouped_product_name' => $grouped_product_name,
								'children'             => $child_product_data,
							);
						}

						// Variable product and it's variation.
						if ( $product->is_type( 'variable' ) ) {
							$product_id     = $post_id;
							$product_object = new WC_Product_Variable( $product_id );
							// Get product ID and name.
							$product_id   = $product_object->get_id();
							$product_name = $product_object->get_name();

							// Get product attributes and their term values.
							$product_attributes = array();
							if ( ! empty( $product_object->get_attributes() ) ) {
								foreach ( $product_object->get_attributes() as $attribute_name => $attribute_data ) {
									$attribute_values = array();

									// If it's a custom attribute, handle it differently.
									if ( strpos( $attribute_name, 'pa_' ) !== 0 ) {
										// Use the existing attribute object to get name and options.
										$attribute_name = $attribute_data->get_name();
										$options        = $attribute_data->get_options();
										if ( ! empty( $options ) ) {
											foreach ( $options as $option ) {
												$attribute_values[] = array(
													'name' => $option,
													'slug' => sanitize_title( $option ), // Generate a slug if needed.
												);
											}
										}
									} else {
										// It's a regular attribute, retrieve its terms.
										$terms = wp_get_post_terms( $product_id, $attribute_name );
										if ( ! is_wp_error( $terms ) ) {
											foreach ( $terms as $term ) {
												$attribute_values[] = array(
													'name' => $term->name,
													'slug' => $term->slug,
												);
											}
										}
									}

									// Add attribute values to the product_attributes array.
									$product_attributes[ $attribute_name ] = array(
										'terms' => $attribute_values,
									);
								}
							}

							// Get variations.
							$variations      = array();
							$variation_posts = get_posts(
								array(
									'post_type'      => 'product_variation',
									'post_parent'    => $product_id,
									'posts_per_page' => -1,
								)
							);

							if ( ! empty( $variation_posts ) ) {
								foreach ( $variation_posts as $variation_post ) {
									$variation      = wc_get_product( $variation_post );
									$variation_data = array(
										'ID'         => $variation->get_id(),
										'sku'        => $variation->get_sku(),
										'attributes' => $variation->get_attributes(),
										'price'      => $variation->get_price(),
									);
									$variations[]   = $variation_data;
								}
							}

							// Add product data to the array.
							$product_data = array(
								'ID'         => $product_id,
								'name'       => $product_name,
								'type'       => 'variable',
								'attributes' => $product_attributes,
								'variations' => $variations,
							);

							$post_data['variable_data'] = $product_data;
						}

						// Export product attributes for simple, external and grouped product.
						$product_attributes = array(); // Initialize attribute array.

						if ( 'simple' === $product_type ) {
							$product_object = new WC_Product_Simple( $post_id );
						} elseif ( 'grouped' === $product_type ) {
							$product_object = new WC_Product_Grouped( $post_id );
						} elseif ( 'external' === $product_type ) {
							$product_object = new WC_Product_External( $post_id );
						}

						// Iterate through product attributes.
						if ( ! empty( $product_object->get_attributes() ) ) {
							foreach ( $product_object->get_attributes() as $attribute_name => $attribute_data ) {
								$attribute_values = array();

								// Check if it's a custom attribute.
								if ( strpos( $attribute_name, 'pa_' ) !== 0 ) {
									// Use the existing attribute object to get name and options.
									$attribute_name = $attribute_data->get_name();
									$options        = $attribute_data->get_options();
									if ( ! empty( $options ) ) {
										foreach ( $options as $option ) {
											$attribute_values[] = array(
												'name' => $option,
												'slug' => sanitize_title( $option ), // Generate a slug if needed.
											);
										}
									}
								} else {
									// Retrieve terms for regular attributes.
									$terms = wp_get_post_terms( $product_object->get_id(), $attribute_name );
									if ( ! is_wp_error( $terms ) ) {
										foreach ( $terms as $term ) {
											$attribute_values[] = array(
												'name' => $term->name,
												'slug' => $term->slug,
											);
										}
									}
								}

								// Add attribute values to the product_attributes array.
								$product_attributes[ $attribute_name ] = array(
									'terms' => $attribute_values,
								);
							}
						}

						if ( $product_attributes && ! empty( $product_attributes ) ) {
							// Add product data to the array.
							$product_data = array(
								'ID'         => $product_object->get_id(),
								'name'       => $product_object->get_name(),
								'type'       => $product_object->get_type(),
								'attributes' => $product_attributes,
							);

							// Set product data based on product type.
							if ( 'simple' === $product_type ) {
								$post_data['simple_attribute_data'] = $product_data;
							} elseif ( 'grouped' === $product_type ) {
								$post_data['grouped_attribute_data'] = $product_data;
							} elseif ( 'external' === $product_type ) {
								$post_data['external_attribute_data'] = $product_data;
							}
						}
					}

					// Add post data to export data array.
					$export_data[] = $post_data;
				}

				// Restore global post data.
				wp_reset_postdata();

				// Convert export data to JSON format.
				$json_data = wp_json_encode( $export_data );

				// Send JSON response.
				wp_send_json_success( $json_data );
			} else {
				wp_send_json_error( 'No posts found for the selected post type or author.' );
			}
		} else {
			// Nonce verification failed.
			wp_die( 'Nonce verification failed.', 'Error', array( 'response' => 403 ) );
		}
	}
}

// AJAX handler for date filter.
add_action( 'wp_ajax_get_date_filter', 'biel_get_date_filter' );
add_action( 'wp_ajax_nopriv_get_date_filter', 'biel_get_date_filter' );

if ( ! function_exists( 'biel_get_date_filter' ) ) {
	// get date details.
	function biel_get_date_filter() {
		// Verify nonce before processing the form data.
        if ( ! isset( $_POST['biel_nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash ( $_POST['biel_nonce'] ) ), 'biel_nonce_action' ) || ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( 'Invalid nonce' );
            wp_die();
        }
        $post_type = isset($_POST['post_type']) ? sanitize_text_field($_POST['post_type']) : ''; // phpcs:ignore
		if ( $post_type ) {
			ob_start(); ?>
			<fieldset>
			<legend class="screen-reader-text">
				<?php esc_html_e( 'Date range:', 'bytes-import-export-lite' ); ?>
			</legend>
			<label for="post-start-date" class="label-responsive"><?php esc_html_e( 'Start date:', 'bytes-import-export-lite' ); ?></label>
			<select name="post_start_date" id="post-start-date">
				<option value="0"><?php esc_html_e( '&mdash; Select &mdash;', 'bytes-import-export-lite' ); ?></option>
				<?php biel_export_date_values( $post_type ); ?>
			</select>
			<label for="post-end-date" class="label-responsive"><?php esc_html_e( 'End date:', 'bytes-import-export-lite' ); ?></label>
			<select name="post_end_date" id="post-end-date">
				<option value="0"><?php esc_html_e( '&mdash; Select &mdash;', 'bytes-import-export-lite' ); ?></option>
				<?php biel_export_date_values( $post_type ); ?>
			</select>
			</fieldset>
			<?php
			$html = ob_get_clean();
            echo $html; // phpcs:ignore
		}
		wp_die();
	}
}

// AJAX handler for author filter.
add_action( 'wp_ajax_get_author_filter', 'biel_get_author_filter' );
add_action( 'wp_ajax_nopriv_get_author_filter', 'biel_get_author_filter' );

if ( ! function_exists( 'biel_get_author_filter' ) ) {
	// get author details.
	function biel_get_author_filter() {
		// Verify nonce before processing the form data.
        if ( ! isset( $_POST['biel_nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash ( $_POST['biel_nonce'] ) ), 'biel_nonce_action' ) || ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( 'Invalid nonce' );
            wp_die();
        }
        $post_type = isset($_POST['post_type']) ? sanitize_text_field($_POST['post_type']) : ''; // phpcs:ignore
		if ( $post_type ) {
			ob_start();
			biel_export_author_values( $post_type );
			$html = ob_get_clean();
            echo $html; // phpcs:ignore
		}
		wp_die();
	}
}

// Function to generate date options.
if ( ! function_exists( 'biel_export_date_values' ) ) {
	// export date details.
	function biel_export_date_values( $post_type = 'post' ) {
		global $wpdb, $wp_locale;

        // phpcs:ignore
        $months = $wpdb->get_results(
			$wpdb->prepare(
				"SELECT DISTINCT YEAR( post_date ) AS year, MONTH( post_date ) AS month
                FROM $wpdb->posts
                WHERE post_type = %s AND post_status != 'auto-draft'
                ORDER BY post_date DESC",
				$post_type
			)
		);

		$month_count = count( $months );
		if ( ! $month_count || ( 1 === $month_count && 0 === (int) $months[0]->month ) ) {
			return;
		}

		foreach ( $months as $date ) {
			if ( 0 === (int) $date->year ) {
				continue;
			}

			$month = zeroise( $date->month, 2 );

			printf(
				'<option value="%1$s">%2$s</option>',
				esc_attr( $date->year . '-' . $month ),
				esc_html( $wp_locale->get_month( $month ) . ' ' . $date->year )
			);
		}
	}
}

// Function to generate author options.
if ( ! function_exists( 'biel_export_author_values' ) ) {
	// Get author details.
	function biel_export_author_values( $post_type = 'post' ) {
		// get exported post status.
		$biel_bytes_import_export_posts_status = get_option( 'biel_bytes_import_export_posts_status' );
		?>
		<label><span class="label-responsive"><?php esc_html_e( 'Authors:', 'bytes-import-export-lite' ); ?></span>
		<?php
		global $wpdb;
		$query = "SELECT DISTINCT post_author FROM {$wpdb->posts} WHERE post_type = %s AND post_status = ";
		if ( '0' === $biel_bytes_import_export_posts_status ) {
			$query       .= '%d';
			$status_param = intval( $biel_bytes_import_export_posts_status );
		} else {
			$query       .= '%s';
			$status_param = $biel_bytes_import_export_posts_status;
		}
        $authors = $wpdb->get_col( $wpdb->prepare( $query, $post_type, $status_param ) ); // phpcs:ignore
		$name    = 'post_author_' . $post_type; // Create a unique name based on post type.
		if ( ! empty( $authors ) ) {
			wp_dropdown_users(
				array(
					'include'         => $authors,
					'name'            => $name,
					'multi'           => true,
					'show_option_all' => __( 'All', 'bytes-import-export-lite' ),
					'show'            => 'display_name_with_login',
				)
			);
		} else {
			?>
			<p>Authors are not available, please check saved post status setting.</p>
		<?php } ?>
		</label>
		<?php
	}
}

// AJAX handler for import custom post.
add_action( 'wp_ajax_biel_import_posts', 'biel_import_custom_post' );
add_action( 'wp_ajax_nopriv_biel_import_posts', 'biel_import_custom_post' );

if ( ! function_exists( 'biel_import_custom_post' ) ) {
	// get custom post details.
	function biel_import_custom_post() {
		// Verify nonce, action.
		if ( isset( $_POST['action'] ) && 'biel_import_posts' === $_POST['action'] && isset( $_POST['security'] ) && wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['security'] ) ), 'biel_import_posts_nonce' ) && current_user_can( 'manage_options' ) ) {
			// Retrive count variable.
			$count = isset( $_POST['count'] ) ? intval( $_POST['count'] ) : 1;

			session_start(); // Start session.
			$import_post_count = isset( $_SESSION['import_post_count'] ) ? intval( sanitize_key( $_SESSION['import_post_count'] ) ) : 0; // Session count variable.

			// Use a conditional check to initialize the array only if the session doesn't exist.
			$author_mapping = isset( $_SESSION['author_map_arr'] ) ? stripslashes( sanitize_text_field( wp_unslash( $_SESSION['author_map_arr'] ) ) ) : array();

			$batch_size      = isset( $_POST['batch_size'] ) ? intval( $_POST['batch_size'] ) : 10; // Default batch size.
			$import_messages = array(); // Array to store import messages.

			if ( 1 === $count ) {
				// Retrieve the serialized data from the hidden field.
				$serialized_data = isset( $_POST['import_author'] ) ? stripslashes( sanitize_text_field( wp_unslash( $_POST['import_author'] ) ) ) : '';

				// Unserialize the data.
				$authors        = json_decode( $serialized_data, true );
				$author_mapping = array();

				if ( ! isset( $_POST['imported_authors'] ) ) {
					$import_messages[] = esc_html__( 'Author details are not available.', 'bytes-import-export-lite' );
				}

				if ( isset( $_POST['imported_authors'] ) && ! empty( $_POST['imported_authors'] ) ) {
					$create_users = true; // Allow to create new user.
					// Sanitize the imported_authors array.
					$imported_authors = array_map( 'sanitize_text_field', wp_unslash( $_POST['imported_authors'] ) );

					foreach ( $imported_authors as $i => $old_login ) {
						// Sanitize here to stop breakage in process_posts.
						$santized_old_login = sanitize_user( $old_login, true );
						$old_id             = isset( $authors[ $old_login ]['author_id'] ) ? intval( $authors[ $old_login ]['author_id'] ) : false;

						if ( ! empty( $_POST['user_map'][ $i ] ) ) {
							$user = get_userdata( intval( $_POST['user_map'][ $i ] ) );
							if ( isset( $user->ID ) ) {
								$author_mapping[ $santized_old_login ] = $user->ID;
							}
						} elseif ( $create_users ) {
							$user_data = array(
								'user_login'   => $old_login,
								'user_pass'    => wp_generate_password(),
								'user_email'   => isset( $authors[ $old_login ]['author_email'] ) ? $authors[ $old_login ]['author_email'] : '',
								'display_name' => $authors[ $old_login ]['author_display_name'],
								'first_name'   => isset( $authors[ $old_login ]['author_first_name'] ) ? $authors[ $old_login ]['author_first_name'] : '',
								'last_name'    => isset( $authors[ $old_login ]['author_last_name'] ) ? $authors[ $old_login ]['author_last_name'] : '',
								'role'         => 'author',
							);
							$user_id   = wp_insert_user( $user_data );

							if ( ! is_wp_error( $user_id ) ) {
								$author_mapping[ $santized_old_login ] = $user_id;
							} elseif ( 1 === $count ) {
									$import_messages[] = sprintf(
										// translators: %s represents the display name of the user whose creation failed.
										esc_html__(
											'Failed to create new user for %s. Their posts will be attributed to the current user.',
											'bytes-import-export-lite'
										),
										esc_html( $authors[ $old_login ]['author_display_name'] )
									);
							}
						}

						// If the user_id was invalid, default to the current user.
						if ( ! isset( $author_mapping[ $santized_old_login ] ) ) {
							$author_mapping[ $santized_old_login ] = (int) get_current_user_id();
						}
					}
					$_SESSION['author_map_arr'] = $author_mapping; // Assign authors to session variable.
				}
			}

			// Retrieve the uploaded file id from the hidden field.
			$import_id = isset( $_POST['import_id'] ) ? intval( $_POST['import_id'] ) : 0;

			// Get uploaded file data.
			$file = get_attached_file( $import_id );

	        $import_data  = file_get_contents($file); // phpcs:ignore
			$custom_posts = json_decode( $import_data, true );

			// Get total posts.
			$total_posts = count( $custom_posts );

			// Process posts in batches.
			$posts_to_process = array_slice( $custom_posts, ( $count - 1 ) * $batch_size, $batch_size );

			// Increment the internal counter for count.
			$_SESSION['import_post_count'] = $import_post_count + count( $posts_to_process );

			$exported_post_type = ( ! empty( $custom_posts[0]['post_type'] ) ) ? $custom_posts[0]['post_type'] : ''; // exported post type.

			$post_type_exists = post_type_exists( $exported_post_type ); // post type exists or not.

			// Process the imported data.
			if ( $posts_to_process && ! empty( $posts_to_process ) && $post_type_exists ) {
				if ( 'product' !== $exported_post_type ) {
					require_once BIEL_PLUGIN_DIR_PATH . 'admin/partials/import/bytes-import-export-lite-custom-posts-type.php';
				} else {
					require_once BIEL_PLUGIN_DIR_PATH . 'admin/partials/import/bytes-import-export-lite-product-type.php';
				}
			} else {
				$import_messages[] = esc_html__( 'Error: Unable to decode imported data. Please check post type exists or not.', 'bytes-import-export-lite' );
			}

			// Send response with messages and total posts (if available).
			$response_data = array( 'messages' => $import_messages );
			if ( $total_posts ) {
				$response_data['totalPosts'] = $total_posts;
			}
			wp_send_json_success( $response_data );
			wp_die();
		} else {
			// Handle invalid or missing form submission.
			wp_send_json_error(
				array(
					'status'  => false,
					'message' => 'Invalid form submission.',
				)
			);
			wp_die();
		}
	}
}
