<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class WCBNPCL_Settings {

    public function __construct() {
        add_action( 'admin_menu', [ $this, 'register_settings_page' ] );
        add_action( 'admin_init', [ $this, 'register_settings' ] );
        add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_admin_assets' ] );
        add_action( 'admin_init', [ $this, 'maybe_reset_settings' ] );
        add_action('admin_notices', [$this, 'render_remote_notices']);
        add_action('wp_ajax_wcbnpc_dismiss_notice', [$this, 'ajax_dismiss_notice']);
    }

    public function register_settings_page() {
        add_submenu_page(
            'woocommerce',
            __( 'Buy Now Popup Settings', 'buy-now-popup-instant-checkout-lite-for-woocommerce' ),
            __( 'Buy Now Popup', 'buy-now-popup-instant-checkout-lite-for-woocommerce' ),
            'manage_woocommerce',
            'wcbnpc-settings',
            [ $this, 'render_settings_page' ]
        );
    }

    public function enqueue_admin_assets() {
        wp_enqueue_style(
            'wcbnpc-admin-style',
            WCBNPCL_ASSETS_URL . 'admin/css/admin.css',
            [],
            WCBNPCL_VERSION
        );
        
        wp_enqueue_style( 'wp-color-picker' );
        
        wp_enqueue_script( 'wp-color-picker' );

        wp_enqueue_script(
            'jquery-ui-sortable' 
        );

        wp_enqueue_script(
            'wcbnpc-admin-script',
            WCBNPCL_ASSETS_URL . 'admin/js/admin.js',
            [ 'jquery', 'jquery-ui-sortable' ],
            WCBNPCL_VERSION,
            true
        );
        wp_localize_script('wcbnpc-admin-script', 'wcbnpc_admin_data', [
            'nonce' => wp_create_nonce('wcbnpc_dismiss_notice'),
        ]);
    }

    public function register_settings() {
        // Register the setting group with sanitization callback
        register_setting(
            'wcbnpc_settings_group',
            'wcbnpc_settings',
            [
                'sanitize_callback' => [ $this, 'sanitize_wcbnpc_settings' ],
            ]
        );

        // Register checkout fields option
        register_setting(
            'wcbnpc_settings_group',
            'wcbnpc_checkout_fields',
            [
                'sanitize_callback' => [ $this, 'sanitize_checkout_fields' ]
            ]
        );

        // Register payment gateways option
        register_setting(
            'wcbnpc_settings_group',
            'wcbnpc_enabled_payment_gateways',
            [
                'sanitize_callback' => [ $this, 'sanitize_array_of_keys' ]
            ]
        );

        // Register shipping methods option
        register_setting(
            'wcbnpc_settings_group',
            'wcbnpc_enabled_shipping_methods',
            [
                'sanitize_callback' => [ $this, 'sanitize_shipping_methods' ]
            ]
        );

        // Add General Settings section
        add_settings_section(
            'wcbnpc_general_section',
            __( 'General Settings', 'buy-now-popup-instant-checkout-lite-for-woocommerce' ),
            null,
            'wcbnpc-settings-general'
        );

        // Enable plugin checkbox
        add_settings_field(
            'wcbnpc_enable',
            __( 'Enable Plugin', 'buy-now-popup-instant-checkout-lite-for-woocommerce' ),
            [ $this, 'render_checkbox_field' ],
            'wcbnpc-settings-general',
            'wcbnpc_general_section',
            [
                'label_for' => 'wcbnpc_enable',
                'option_key' => 'wcbnpc_enable',
                'description' => __( 'Enable the Buy Now Popup functionality.', 'buy-now-popup-instant-checkout-lite-for-woocommerce' )
            ]
        );

        // Enable button on shop page
        add_settings_field(
            'wcbnpc_button_enable_shop_page',
            __( 'Shop Page', 'buy-now-popup-instant-checkout-lite-for-woocommerce' ),
            [ $this, 'render_checkbox_field' ],
            'wcbnpc-settings-general',
            'wcbnpc_general_section',
            [
                'label_for' => 'wcbnpc_button_enable_shop_page',
                'option_key' => 'wcbnpc_button_enable_shop_page',
                'description' => __( 'Enable the Buy Now Popup button on shop page.', 'buy-now-popup-instant-checkout-lite-for-woocommerce' )
            ]
        );

        // Button text
        add_settings_field(
            'wcbnpc_button_text',
            __( 'Button Text', 'buy-now-popup-instant-checkout-lite-for-woocommerce' ),
            [ $this, 'render_text_field' ],
            'wcbnpc-settings-general',
            'wcbnpc_general_section',
            [
                'label_for' => 'wcbnpc_button_text',
                'option_key' => 'wcbnpc_button_text',
                'description' => __( 'Text displayed on the Buy Now button.', 'buy-now-popup-instant-checkout-lite-for-woocommerce' )
            ]
        );

        // Button position
        add_settings_field(
            'wcbnpc_button_position',
            __( 'Button Position', 'buy-now-popup-instant-checkout-lite-for-woocommerce' ),
            [ $this, 'render_select_field' ],
            'wcbnpc-settings-general',
            'wcbnpc_general_section',
            [
                'label_for' => 'wcbnpc_button_position',
                'option_key' => 'wcbnpc_button_position',
                'options' => [
                    'woocommerce_after_add_to_cart_button' => __( 'After Add to Cart button', 'buy-now-popup-instant-checkout-lite-for-woocommerce' ),
                    'woocommerce_before_add_to_cart_form' => __( 'Before Add to Cart Form', 'buy-now-popup-instant-checkout-lite-for-woocommerce' )
                ],
                'description' => __( 'Where to place the Buy Now button on the product page.', 'buy-now-popup-instant-checkout-lite-for-woocommerce' )
            ]
        );

        // Place Order Button Text
        add_settings_field(
            'wcbnpc_place_order_text',
            __( 'Place Order Button Text', 'buy-now-popup-instant-checkout-lite-for-woocommerce' ),
            [ $this, 'render_text_field' ],
            'wcbnpc-settings-general',
            'wcbnpc_general_section',
            [
                'label_for' => 'wcbnpc_place_order_text',
                'option_key' => 'wcbnpc_place_order_text',
                'description' => __( 'Custom text for the Place Order button.', 'buy-now-popup-instant-checkout-lite-for-woocommerce' )
            ]
        );

        // Billing Section Title
        add_settings_field(
            'wcbnpc_section_title_billing',
            __( 'Billing Section Title', 'buy-now-popup-instant-checkout-lite-for-woocommerce' ),
            [ $this, 'render_text_field' ],
            'wcbnpc-settings-general',
            'wcbnpc_general_section',
            [
                'label_for' => 'wcbnpc_section_title_billing',
                'option_key' => 'wcbnpc_section_title_billing'
            ]
        );

        // Shipping Section Title
        add_settings_field(
            'wcbnpc_section_title_shipping',
            __( 'Shipping Section Title', 'buy-now-popup-instant-checkout-lite-for-woocommerce' ),
            [ $this, 'render_text_field' ],
            'wcbnpc-settings-general',
            'wcbnpc_general_section',
            [
                'label_for' => 'wcbnpc_section_title_shipping',
                'option_key' => 'wcbnpc_section_title_shipping'
            ]
        );

        // Shipping Methods Title
        add_settings_field(
            'wcbnpc_section_title_shipping_method',
            __( 'Shipping Methods Title', 'buy-now-popup-instant-checkout-lite-for-woocommerce' ),
            [ $this, 'render_text_field' ],
            'wcbnpc-settings-general',
            'wcbnpc_general_section',
            [
                'label_for' => 'wcbnpc_section_title_shipping_method',
                'option_key' => 'wcbnpc_section_title_shipping_method'
            ]
        );

        // Payment Methods Title
        add_settings_field(
            'wcbnpc_section_title_payment_method',
            __( 'Payment Methods Title', 'buy-now-popup-instant-checkout-lite-for-woocommerce' ),
            [ $this, 'render_text_field' ],
            'wcbnpc-settings-general',
            'wcbnpc_general_section',
            [
                'label_for' => 'wcbnpc_section_title_payment_method',
                'option_key' => 'wcbnpc_section_title_payment_method'
            ]
        );

        // Show Sale Badge
        add_settings_field(
            'wcbnpc_show_badge',
            __( 'Show Sale Badge', 'buy-now-popup-instant-checkout-lite-for-woocommerce' ),
            [ $this, 'render_checkbox_field' ],
            'wcbnpc-settings-general',
            'wcbnpc_general_section',
            [
                'label_for' => 'wcbnpc_show_badge',
                'option_key' => 'wcbnpc_show_badge'
            ]
        );

        // Badge Type
        add_settings_field(
            'wcbnpc_badge_type',
            __( 'Badge Type', 'buy-now-popup-instant-checkout-lite-for-woocommerce' ),
            [ $this, 'render_select_field' ],
            'wcbnpc-settings-general',
            'wcbnpc_general_section',
            [
                'label_for' => 'wcbnpc_badge_type',
                'option_key' => 'wcbnpc_badge_type',
                'options' => [
                    'text' => 'Text',
                    'percentage_pro' => 'Percentage (Upgrade to PRO)'
                ]
            ]
        );

        // Badge Text
        add_settings_field(
            'wcbnpc_badge_text',
            __( 'Badge Text (if "Text")', 'buy-now-popup-instant-checkout-lite-for-woocommerce' ),
            [ $this, 'render_text_field' ],
            'wcbnpc-settings-general',
            'wcbnpc_general_section',
            [
                'label_for' => 'wcbnpc_badge_text',
                'option_key' => 'wcbnpc_badge_text'
            ]
        );

        // Badge Color
        add_settings_field(
            'wcbnpc_badge_color',
            __( 'Badge Color', 'buy-now-popup-instant-checkout-lite-for-woocommerce' ),
            [ $this, 'render_color_picker_field' ],
            'wcbnpc-settings-general',
            'wcbnpc_general_section',
            [
                'label_for' => 'wcbnpc_badge_color',
                'option_key' => 'wcbnpc_badge_color',
                'description' => __( 'Use any CSS color code (e.g. #ff0000)', 'buy-now-popup-instant-checkout-lite-for-woocommerce' )
            ]
        );

        // Button Color
        add_settings_field(
            'wcbnpc_button_color',
            __( 'Button Color', 'buy-now-popup-instant-checkout-lite-for-woocommerce' ),
            [ $this, 'render_color_picker_field' ],
            'wcbnpc-settings-general',
            'wcbnpc_general_section',
            [
                'label_for' => 'wcbnpc_button_color',
                'option_key' => 'wcbnpc_button_color',
                'description' => __( 'Use any CSS color code (e.g. #00aa00)', 'buy-now-popup-instant-checkout-lite-for-woocommerce' )
            ]
        );

        // Checkout Fields section
        add_settings_section(
            'wcbnpc_checkout_fields_section',
            null,
            null,
            'wcbnpc-settings-general'
        );

        // Payment Methods section
        add_settings_section(
            'wcbnpc_payment_section',
            __( 'Payment Methods', 'buy-now-popup-instant-checkout-lite-for-woocommerce' ),
            null,
            'wcbnpc-settings-payment'
        );

        add_settings_field(
            'wcbnpc_enabled_payment_gateways',
            __( 'Enabled Gateways', 'buy-now-popup-instant-checkout-lite-for-woocommerce' ),
            [ $this, 'render_payment_gateways_field' ],
            'wcbnpc-settings-payment',
            'wcbnpc_payment_section'
        );

        // Shipping Methods section
        add_settings_section(
            'wcbnpc_shipping_section',
            __( 'Shipping Methods', 'buy-now-popup-instant-checkout-lite-for-woocommerce' ),
            null,
            'wcbnpc-settings-shipping'
        );

        add_settings_field(
            'wcbnpc_enabled_shipping_methods',
            '', // no label
            [ $this, 'render_shipping_methods_table' ],
            'wcbnpc-settings-shipping',
            'wcbnpc_shipping_section'
        );

        // Free Shipping section
        add_settings_section(
            'wcbnpc_free_shipping_section',
            __( 'Free Shipping', 'buy-now-popup-instant-checkout-lite-for-woocommerce' ),
            null,
            'wcbnpc-settings-shipping'
        );

        // Free Shipping Notice - Text
        add_settings_field(
            'wcbnpc_free_shipping_text',
            __( 'Free Shipping Message', 'buy-now-popup-instant-checkout-lite-for-woocommerce' ),
            [ $this, 'render_text_field' ],
            'wcbnpc-settings-shipping',
            'wcbnpc_free_shipping_section',
            [
                'label_for' => 'wcbnpc_free_shipping_text',
                'option_key' => 'wcbnpc_free_shipping_text',
                'description' => __( 'Message displayed when free shipping is applied.', 'buy-now-popup-instant-checkout-lite-for-woocommerce' )
            ]
        );

        // Free Shipping Background Color
        add_settings_field(
            'wcbnpc_free_shipping_bg',
            __( 'Free Shipping BG Color', 'buy-now-popup-instant-checkout-lite-for-woocommerce' ),
            [ $this, 'render_color_picker_field' ],
            'wcbnpc-settings-shipping',
            'wcbnpc_free_shipping_section',
            [
                'label_for' => 'wcbnpc_free_shipping_bg',
                'option_key' => 'wcbnpc_free_shipping_bg',
                'description' => __( 'Use any CSS color code (e.g. #ff0000)', 'buy-now-popup-instant-checkout-lite-for-woocommerce' )
            ]
        );

        // Free Shipping Text Color
        add_settings_field(
            'wcbnpc_free_shipping_color',
            __( 'Free Shipping Text Color', 'buy-now-popup-instant-checkout-lite-for-woocommerce' ),
            [ $this, 'render_color_picker_field' ],
            'wcbnpc-settings-shipping',
            'wcbnpc_free_shipping_section',
            [
                'label_for' => 'wcbnpc_free_shipping_color',
                'option_key' => 'wcbnpc_free_shipping_color',
                'description' => __( 'Use any CSS color code (e.g. #ff0000)', 'buy-now-popup-instant-checkout-lite-for-woocommerce' )
            ]
        );
    }

    /**
     * Sanitize the wcbnpc_settings option.
     *
     * @param mixed $input The input data to sanitize.
     * @return array Sanitized settings array.
     */
    public function sanitize_wcbnpc_settings( $input ) {
        $sanitized_input = [];

        // If input is not an array, return an empty array to prevent errors
        if ( ! is_array( $input ) ) {
            return $sanitized_input;
        }

        // Checkboxes: Treat as boolean (1 or 0)
        $checkboxes = [
            'wcbnpc_enable',
            'wcbnpc_button_enable_shop_page',
            'wcbnpc_show_badge',
        ];
        foreach ( $checkboxes as $key ) {
            $sanitized_input[ $key ] = ! empty( $input[ $key ] ) ? '1' : '0';
        }

        // Text fields: Sanitize as plain text
        $text_fields = [
            'wcbnpc_button_text',
            'wcbnpc_place_order_text',
            'wcbnpc_section_title_billing',
            'wcbnpc_section_title_shipping',
            'wcbnpc_section_title_shipping_method',
            'wcbnpc_section_title_payment_method',
            'wcbnpc_badge_text',
            'wcbnpc_free_shipping_text',
        ];
        foreach ( $text_fields as $key ) {
            if ( isset( $input[ $key ] ) ) {
                $sanitized_input[ $key ] = sanitize_text_field( $input[ $key ] );
            }
        }

        // Select fields: Validate against allowed options
        if ( isset( $input['wcbnpc_button_position'] ) ) {
            $allowed_positions = [
                'woocommerce_after_add_to_cart_button',
                'woocommerce_before_add_to_cart_form',
            ];
            $sanitized_input['wcbnpc_button_position'] = in_array( $input['wcbnpc_button_position'], $allowed_positions, true )
                ? $input['wcbnpc_button_position']
                : 'woocommerce_after_add_to_cart_button'; // Default value
        }

        if ( isset( $input['wcbnpc_badge_type'] ) ) {
            $allowed_types = [ 'text', 'percentage' ];
            $sanitized_input['wcbnpc_badge_type'] = in_array( $input['wcbnpc_badge_type'], $allowed_types, true )
                ? $input['wcbnpc_badge_type']
                : 'text'; // Default value
        }

        // Color fields: Sanitize as hexadecimal color codes
        $color_fields = [
            'wcbnpc_button_color',
            'wcbnpc_badge_color',
            'wcbnpc_free_shipping_bg',
            'wcbnpc_free_shipping_color',
        ];
        foreach ( $color_fields as $key ) {
            if ( isset( $input[ $key ] ) ) {
                // Ensure it's a valid hex color (e.g., #ffffff or #fff)
                $sanitized_input[ $key ] = preg_match( '/^#([A-Fa-f0-9]{6}|[A-Fa-f0-9]{3})$/', $input[ $key ] )
                    ? $input[ $key ]
                    : ''; // Empty if invalid
            }
        }

        // Field order: Sanitize as comma-separated list of keys
        if ( isset( $input['field_order'] ) ) {
            $order = explode( ',', sanitize_text_field( wp_unslash( $input['field_order'] ) ) );
            $sanitized_input['field_order'] = array_map( 'sanitize_key', array_filter( $order ) );
        }

        return $sanitized_input;
    }

    public function render_settings_page() {
        nocache_headers();
        // Verify nonce for reset action
        $show_reset_notice = false;
        if ( isset( $_GET['reset'] ) && '1' === $_GET['reset'] ) {
            $nonce = isset( $_GET['reset_nonce'] ) ? sanitize_text_field( wp_unslash( $_GET['reset_nonce'] ) ) : '';
            if ( wp_verify_nonce( $nonce, 'wcbnpc_reset_action' ) ) {
                $show_reset_notice = true;
            }
        }
    ?>
    <div class="wcbnpc-settings-wrapper">
        <h1><?php esc_html_e( 'Buy Now Popup Instant Checkout for WooCommerce', 'buy-now-popup-instant-checkout-lite-for-woocommerce' ); ?></h1>
        <?php if ( $show_reset_notice ) : ?>
            <div class="notice notice-success is-dismissible"><p><?php esc_html_e( 'Settings have been reset.', 'buy-now-popup-instant-checkout-lite-for-woocommerce' ); ?></p></div>
        <?php endif; ?>
        <nav class="wcbnpc-tabs">
            <a href="#" class="wcbnpc-tab active" data-tab="general"><?php esc_html_e( 'General', 'buy-now-popup-instant-checkout-lite-for-woocommerce' ); ?></a>
            <a href="#" class="wcbnpc-tab" data-tab="checkout"><?php esc_html_e( 'Checkout Fields', 'buy-now-popup-instant-checkout-lite-for-woocommerce' ); ?></a>
            <a href="#" class="wcbnpc-tab" data-tab="payment"><?php esc_html_e( 'Payment Methods', 'buy-now-popup-instant-checkout-lite-for-woocommerce' ); ?></a>
            <a href="#" class="wcbnpc-tab" data-tab="shipping"><?php esc_html_e( 'Shipping Methods', 'buy-now-popup-instant-checkout-lite-for-woocommerce' ); ?></a>
            <a href="#" class="wcbnpc-tab" style="background:red; color:#fff;" data-tab="pro-version"><?php esc_html_e( 'Upgrade to PRO', 'buy-now-popup-instant-checkout-lite-for-woocommerce' ); ?></a>
        </nav>

        <form method="post" action="options.php">
            <?php
            // Verify nonce for form submission
            if ( isset( $_POST['wcbnpc_settings'] ) || isset( $_POST['wcbnpc_checkout_fields'] ) || isset( $_POST['wcbnpc_enabled_payment_gateways'] ) || isset( $_POST['wcbnpc_enabled_shipping_methods'] ) ) {
                $nonce = isset( $_POST['wcbnpc_settings_nonce'] ) ? sanitize_text_field( wp_unslash( $_POST['wcbnpc_settings_nonce'] ) ) : '';
                if ( ! wp_verify_nonce( $nonce, 'wcbnpc_settings_action' ) ) {
                    wp_die( esc_html__( 'Security check failed.', 'buy-now-popup-instant-checkout-lite-for-woocommerce' ) );
                }
            }

            settings_fields( 'wcbnpc_settings_group' );
            ?>

            <!-- General Tab -->
            <div class="wcbnpc-tab-content" data-tab="general" style="display: block;">
                <?php do_settings_sections( 'wcbnpc-settings-general' ); ?>
                <div class="wcbnpc-pro-card">Shortcode: [wcbnpc_buy_now id='PRODUCT_ID']</div>
            </div>

            <!-- Checkout Fields -->
            <div class="wcbnpc-tab-content" data-tab="checkout" style="display: none;">
                <h2><?php esc_html_e( 'Checkout Fields', 'buy-now-popup-instant-checkout-lite-for-woocommerce' ); ?></h2>
                <?php $this->render_checkout_fields_table(); ?>
            </div>

            <!-- Payment -->
            <div class="wcbnpc-tab-content" data-tab="payment" style="display: none;">
                <?php do_settings_sections( 'wcbnpc-settings-payment' ); ?>
            </div>

            <!-- Shipping -->
            <div class="wcbnpc-tab-content" data-tab="shipping" style="display: none;">
                <?php do_settings_sections( 'wcbnpc-settings-shipping' ); ?>
            </div>
            
            <!-- PRO VERSION -->
            <div class="wcbnpc-tab-content" data-tab="pro-version" style="display: none;">
                <?php $this->render_pro_promo_card(); ?>
            </div>
            
            <?php submit_button(); ?>
            <input type="submit" name="wcbnpc_reset_settings" class="button button-secondary" value="<?php esc_attr_e( 'Reset Settings', 'buy-now-popup-instant-checkout-lite-for-woocommerce' ); ?>" onclick="return confirm('<?php esc_attr_e( 'Are you sure you want to reset all settings?', 'buy-now-popup-instant-checkout-lite-for-woocommerce' ); ?>');">
            <?php wp_nonce_field( 'wcbnpc_reset_settings_action', 'wcbnpc_reset_settings_nonce' ); ?>
        </form>
    </div>
    <?php
    }

    public function render_checkbox_field( $args ) {
        $options = get_option( 'wcbnpc_settings', [] );
        $checked = isset( $options[ $args['option_key'] ] ) && '1' === $options[ $args['option_key'] ];
        echo '<input type="checkbox" id="' . esc_attr( $args['label_for'] ) . '" name="wcbnpc_settings[' . esc_attr( $args['option_key'] ) . ']" value="1" ' . checked( $checked, true, false ) . ' />';
        if ( ! empty( $args['description'] ) ) {
            echo '<p class="description">' . esc_html( $args['description'] ) . '</p>';
        }
    }

    public function render_text_field( $args ) {
        $options = get_option( 'wcbnpc_settings', [] );
        $value = isset( $options[ $args['option_key'] ] ) ? esc_attr( $options[ $args['option_key'] ] ) : '';
        echo '<input type="text" id="' . esc_attr( $args['label_for'] ) . '" name="wcbnpc_settings[' . esc_attr( $args['option_key'] ) . ']" value="' . esc_attr( $value ) . '" class="regular-text" />';
        if ( ! empty( $args['description'] ) ) {
            echo '<p class="description">' . esc_html( $args['description'] ) . '</p>';
        }
    }

    public function render_select_field( $args ) {
        $options = get_option( 'wcbnpc_settings', [] );
        $selected = isset( $options[ $args['option_key'] ] ) ? $options[ $args['option_key'] ] : '';
        echo '<select id="' . esc_attr( $args['label_for'] ) . '" name="wcbnpc_settings[' . esc_attr( $args['option_key'] ) . ']">';
        foreach ( $args['options'] as $value => $label ) {
            echo '<option value="' . esc_attr( $value ) . '" ' . selected( $selected, $value, false ) . '>' . esc_html( $label ) . '</option>';
        }
        echo '</select>';
        if ( ! empty( $args['description'] ) ) {
            echo '<p class="description">' . esc_html( $args['description'] ) . '</p>';
        }
    }
    
    public static function get_removable_native_fields() {
        return [
            'billing_company',
            'billing_country',
            'billing_address_2',
            'billing_postcode',
            'billing_phone',
            'shipping_company',
            'shipping_country',
            'shipping_address_2',
            'shipping_postcode',
            'shipping_phone',
            'order_comments'
        ];
    }
    private function render_pro_promo_card() {
    // Don’t show if PRO is active (adjust this condition to your PRO check).
    if ( defined( 'WCBNPC_PRO_ACTIVE' ) && WCBNPC_PRO_ACTIVE ) {
        return;
    }

    $upgrade_url = 'https://wizewp.com/plugins/buy-now-instant-checkout/?utm_source=plugin&utm_medium=lite-admin&utm_campaign=buy-now-pro';
    ?>
    <div class="wcbnpc-pro-card">
        <div class="wcbnpc-pro-card__left">
            <div class="wcbnpc-pro-badge">PRO</div>
            <h3 class="wcbnpc-pro-title">Unlock full power: more gateways, field editor, per-product control</h3>
            <p class="wcbnpc-pro-desc">
                Go beyond the basics. Enable Stripe & PayPal, drag-and-drop checkout fields, custom fields, and per-product "Buy Now" control.
            </p>
            <ul class="wcbnpc-pro-list">
                <li><span class="dashicons dashicons-yes"></span> Stripe & PayPal (plus more gateways)</li>
                <li><span class="dashicons dashicons-yes"></span> Drag-and-drop Checkout Fields Editor</li>
                <li><span class="dashicons dashicons-yes"></span> Custom checkout fields & labels</li>
                <li><span class="dashicons dashicons-yes"></span> Per-product “Buy Now” control</li>
                <li><span class="dashicons dashicons-yes"></span> Advanced Shortcode [wcbnpc_buy_now id='PRODUCT_ID' class='CUSTOM_CSS_CLASS' text='CUSTOM_TEXT']</li>
                <li><span class="dashicons dashicons-yes"></span> Priority support & updates</li>
            </ul>

            <a class="button button-primary button-hero wcbnpc-pro-cta"
               href="<?php echo esc_url( $upgrade_url ); ?>"
               target="_blank" rel="noopener">
               Upgrade to PRO
            </a>

            <p class="wcbnpc-pro-note">
                You’re using the LITE version. The PRO version adds more payment gateways, full checkout customization, and advanced shipping logic.
            </p>
        </div>
    </div>
    <?php
    }
    public function render_checkout_fields_table() {
        $fields = get_option( 'wcbnpc_checkout_fields', [] );
        $settings = get_option( 'wcbnpc_settings', [] );
        $field_order = isset( $settings['field_order'] ) ? $settings['field_order'] : array_keys( $fields );

        if ( empty( $fields ) ) {
            $default_fields = WC()->checkout->get_checkout_fields();

            $fields = [];
            foreach ( $default_fields as $section => $section_fields ) {
                foreach ( $section_fields as $key => $field ) {
                    $fields[ $key ] = [
                        'label' => $field['label'] ?? $key,
                        'required' => ! empty( $field['required'] ),
                        'visible' => true,
                        'custom' => false
                    ];
                }
            }
            if ( ! isset( $fields['selected_location'] ) ) {
                $fields['selected_location'] = [
                    'label' => __( 'Ship to a different address?', 'buy-now-popup-instant-checkout-lite-for-woocommerce'),
                    'required' => false,
                    'visible' => true,
                    'custom' => false
                ];
            }
            update_option( 'wcbnpc_checkout_fields', $fields );
            $field_order = array_keys( $fields );
            $settings['field_order'] = $field_order;
            update_option( 'wcbnpc_settings', $settings );
        }
        $promo_img = WCBNPCL_ASSETS_URL . 'admin/img/checkout_fields_admin.gif';
        echo '<img src="'.esc_attr($promo_img).'" />';
        $this->render_pro_promo_card();
    }

    public function sanitize_checkout_fields( $input ) {
        $sanitized   = [];
        $settings    = get_option( 'wcbnpc_settings', [] );
        $field_order = isset( $settings['field_order'] ) && is_array( $settings['field_order'] ) ? $settings['field_order'] : [];

        if ( ! is_array( $input ) ) {
            return $sanitized;
        }

        $wc_fields_billing  = WC()->checkout->get_checkout_fields( 'billing' );
        $wc_fields_shipping = WC()->checkout->get_checkout_fields( 'shipping' );
        $allowed_widths = [ '3', '4', '6', '12' ];

        foreach ( $input as $key => $field ) {
            if ( ! is_array( $field ) ) {
                continue;
            }

            $raw_key   = isset( $field['key'] ) ? $field['key'] : $key;
            $field_key = sanitize_key( $raw_key );

            if ( empty( $field_key ) || ! preg_match( '/^[a-z0-9_]+$/', $field_key ) ) {
                continue;
            }

            $label = isset( $field['label'] ) ? sanitize_text_field( $field['label'] ) : $field_key;
            $required = ! empty( $field['required'] ) && ( $field['required'] === '1' || $field['required'] === 1 || $field['required'] === true || $field['required'] === 'on' );
            $visible  = ! empty( $field['visible'] )  && ( $field['visible']  === '1' || $field['visible']  === 1 || $field['visible']  === true || $field['visible']  === 'on' );
        
            $width_raw = isset( $field['width'] ) ? (string) $field['width'] : '12';
            $width     = in_array( $width_raw, $allowed_widths, true ) ? $width_raw : '12';

            $is_custom = ! array_key_exists( $field_key, $wc_fields_billing )
                  && ! array_key_exists( $field_key, $wc_fields_shipping );
            $sanitized[ $field_key ] = [
                'label'    => $label,
                'required' => $required,
                'visible'  => $visible,
                'width'    => $width,
                'custom'   => $is_custom,
            ];
        }

        if ( ! empty( $field_order ) ) {
            $ordered = [];
            foreach ( $field_order as $k ) {
                if ( isset( $sanitized[ $k ] ) ) {
                    $ordered[ $k ] = $sanitized[ $k ];
                }
            }
            foreach ( $sanitized as $k => $v ) {
                if ( ! isset( $ordered[ $k ] ) ) {
                    $ordered[ $k ] = $v;
                }
            }
            $sanitized = $ordered;
        }
        return $sanitized;
    }

    public function render_payment_gateways_field() {
        $selected = get_option( 'wcbnpc_enabled_payment_gateways', [] );
        $gateways = WC()->payment_gateways()->get_available_payment_gateways();

        echo '<div class="wcbnpc-payment-gateways">';
        foreach ( $gateways as $gateway_id => $gateway ) {
            if($gateway_id != 'cod') continue;
            echo '<label style="display:block; margin-bottom: 8px;">';
            echo '<input type="checkbox" name="wcbnpc_enabled_payment_gateways[]" value="' . esc_attr( $gateway_id ) . '" ' . checked( in_array( $gateway_id, $selected, true ), true, false ) . ' /> ';
            echo esc_html( $gateway->get_title() ) . ' (' . esc_html( $gateway_id ) . ')';
            echo '</label>';
        }
        echo '</div>';
        $this->render_pro_promo_card();
    }

    public function sanitize_array_of_keys( $input ) {
        if ( ! is_array( $input ) ) {
            return [];
        }

        return array_map( 'sanitize_key', $input );
    }

    public function render_shipping_methods_table() {
        $saved = get_option( 'wcbnpc_enabled_shipping_methods', [] );
        $zones = WC_Shipping_Zones::get_zones();
        $default_zone = new WC_Shipping_Zone( 0 );
        $zones[] = [
            'zone_id' => 0,
            'zone_name' => $default_zone->get_zone_name(),
            'shipping_methods' => $default_zone->get_shipping_methods(),
        ];

        echo '<table class="widefat wcbnpc-shipping-table" style="min-width: 800px">';
        echo '<thead><tr>
            <th>' . esc_html__( 'Method ID', 'buy-now-popup-instant-checkout-lite-for-woocommerce' ) . '</th>
            <th>' . esc_html__( 'Label', 'buy-now-popup-instant-checkout-lite-for-woocommerce' ) . '</th>
            <th>' . esc_html__( 'Cost', 'buy-now-popup-instant-checkout-lite-for-woocommerce' ) . '</th>
            <th>' . esc_html__( 'Free Shipping Threshold', 'buy-now-popup-instant-checkout-lite-for-woocommerce' ) . '</th>
            <th>' . esc_html__( 'Enable', 'buy-now-popup-instant-checkout-lite-for-woocommerce' ) . '</th>
        </tr></thead><tbody>';

        foreach ( $zones as $zone ) {
            foreach ( $zone['shipping_methods'] as $method ) {
                $key = $method->id . $method->instance_id;
                $saved_data = $saved[ $key ] ?? [];
                if ( $key === 'free_shipping1' ) continue;
                echo '<tr>';
                echo '<td>' . esc_html( $key ) . '</td>';
                echo '<td><input type="text" name="wcbnpc_enabled_shipping_methods[' . esc_attr( $key ) . '][label]" value="' . esc_attr( $saved_data['label'] ?? $method->get_title() ) . '" /></td>';
                echo '<td><input type="number" step="0.01" name="wcbnpc_enabled_shipping_methods[' . esc_attr( $key ) . '][cost]" value="' . esc_attr( $saved_data['cost'] ?? '' ) . '" /></td>';
                echo '<td><input type="number" step="0.01" name="wcbnpc_enabled_shipping_methods[' . esc_attr( $key ) . '][free_shipping_threshold]" value="' . esc_attr( $saved_data['free_shipping_threshold'] ?? '' ) . '" /></td>';
                echo '<td>
                    <input type="hidden" name="wcbnpc_enabled_shipping_methods[' . esc_attr( $key ) . '][enabled]" value="0" />
                    <input type="checkbox" name="wcbnpc_enabled_shipping_methods[' . esc_attr( $key ) . '][enabled]" value="1" ' . checked( ! empty( $saved_data['enabled'] ), true, false ) . ' />
                </td>';
                echo '</tr>';
            }
        }

        echo '</tbody></table>';
    }

    public function render_color_picker_field( $args ) {
        $options = get_option( 'wcbnpc_settings', [] );
        $value = isset( $options[ $args['option_key'] ] ) ? esc_attr( $options[ $args['option_key'] ] ) : '';
        echo '<input type="text" class="wcbnpc-color-picker" id="' . esc_attr( $args['label_for'] ) . '" name="wcbnpc_settings[' . esc_attr( $args['option_key'] ) . ']" value="' . esc_attr( $value ) . '" />';
        if ( ! empty( $args['description'] ) ) {
            echo '<p class="description">' . esc_html( $args['description'] ) . '</p>';
        }
    }

    public function sanitize_shipping_methods( $input ) {
        $sanitized = [];

        if ( ! is_array( $input ) ) {
            return $sanitized;
        }

        foreach ( $input as $key => $data ) {
            if ( empty( $data['enabled'] ) ) {
                continue; // skip unchecked
            }

            $sanitized[ sanitize_key( $key ) ] = [
                'label' => sanitize_text_field( $data['label'] ?? '' ),
                'cost' => is_numeric( $data['cost'] ) ? floatval( $data['cost'] ) : 0,
                'free_shipping_threshold' => is_numeric( $data['free_shipping_threshold'] ) ? floatval( $data['free_shipping_threshold'] ) : 0,
                'enabled' => true,
            ];
        }

        return $sanitized;
    }

    public function maybe_reset_settings() {
        if ( isset( $_POST['wcbnpc_reset_settings'] ) ) {
            $nonce = isset( $_POST['wcbnpc_reset_settings_nonce'] ) ? sanitize_text_field( wp_unslash( $_POST['wcbnpc_reset_settings_nonce'] ) ) : '';
            if ( ! wp_verify_nonce( $nonce, 'wcbnpc_reset_settings_action' ) ) {
                wp_die(
                    esc_html__( 'Security check failed. Please refresh the page and try again.', 'buy-now-popup-instant-checkout-lite-for-woocommerce' ),
                    [ 'back_link' => true ]
                );
            }

            delete_option( 'wcbnpc_settings' );
            delete_option( 'wcbnpc_checkout_fields' );
            delete_option( 'wcbnpc_enabled_payment_gateways' );
            delete_option( 'wcbnpc_enabled_shipping_methods' );
            wp_safe_redirect( add_query_arg(
                [
                    'page'        => 'wcbnpc-settings',
                    'reset'       => '1',
                    'reset_nonce' => wp_create_nonce( 'wcbnpc_reset_action' ),
                ],
                admin_url( 'admin.php' )
            ) );
            exit;
        }
    }

    public static function render_remote_notices() {
        if (!current_user_can('manage_options')) return;
        $nonce = wp_create_nonce('wcbnpc_dismiss_notice');
        $dismissed = get_user_meta(get_current_user_id(), 'wcbnpc_dismissed_notices', true);
        if (!is_array($dismissed)) $dismissed = [];

        $response = wp_remote_get('https://wizewp.com/info/notices.json', ['timeout' => 5]);
        if (is_wp_error($response)) return;

        $notices = json_decode(wp_remote_retrieve_body($response), true);
        if (empty($notices) || !is_array($notices)) return;

        foreach ($notices as $notice) {
            if (empty($notice['id']) || in_array($notice['id'], $dismissed)) continue;
            ?>
            <div class="notice notice-info is-dismissible wcbnpc-remote-notice" data-notice-id="<?php echo esc_attr($notice['id']); ?>">
                <p><strong><?php echo esc_html($notice['title']); ?></strong></p>
                <p><?php echo wp_kses_post($notice['message']); ?>
                    <?php if (!empty($notice['link_url']) && !empty($notice['link_text'])): ?>
                        <a href="<?php echo esc_url($notice['link_url']); ?>" target="_blank" class="button button-primary">
                            <?php echo esc_html($notice['link_text']); ?>
                        </a>
                    <?php endif; ?>
                </p>
            </div>
            <?php
        }
    }
    

    public static function ajax_dismiss_notice() {
        $nonce = isset($_POST['_ajax_nonce']) ? sanitize_text_field(wp_unslash($_POST['_ajax_nonce'])) : '';
        if (!wp_verify_nonce($nonce, 'wcbnpc_dismiss_notice')) {
            wp_send_json_error(['message' => 'Nonce check failed']);
            wp_die();
        }
        $notice_id = isset($_POST['notice_id']) ? sanitize_text_field(wp_unslash($_POST['notice_id'])) : '';
        if (!$notice_id) {
            wp_send_json_error(['message' => 'Invalid notice ID']);
            wp_die();
        }
        $dismissed = get_user_meta(get_current_user_id(), 'wcbnpc_dismissed_notices', true);
        if (!is_array($dismissed)) $dismissed = [];
        $dismissed[] = $notice_id;
        $dismissed = array_unique($dismissed);
        update_user_meta(get_current_user_id(), 'wcbnpc_dismissed_notices', $dismissed);

        wp_send_json_success();
    }
}

new WCBNPCL_Settings();