<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Handles frontend functionality for the Buy Now Popup Instant Checkout for WooCommerce plugin.
 */
class WCBNPCL_Frontend {

    /** @var array Settings retrieved from the database */
    private $settings;

    /**
     * Constructor to initialize hooks and settings.
     */
    public function __construct() {
        $this->settings = get_option( 'wcbnpc_settings', [] );
        $position       = $this->settings['wcbnpc_button_position'] ?? 'woocommerce_after_add_to_cart_button';
        $shop_page      = isset( $this->settings['wcbnpc_button_enable_shop_page'] ) && $this->settings['wcbnpc_button_enable_shop_page'] === '1';

        add_action( 'wp_enqueue_scripts', [ $this, 'enqueue_assets' ] );
        add_action( $position, [ $this, 'render_buy_now_button' ] );
        if ( $shop_page ) {
            if ( function_exists( 'wp_is_block_theme' ) && wp_is_block_theme() ){
                add_filter( 'render_block', [ $this, 'render_buy_now_button_in_blocks' ], 10, 2 );
            }
            else{
                add_action( 'woocommerce_after_shop_loop_item', [ $this, 'render_buy_now_button_shop_page' ], 12 );
            }
        }
        add_action( 'wp_footer', [ $this, 'render_modal_container' ] );
        add_action( 'wp_ajax_wcbnpc_get_product_data', [ $this, 'ajax_get_product_data' ] );
        add_action( 'wp_ajax_nopriv_wcbnpc_get_product_data', [ $this, 'ajax_get_product_data' ] );
        add_shortcode( 'wcbnpc_buy_now', [ $this, 'shortcode_buy_now_button' ] );
    }

    /**
     * Enqueues frontend assets (styles and scripts).
     */
    public function enqueue_assets() {
        // Enqueue WooCommerce dependencies
        wp_enqueue_style( 'select2' );
        wp_enqueue_script( 'selectWoo' );
        wp_enqueue_script( 'wc-country-select' );

        // Enqueue custom assets
        wp_enqueue_style(
            'wcbnpc-frontend-style',
            WCBNPCL_ASSETS_URL . 'frontend/css/buy-now.css',
            [],
            WCBNPCL_VERSION
        );

        wp_enqueue_script(
            'wcbnpc-frontend-script',
            WCBNPCL_ASSETS_URL . 'frontend/js/buy-now.js',
            [ 'jquery', 'selectWoo', 'wc-country-select' ],
            WCBNPCL_VERSION,
            true
        );

        // Localize script with AJAX URL and variables
        wp_localize_script( 'wcbnpc-frontend-script', 'wcbnpc_ajax', [
            'ajax_url' => admin_url( 'admin-ajax.php' ),
        ]);

        wp_localize_script( 'wcbnpc-frontend-script', 'wcbnpc_vars', [
            'formatPrice'       => function($price) {
                return wp_strip_all_tags( wp_kses_decode_entities( wc_price( $price ) ) );
                },
            'decimalSeparator'  => wc_get_price_decimal_separator(),
            'thousandSeparator' => wc_get_price_thousand_separator(),
            'currencySymbol'    => get_woocommerce_currency_symbol(),
            'currencyPosition'  => get_option( 'woocommerce_currency_pos' ),
            'priceDecimals'     => wc_get_price_decimals(),
            'nonce'             => wp_create_nonce( 'wcbnpc_get_product_data' ),
            'nonce_submit'      => wp_create_nonce( 'wcbnpc_submit_order' ),
            'freeShippingText'  => __( 'Free Shipping', 'buy-now-popup-instant-checkout-lite-for-woocommerce' ),
        ]);
    }

    /**
     * Renders the "Buy Now" button via shortcode.
     *
     * @param array  $atts Shortcode attributes.
     * @return string HTML button or empty string if invalid.
     */
    public function shortcode_buy_now_button( $atts ) {
        $atts = shortcode_atts( [
            'id'    => 0,
        ], $atts, 'wcbnpc_buy_now' );

        $product_id = intval( $atts['id'] );
        if ( $product_id <= 0 ) {
            return '';
        }

        $product = wc_get_product( $product_id );
        if ( ! $product ) {
            return '';
        }
        $settings = $this->settings;
        $button_color = esc_attr( $settings['wcbnpc_button_color'] ?? '#0071a1' );
        $style = sprintf( 'background-color:%s', $button_color );
        $button_text = ! empty( $this->settings['wcbnpc_button_text'] ) ? esc_html( $this->settings['wcbnpc_button_text'] ) : __( 'Buy Now', 'buy-now-popup-instant-checkout-lite-for-woocommerce' );
        if(wp_is_block_theme()){
            return sprintf(
            '<div class="wp-block-button wc-block-components-product-button"><button type="button" class="wcbnpc-buy-now button alt wp-block-button__link wp-element-button wc-block-components-product-button__button add_to_cart_button" style="%s" data-product_id="%d">%s</button></div>',
            esc_attr( $style ),
            esc_attr( $product_id ),
            $button_text
        );
        }
        else{
            return sprintf(
            '<button type="button" class="wcbnpc-buy-now button alt %s" style="%s" data-product_id="%d">%s</button>',
            esc_attr( $atts['class'] ),
            esc_attr( $style ),
            esc_attr( $product_id ),
            $button_text
        );
        }
    }

    /**
     * Renders the "Buy Now" button in woocommerce blocks
     */
    public function render_buy_now_button_in_blocks( $content, $block){
        if ( is_admin() ) return $content;
        if ( empty( $block['blockName'] ) ) return $content;
        if ( 'woocommerce/product-button' !== $block['blockName'] ) return $content;
        $product_id = get_the_ID();
        $product    = $product_id ? wc_get_product( $product_id ) : null;
        if ( ! $product || ! $product->is_type( 'simple' ) ) return $content;
        if ( get_post_meta( $product_id, '_wcbnpc_disable_buy_now_shop_page', true ) === 'yes' ) {
            return $content;
        }
        $button_text = ! empty( $this->settings['wcbnpc_button_text'] ) ? esc_html( $this->settings['wcbnpc_button_text'] ) : __( 'Buy Now', 'buy-now-popup-instant-checkout-lite-for-woocommerce' );
        $settings = $this->settings;
        $button_color = esc_attr( $settings['wcbnpc_button_color'] ?? '#0071a1' );
        $style = sprintf( 'background-color:%s', $button_color );
        $btn  = sprintf(
        '<div class="wp-block-button wc-block-components-product-button"><button type="button" class="wp-block-button wc-block-components-product-button wc-block-components-product-button__button button wcbnpc-buy-now" style="%s" data-product_id="%d">%s</button></div>',
            esc_attr( $style ),
            esc_attr( $product->get_id() ),
            esc_html( $button_text )
        );
        return $content . $btn;
    }

    /**
     * Renders the "Buy Now" button on shop pages.
     */
    public function render_buy_now_button_shop_page() {
        global $product;
        if ( ! $product || ! $product->is_type( 'simple' ) ) {
            return;
        }

        if ( empty( $this->settings['wcbnpc_enable'] ) ) {
            return;
        }

        $product_id = $product->get_id();
        if ( get_post_meta( $product_id, '_wcbnpc_disable_buy_now_shop_page', true ) === 'yes' ) {
            return;
        }
        $settings = $this->settings;
        $button_color = esc_attr( $settings['wcbnpc_button_color'] ?? '#0071a1' );
        $style = sprintf( 'background-color:%s', $button_color );
        $button_text = ! empty( $this->settings['wcbnpc_button_text'] ) ? esc_html( $this->settings['wcbnpc_button_text'] ) : __( 'Buy Now', 'buy-now-popup-instant-checkout-lite-for-woocommerce' );
        printf(
            '<button type="button" class="wcbnpc-buy-now button alt" style="%s" data-product_id="%d">%s</button>',
            esc_attr( $style ),
            esc_attr( $product_id ),
            esc_html($button_text)
        );
    }

    /**
     * Renders the "Buy Now" button on product pages.
     */
    public function render_buy_now_button() {
        global $product;
        if ( ! $product || empty( $this->settings['wcbnpc_enable'] ) ) {
            return;
        }
        $settings = $this->settings;
        $button_color = esc_attr( $settings['wcbnpc_button_color'] ?? '#0071a1' );
        $style = sprintf( 'background-color:%s', $button_color );
        $product_id = $product->get_id();
        if ( get_post_meta( $product_id, '_wcbnpc_disable_buy_now', true ) === 'yes' ) {
            return;
        }

        $button_text = ! empty( $this->settings['wcbnpc_button_text'] ) ? esc_html( $this->settings['wcbnpc_button_text'] ) : __( 'Buy Now', 'buy-now-popup-instant-checkout-lite-for-woocommerce' );
        $variation_data = $product->is_type( 'variable' ) ? ' data-variation_id="" data-attributes="{}"' : '';

        if(wp_is_block_theme()){
            printf(
            '<button type="button" class="wcbnpc-buy-now button alt wp-element-button" style="%s" data-product_id="%d"%s>%s</button>',
            esc_attr( $style ),
            esc_attr( $product_id ),
            esc_attr($variation_data),
            esc_html($button_text)
        );
        }
        else{
            printf(
            '<button type="button" class="wcbnpc-buy-now button alt" style="%s" data-product_id="%d"%s>%s</button>',
            esc_attr( $style ),
            esc_attr( $product_id ),
            esc_attr($variation_data),
            esc_html($button_text)
            );
        }
    }

    /**
     * Renders the modal container in the footer.
     */
    public function render_modal_container() {
        printf(
            '<div id="wcbnpc-modal" class="wcbnpc-modal" style="display:none;">
                <div class="wcbnpc-modal-content">
                    <span class="wcbnpc-close">&times;</span>
                    <div id="wcbnpc-modal-body"></div>
                </div>
             </div>'
        );
    }

    /**
     * Handles AJAX request to fetch product data and generate checkout form.
     */
    public function ajax_get_product_data() {
        // Verify nonce
        if ( ! isset( $_POST['_wpnonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['_wpnonce'] ) ), 'wcbnpc_get_product_data' ) ) {
            wp_send_json_error( [ 'message' => __( 'Security check failed.', 'buy-now-popup-instant-checkout-lite-for-woocommerce' ) ] );
            return;
        }

        // Validate product ID
        if ( empty( $_POST['product_id'] ) ) {
            wp_send_json_error( [ 'message' => __( 'Missing product ID.', 'buy-now-popup-instant-checkout-lite-for-woocommerce' ) ] );
            return;
        }

        $product_id   = absint( $_POST['product_id'] );
        $variation_id = isset( $_POST['variation_id'] ) ? absint( $_POST['variation_id'] ) : 0;
        $product      = wc_get_product( $variation_id > 0 ? $variation_id : $product_id );
        if ( ! $product ) {
            wp_send_json_error( [ 'message' => __( 'Invalid product.', 'buy-now-popup-instant-checkout-lite-for-woocommerce' ) ] );
            return;
        }

        ob_start();
        $settings           = $this->settings;
        $thumbnail          = $product->get_image( 'woocommerce_thumbnail' );
        $title              = $product->get_name();
        $price_html         = $product->get_price_html();
        $regular_price      = $product->get_regular_price();
        $sale_price         = $product->get_sale_price();
        $qty                = 1;
        $all_fields         = get_option( 'wcbnpc_checkout_fields', [] );
        $field_order        = $settings['field_order'] ?? []; // Expecting array from settings
        $ordered_keys       = is_array( $field_order ) ? $field_order : array_filter( array_map( 'trim', explode( ',', (string) $field_order ) ) );
        $fields             = [];

        // Build ordered fields
        foreach ( $ordered_keys as $key ) {
            if ( isset( $all_fields[ $key ] ) ) {
                $fields[ $key ] = $all_fields[ $key ];
            }
        }
        foreach ( $all_fields as $key => $field ) {
            if ( ! isset( $fields[ $key ] ) ) {
                $fields[ $key ] = $field;
            }
        }
        $show_badge         = $settings['wcbnpc_show_badge'] ?? '';
        $badge_type         = $settings['wcbnpc_badge_type'] ?? 'text';
        $badge_text         = $settings['wcbnpc_badge_text'] ?? __( 'SALE!', 'buy-now-popup-instant-checkout-lite-for-woocommerce' );
        $badge_color        = $settings['wcbnpc_badge_color'] ?? '#dd3333';
        $title_billing      = $settings['wcbnpc_section_title_billing'] ?? __( 'Billing Information', 'buy-now-popup-instant-checkout-lite-for-woocommerce' );
        $title_shipping     = $settings['wcbnpc_section_title_shipping'] ?? __( 'Shipping Information', 'buy-now-popup-instant-checkout-lite-for-woocommerce' );
        $title_payment      = $settings['wcbnpc_section_title_payment_method'] ?? __( 'Payment Method', 'buy-now-popup-instant-checkout-lite-for-woocommerce' );
        $title_shipping_methods = $settings['wcbnpc_section_title_shipping_method'] ?? __( 'Shipping Method', 'buy-now-popup-instant-checkout-lite-for-woocommerce' );
        $free_text          = $settings['wcbnpc_free_shipping_text'] ?? __( 'You qualify for free shipping!', 'buy-now-popup-instant-checkout-lite-for-woocommerce' );
        $bg_color           = $settings['wcbnpc_free_shipping_bg'] ?? '#e1f7e7';
        $text_color         = $settings['wcbnpc_free_shipping_color'] ?? '#2d7d46';
        $shipping_methods   = get_option( 'wcbnpc_enabled_shipping_methods', [] );
        $shipping_label     = __( 'Delivery', 'buy-now-popup-instant-checkout-lite-for-woocommerce' );
        if ( ! empty( $shipping_methods ) ) {
            $shipping_label = esc_html( reset( $shipping_methods )['label'] ?? $shipping_label );
        }
        echo '<form id="wcbnpc-checkout-form">';
        echo '<div class="wcbnpc-product-summary" 
            data-product_id="' . esc_attr( $product->get_id() ) . '" 
            data-variation_id="' . esc_attr( $variation_id ) . '" 
            data-quantity="' . esc_attr( $qty ) . '" 
            data-price="' . esc_attr( $product->get_price() ) . '">';
        echo '<div class="wcbnpc-product-image">';
        if(!empty($regular_price) && !empty($sale_price) && ($regular_price > $sale_price)){
            if($show_badge == 1){
            $style = sprintf( 'background-color:%s', $badge_color );
            if($badge_type == 'text'){
                printf(
                '<span class="wcbnpc-badge" style="%s">%s</span>',
                esc_attr( $style ),
                esc_html($badge_text)
                );
            }
            }
        }
        echo wp_kses_post( $thumbnail );
        echo '</div>';
        echo '<div class="wcbnpc-product-info"><h3>' . esc_html( $title ) . '</h3><div class="wcbnpc-product-price">' . wp_kses_post( $price_html ) . ' <span>&times; ' . esc_html( $qty ) . '</span></div></div>';
        echo '</div>';
        echo '<div id="wcbnpc-order-summary" class="wcbnpc-order-summary">';
        echo '<div class="wcbnpc-summary-row">';
        echo '<span class="label">' . esc_html__( 'Subtotal', 'buy-now-popup-instant-checkout-lite-for-woocommerce' ) . '</span>';
        echo '<span class="value" id="wcbnpc-subtotal">–</span>';
        echo '</div>';
        echo '<div class="wcbnpc-summary-row">';
        echo '<span class="label">' . esc_html( $shipping_label ) . '</span>';
        echo '<span class="value" id="wcbnpc-shipping">–</span>';
        echo '</div>';
        echo '<div class="wcbnpc-summary-row">';
        echo '<span class="label"><strong>' . esc_html__( 'Total', 'buy-now-popup-instant-checkout-lite-for-woocommerce' ) . '</strong></span>';
        echo '<span class="value" id="wcbnpc-total"><strong>–</strong></span>';
        echo '</div>';
        echo '</div>';
        echo '<div id="wcbnpc-free-shipping-message" style="display:none; margin-bottom:5px; padding:10px; text-align:center; font-weight:600; border-radius:6px; background:' . esc_attr( $bg_color ) . '; color:' . esc_attr( $text_color ) . '">' . esc_html( $free_text ) . '</div>';
        echo '<div class="wcbnpc-form-fields">';
        $allowed_html = array(
            'select' => array(
                'name' => array(),
                'id' => array(),
                'class' => array(),
                'autocomplete' => array(),
                'data-label' => array(),
            ),
            'option' => array(
                'value' => array(),
                'selected' => array(),
            ),
        );
        // Render fields with a callback function
        $render_field = function( $key, $field, $extra_class = '' ) use ( $fields, $allowed_html ) {
            if ( empty( $field['visible'] ) ) {
                return;
            }

            $label    = esc_html( $field['label'] ?? $key );
            $required = ! empty( $field['required'] ) ? 'required' : '';
            $is_shipping = strpos( $key, 'shipping_' ) === 0;
            $width    = isset( $field['width'] ) ? intval( $field['width'] ) : 12;
            $extra_class = 'wcbnpc-col-' . $width . ( $is_shipping ? ' wcbnpc-shipping-field' : '' );
            $style    = $is_shipping ? 'style="display:none;"' : '';

            printf(
                '<div class="form-row form-row-wide wcbnpc-row %s" %s>',
                esc_attr( $extra_class ),
                esc_attr($style)
            );

            if ( $key === 'billing_country' ) {
                printf(
                    '<select name="billing_country" id="billing_country" class="country_to_state country_select" autocomplete="country" data-label="Country">
                        <option value="">%s</option>%s</select>',
                    esc_html__( 'Select a country', 'buy-now-popup-instant-checkout-lite-for-woocommerce' ),
                    wp_kses($this->get_country_options(), $allowed_html)
                );
            } elseif ( $key === 'shipping_country' ) {
                printf(
                    '<select name="shipping_country" id="shipping_country" class="country_to_state country_select" autocomplete="country" data-label="Country">
                        <option value="">%s</option>%s</select>',
                    esc_html__( 'Select a country', 'buy-now-popup-instant-checkout-lite-for-woocommerce' ),
                    wp_kses($this->get_country_options(), $allowed_html)
                );
            } elseif ( $key === 'billing_state' ) {
                printf(
                    '<select name="billing_state" id="billing_state" class="state_select" data-input-classes="" autocomplete="address-level1">
                        <option value="">%s</option>
                    </select>',
                    esc_html__( 'Select a state', 'buy-now-popup-instant-checkout-lite-for-woocommerce' )
                );
            } elseif ( $key === 'shipping_state' ) {
                printf(
                    '<select name="shipping_state" id="shipping_state" class="state_select" data-input-classes="" autocomplete="address-level1">
                        <option value="">%s</option>
                    </select>',
                    esc_html__( 'Select a state', 'buy-now-popup-instant-checkout-lite-for-woocommerce' )
                );
            } elseif ( $key === 'selected_location' ) {
                printf(
                    '<label class="wcbnpc-checkbox"><input type="checkbox" name="%s" value="1" class="wcbnpc-toggle-shipping" /> %s</label>',
                    esc_attr( $key ),
                    esc_attr( $label )
                );
            } else {
                printf(
                    '<input type="text" name="%s" placeholder="%s" %s />',
                    esc_attr( $key ),
                    esc_attr( $label ),
                    esc_attr($required)
                );
            }
            echo '</div>';
        };

        $rendered_billing  = false;
        $rendered_shipping = false;
        foreach ( $fields as $key => $field ) {
            if ( strpos( $key, 'billing_' ) === 0 && ! $rendered_billing ) {
                echo '<h3 class="wcbnpc-title">' . esc_html( $title_billing ) . '</h3>';
                $rendered_billing = true;
            }
            if ( strpos( $key, 'shipping_' ) === 0 && ! $rendered_shipping ) {
                echo '<h3 class="wcbnpc-title wcbnpc-shipping-field">' . esc_html( $title_shipping ) . '</h3>';
                $rendered_shipping = true;
            }
            $render_field( $key, $field );
        }
        // Shipping methods
        $method_count = count( $shipping_methods );
        if (!empty($shipping_methods)) {
            if ( $method_count < 2 ) {
                echo '<div style="display:none;">';
            }
        echo '<h3>' . esc_html($title_shipping_methods) . '</h3>';
        foreach ($shipping_methods as $method_id => $method) {
            $free_shipping_threshold = floatval($method['free_shipping_threshold'] ?? 0);
            $method_cost = floatval($method['cost'] ?? 0);
            $initial_cost = ($free_shipping_threshold > 0 && $product->get_price() >= $free_shipping_threshold) ? 0 : $method_cost;
            printf(
            '<label style="display:block; margin-bottom: 8px;" class="wcbnpc-shipping-option">
                <input type="radio" name="shipping_method" value="%s" data-threshold="%s" data-cost="%s" data-calculated-cost="%s" required %s>
                <span class="wcbnpc-shipping-label" data-original-title="%s">%s - <span class="woocommerce-Price-amount"><bdi>%s</bdi></span></span>
            </label>',
            esc_attr($method_id),
            esc_attr($free_shipping_threshold),
            esc_attr($method_cost),
            esc_attr($initial_cost),
            count($shipping_methods) === 1 ? 'checked="checked"' : '',
            esc_attr($method['label']),
            esc_html($method['label']),
            wp_kses_post(wc_price($initial_cost))
            );
        }
            if ( $method_count < 2 ) {
                echo '</div>';
            }
        }
        // Payment methods
        $payment_gateways = WC()->payment_gateways()->get_available_payment_gateways();
        $enabled_gateways = get_option( 'wcbnpc_enabled_payment_gateways', [] );
        if ( ! empty( $enabled_gateways ) && count( $enabled_gateways ) > 1 ) {
            echo '<h3>' . esc_html( $title_payment ) . '</h3>';
            foreach ( $payment_gateways as $gateway_id => $gateway ) {
                if ( in_array( $gateway_id, $enabled_gateways, true ) ) {
                    printf(
                        '<label style="display:block; margin-bottom: 8px;">
                            <input type="radio" name="payment_method" value="%s" required %s>
                            %s
                        </label>',
                        esc_attr( $gateway_id ),
                        count( $enabled_gateways ) === 1 ? 'checked="checked"' : '',
                        esc_html( $gateway->get_title() )
                    );
                }
            }
        }

        echo '</div>'; // form fields
        $button_color = esc_attr( $settings['wcbnpc_button_color'] ?? '#0071a1' );
        $place_order_text = esc_html( $settings['wcbnpc_place_order_text'] ?? __( 'Place Order', 'buy-now-popup-instant-checkout-lite-for-woocommerce' ) );
        $style = sprintf( 'width:100%%; background-color:%s', $button_color );
        printf(
            '<button type="submit" class="button alt" style="%s">%s</button>',
            esc_attr( $style ),
            esc_html($place_order_text)
        );
        echo '</form>';

        $html = ob_get_clean();
        wp_send_json_success( [ 'html' => $html ] );
    }

    /**
     * Generates HTML options for country select fields.
     *
     * @return string HTML option tags.
     */
    private function get_country_options() {
        $options = '';
        foreach ( WC()->countries->get_allowed_countries() as $code => $country_name ) {
            $options .= sprintf( '<option value="%s">%s</option>', esc_attr( $code ), esc_html( $country_name ) );
        }
        return $options;
    }
}

new WCBNPCL_Frontend();