<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class WCBNPCL_Ajax_Handler {

    public function __construct() {
        add_action( 'wp_ajax_wcbnpc_submit_order', [ $this, 'submit_order' ] );
        add_action( 'wp_ajax_nopriv_wcbnpc_submit_order', [ $this, 'submit_order' ] );
    }

    public function submit_order() {
        // Verify nonce
        if ( ! isset( $_POST['_wpnonce_submit'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['_wpnonce_submit'] ) ), 'wcbnpc_submit_order' ) ) {
            wp_send_json_error( [ 'message' => __( 'Security check failed.', 'buy-now-popup-instant-checkout-lite-for-woocommerce' ) ] );
        }

        // Validate form data
        if ( ! isset( $_POST['fields'] ) || ! is_array( $_POST['fields'] ) ) {
            wp_send_json_error( [ 'message' => __( 'Invalid form data.', 'buy-now-popup-instant-checkout-lite-for-woocommerce' ) ] );
        }

        // Sanitize inputs
        $fields = array_map( 'sanitize_text_field', wp_unslash( $_POST['fields'] ) );
        $product_id   = absint( $_POST['product_id'] ?? 0 );
        $quantity     = absint( $_POST['quantity'] ?? 1 );
        $variation_id = absint( $_POST['variation_id'] ?? 0 );
        $attributes = isset( $_POST['attributes'] ) && is_array( $_POST['attributes'] ) ? array_map( 'sanitize_text_field', wp_unslash( $_POST['attributes'] ) ) : [];

        // Validate product and quantity
        if ( ! $product_id || $quantity <= 0 ) {
            wp_send_json_error( [ 'message' => __( 'Invalid product data.', 'buy-now-popup-instant-checkout-lite-for-woocommerce' ) ] );
        }

        // Create order
        $order = wc_create_order();
        $order->set_created_via( 'wcbnpc' );
        $order->set_customer_id( absint( $fields['user_id'] ?? 0 ) );
        $order->set_currency( get_woocommerce_currency() );
        $order->set_prices_include_tax( 'yes' === get_option( 'woocommerce_prices_include_tax' ) );

        // Set payment method
        $payment_method_id = sanitize_text_field( $fields['payment_method'] ?? '' );
        $gateways = WC()->payment_gateways()->get_available_payment_gateways();
        if ( ! isset( $gateways[ $payment_method_id ] ) ) {
            wp_send_json_error( [ 'message' => __( 'Invalid payment method.', 'buy-now-popup-instant-checkout-lite-for-woocommerce' ) ] );
        }
        $order->set_payment_method( $gateways[ $payment_method_id ] );

        // Add product to order
        if ( $variation_id > 0 ) {
            // Variable product
            $variation = wc_get_product( $variation_id );
            if ( ! $variation || ! $variation->is_type( 'variation' ) ) {
                wp_send_json_error( [ 'message' => __( 'Variation not found or invalid.', 'buy-now-popup-instant-checkout-lite-for-woocommerce' ) ] );
            }
            $order->add_product( $variation, $quantity, [ 'variation' => $attributes ] );
        } else {
            // Simple product
            $product = wc_get_product( $product_id );
            if ( ! $product ) {
                wp_send_json_error( [ 'message' => __( 'Product not found.', 'buy-now-popup-instant-checkout-lite-for-woocommerce' ) ] );
            }
            $order->add_product( $product, $quantity );
        }

        $order_id = $order->get_id();

        // Set billing and shipping addresses
        $billing = [
            'first_name' => $fields['billing_first_name'] ?? '',
            'last_name'  => $fields['billing_last_name'] ?? '',
            'email'      => $fields['billing_email'] ?? '',
            'phone'      => $fields['billing_phone'] ?? '',
            'address_1'  => $fields['billing_address_1'] ?? '',
            'address_2'  => $fields['billing_address_2'] ?? '',
            'city'       => $fields['billing_city'] ?? '',
            'state'      => $fields['billing_state'] ?? '',
            'postcode'   => $fields['billing_postcode'] ?? '',
            'country'    => $fields['billing_country'] ?? '',
        ];
        $shipping = [
            'first_name' => $fields['shipping_first_name'] ?? $fields['billing_first_name'],
            'last_name'  => $fields['shipping_last_name'] ?? $fields['billing_last_name'],
            'address_1'  => $fields['shipping_address_1'] ?? $fields['billing_address_1'],
            'address_2'  => $fields['shipping_address_2'] ?? $fields['billing_address_2'],
            'city'       => $fields['shipping_city'] ?? $fields['billing_city'],
            'state'      => $fields['shipping_state'] ?? $fields['billing_state'],
            'postcode'   => $fields['shipping_postcode'] ?? $fields['billing_postcode'],
            'country'    => $fields['shipping_country'] ?? $fields['billing_country'],
        ];

        $order->set_address( $billing, 'billing' );
        if ( ! empty( $shipping['first_name'] ) ) {
            $order->set_address( $shipping, 'shipping' );
        } else {
            $order->set_address( $billing, 'shipping' );
        }

        // Save custom fields
        $defined_fields = get_option( 'wcbnpc_checkout_fields', [] );
        foreach ( $defined_fields as $key => $field ) {
            if ( ! isset( $fields[ $key ] ) ) {
                continue;
            }

            $value = sanitize_text_field( $fields[ $key ] );

            if ( ! empty( $field['custom'] ) ) {
                $order->update_meta_data( '_wcbnpc_' . $key, $value );
            } elseif ( ! in_array( $key, array_keys( $billing ), true ) ) {
                $order->update_meta_data( $key, $value );
            }
        }

        // Set shipping method
        $shipping_methods = get_option( 'wcbnpc_enabled_shipping_methods', [] );
        $chosen_shipping = sanitize_text_field( $fields['shipping_method'] ?? '' );

        if ( isset( $shipping_methods[ $chosen_shipping ] ) ) {
            $method_data = $shipping_methods[ $chosen_shipping ];
            $label = $method_data['label'] ?? 'Shipping';
            $base_cost = floatval( $method_data['cost'] ?? 0 );
            $threshold = floatval( $method_data['free_shipping_threshold'] ?? 0 );

            // Calculate subtotal
            $order_items = $order->get_items();
            $subtotal = 0;
            foreach ( $order_items as $item ) {
                $subtotal += $item->get_total();
            }

            $cost = ( $threshold > 0 && $subtotal >= $threshold ) ? 0 : $base_cost;

            $item = new WC_Order_Item_Shipping();
            $item->set_method_title( $label );

            if ( strpos( $chosen_shipping, ':' ) !== false ) {
                list( $method_id, $instance_id ) = explode( ':', $chosen_shipping );
                $item->set_method_id( $method_id );
                $item->set_instance_id( $instance_id );
            } else {
                $item->set_method_id( $chosen_shipping );
            }

            $item->set_total( $cost );
            $order->add_item( $item );
        }

        // Calculate totals
        $order->calculate_totals();

        // Set order status based on payment method
        if ( $payment_method_id === 'cheque' || $payment_method_id === 'stripe' ) {
            $order->set_status( 'wc-pending' );
        } else {
            $order->set_status( 'wc-processing' );
        }
        $order->update_meta_data( '_wc_order_attribution_utm_source', 'Buy Now' );
        $order->save();

        // Trigger WooCommerce order processed action
        do_action( 'woocommerce_checkout_order_processed', $order_id, $_POST, $order );

        // Determine redirect URL
        $redirect_url = ( ($payment_method_id === 'cod') || ($payment_method_id === 'cheque')) ? $order->get_checkout_order_received_url() : $order->get_checkout_payment_url();
        // Send success response
        wp_send_json_success( [
            'message'      => __( 'Order created successfully.', 'buy-now-popup-instant-checkout-lite-for-woocommerce' ),
            'redirect_url' => $redirect_url,
        ] );
    }
}

new WCBNPCL_Ajax_Handler();