jQuery(document).ready(function ($) {
    function isValidEmail(email) {
        const pattern = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        return pattern.test(email);
    }

    function isValidPhone(phone) {
        const pattern = /^[0-9+\s().-]*$/;
        return pattern.test(phone);
    }

    function validateField($field) {
        const name = $field.attr('name');
        const value = $field.val();
        const required = $field.prop('required');

        if ($field.closest('.wcbnpc-row').is(':hidden')) {
            return true;
        }

        $field.removeClass('wcbnpc-error');
        $field.closest('.form-row, .wcbnpc-row').removeClass('wcbnpc-has-error');

        if (required && !value) {
            $field.closest('.form-row, .wcbnpc-row').addClass('wcbnpc-has-error');
            return false;
        }

        if (name.includes('email') && value && !isValidEmail(value)) {
            $field.closest('.form-row, .wcbnpc-row').addClass('wcbnpc-has-error');
            return false;
        }

        if (name.includes('phone') && value && !isValidPhone(value)) {
            $field.closest('.form-row, .wcbnpc-row').addClass('wcbnpc-has-error');
            return false;
        }

        return true;
    }

    function formatPrice(amount) {
        const decimals = parseInt(wcbnpc_vars.priceDecimals) || 2;
        const decSep = wcbnpc_vars.decimalSeparator || '.';
        const thouSep = wcbnpc_vars.thousandSeparator || ',';
        const symbol = wcbnpc_vars.currencySymbol || 'lei';
        const position = wcbnpc_vars.currencyPosition || 'right';
        let formatted = parseFloat(amount).toFixed(decimals).replace(/\B(?=(\d{3})+(?!\d))/g, thouSep).replace('.', decSep);

        switch (position) {
            case 'left': return symbol + formatted;
            case 'left_space': return symbol + ' ' + formatted;
            case 'right_space': return formatted + ' ' + symbol;
            default: return formatted + symbol;
        }
    }

    function getSubtotal() {
        const $productSummary = $('.wcbnpc-product-summary');
        const price = parseFloat($productSummary.data('price') || 0);
        const quantity = parseInt($productSummary.data('quantity') || 1);
        return price * quantity;
    }

    function updateOrderSummary(subtotal) {
        const $shippingOption = $('input[name="shipping_method"]:checked');
        const shippingCost = parseFloat($shippingOption.data('calculated-cost') || 0);
        const total = subtotal + shippingCost;

        $('#wcbnpc-subtotal').text(formatPrice(subtotal));
        $('#wcbnpc-shipping').text(formatPrice(shippingCost));
        $('#wcbnpc-total').text(formatPrice(total));
    }

    function applyFreeShippingDisplay(subtotal) {
        const $shippingOptions = $('input[name="shipping_method"]');
        let hasFreeShipping = false;

        $shippingOptions.each(function () {
            const $option = $(this);
            const threshold = parseFloat($option.data('threshold') || 0);
            const baseCost = parseFloat($option.data('cost') || 0);
            const $label = $option.closest('label').find('.wcbnpc-shipping-label');
            const $priceBdi = $label.find('.woocommerce-Price-amount bdi');

            if (!$label.length || !$priceBdi.length) return;

            if (!$label.attr('data-original-title')) {
                const originalTitle = $label.text().split('-')[0].trim();
                $label.attr('data-original-title', originalTitle);
            }

            const calculatedCost = (threshold > 0 && subtotal >= threshold) ? 0 : baseCost;
            $option.data('calculated-cost', calculatedCost);
            const formattedCost = calculatedCost === 0 ? wcbnpc_vars.freeShippingText : formatPrice(calculatedCost);
            $priceBdi.text(formattedCost);

            if (calculatedCost === 0) {
                hasFreeShipping = true;
            }
        });

        const $selectedOption = $shippingOptions.filter(':checked');
        const isFreeShipping = $selectedOption.length && parseFloat($selectedOption.data('calculated-cost') || 0) === 0;
        $('#wcbnpc-free-shipping-message').toggle(!hasFreeShipping && !isFreeShipping).slideToggle(100);

        updateOrderSummary(subtotal);
    }

    $(document).on('select2:open select2:close', function () {
        $('#wcbnpc-modal .wcbnpc-modal-content').css('overflow', 'auto');
    });

    $(document).on('click', '.wcbnpc-buy-now', function (e) {
        e.preventDefault();

        const productId = $(this).data('product_id');
        let variationId = 0;
        let attributes = {};

        const $variationForm = $('.variations_form');
        if ($variationForm.length) {
            variationId = parseInt($variationForm.find('input[name="variation_id"]').val()) || 0;

            $variationForm.find('select[name^="attribute_"]').each(function () {
                attributes[$(this).attr('name')] = $(this).val();
            });

            if (!variationId) {
                alert('Selectează toate opțiunile produsului înainte de a continua.');
                return;
            }
        }

        window.wcbnpcSelectedVariation = { variation_id: variationId, attributes: attributes };

        $.post(wcbnpc_ajax.ajax_url, {
            action: 'wcbnpc_get_product_data',
            product_id: productId,
            _wpnonce: wcbnpc_vars.nonce,
            variation_id: variationId
        }, function (response) {
            if (response.success) {
                $('#wcbnpc-modal-body').html(response.data.html);
                $('#wcbnpc-modal').fadeIn(200);

                const subtotal = getSubtotal();
                applyFreeShippingDisplay(subtotal);
            } else {
                alert(response.data.message || 'Eroare la încărcarea produsului.');
            }
        }).fail(function () {
            alert('Eroare de conexiune la server.');
        });
    });

    $(document).on('select2:close', function () {
        $('body').css('overflow', 'hidden');
    });

    $(document).on('click', '.wcbnpc-close, #wcbnpc-modal', function (e) {
        if (e.target === this) {
            $('#wcbnpc-modal').fadeOut();
            $('body').css('overflow', '');
        }
    });

    $(document).on('change', '.wcbnpc-toggle-shipping', function () {
        $('.wcbnpc-shipping-field').slideToggle($(this).is(':checked'));
    });

    $(document).on('blur input change', '#wcbnpc-checkout-form input, #wcbnpc-checkout-form select', function () {
        validateField($(this));
    });

    $(document).on('click', '.wcbnpc-buy-now', function () {
        $('body').css('overflow', 'hidden');
    });

    $(document).on('submit', '#wcbnpc-checkout-form', function (e) {
        e.preventDefault();

        const $form = $(this);
        let isValid = true;

        const $hiddenRequired = $form.find(':input[required]').filter(function () {
            return $(this).closest('.wcbnpc-row').is(':hidden');
        });
        $hiddenRequired.each(function () {
            $(this).data('original-required', true).prop('required', false);
        });

        $form.find('input:visible, select:visible').each(function () {
            if (!validateField($(this))) {
                isValid = false;
            }
        });

        $hiddenRequired.each(function () {
            if ($(this).data('original-required')) {
                $(this).prop('required', true);
            }
        });

        if (!isValid) return;

        let formData = $form.serializeArray();
        let data = {
            action: 'wcbnpc_submit_order',
            _wpnonce_submit: wcbnpc_vars.nonce_submit,
            fields: {},
            attributes: {},
            variation_id: 0
        };

        formData.forEach(item => {
            data.fields[item.name] = item.value;
        });

        data.product_id = $('.wcbnpc-product-summary').data('product_id') || '';
        data.quantity = $('.wcbnpc-product-summary').data('quantity') || 1;

        if (window.wcbnpcSelectedVariation && window.wcbnpcSelectedVariation.variation_id) {
            data.variation_id = window.wcbnpcSelectedVariation.variation_id;
            data.attributes = window.wcbnpcSelectedVariation.attributes;
        }

        $.post(wcbnpc_ajax.ajax_url, data, function (response) {
            $form.find('button[type="submit"]').prop('disabled', false).text('Place Order');

            if (response.success && response.data.redirect_url) {
                window.location.href = response.data.redirect_url;
            } else {
                alert(response.data.message || 'A apărut o eroare.');
            }
        }).fail(function () {
            $form.find('button[type="submit"]').prop('disabled', false).text('Place Order');
            alert('Eroare de conexiune la server.');
        });
    });

    function updateStateDropdown($countrySelect) {
        const country = $countrySelect.val();
        const linkedStateId = $countrySelect.data('linked-state');
        const $stateSelect = $('#' + linkedStateId);

        if (!$stateSelect.length) return;

        const states = wc_country_select_params.countries[country] || {};
        $stateSelect.empty();

        if ($.isEmptyObject(states)) {
            $stateSelect.append('<option value="">' + wc_country_select_params.i18n_select_state_text + '</option>');
        } else {
            $stateSelect.append('<option value="">' + wc_country_select_params.i18n_select_state_text + '</option>');
            $.each(states, function (code, name) {
                $stateSelect.append('<option value="' + code + '">' + name + '</option>');
            });
        }

        $stateSelect.trigger('change');
    }

    $(document).on('change', '.country_to_state', function () {
        const $this = $(this);
        const country = $this.val();
        const linkedStateId = $this.data('linked-state');
        updateStateDropdown($this);
        if ($this.attr('id') === 'billing_country') {
            const $shippingCountry = $('#shipping_country');
            if ($shippingCountry.length) {
                $shippingCountry.val(country).trigger('change');
            }
        }
        setTimeout(() => {
            $('[name^="shipping_"]').each(function () {
                $(this).closest('.wcbnpc-row').hide();
            });
        }, 100);
    });

    $(document).on('change', '#billing_state', function () {
        const state = $(this).val();
        const $shippingState = $('#shipping_state');
        if ($shippingState.length) {
            $shippingState.val(state).trigger('change');
        }
    });

    $(document).on('change', 'input[name="shipping_method"]', function () {
        const subtotal = getSubtotal();
        applyFreeShippingDisplay(subtotal);
    });
});