<?php
/**
 * Plugin Name:         Bulk Upsell & Cross-Sell Editor
 * Plugin URI:          https://apusnest.com/woocommerce/plugins/bulk-upsell-cross-sell-editor
 * Description:         A plugin to bulk edit WooCommerce up-sells and cross-sells with custom discount offers.
 * Version:             1.0.0
 * Author:              ApusNest
 * Author URI:          https://apusnest.com/
 * License:             GPL v2 or later
 * Text Domain:         bulk-upsell-cross-sell-editor
 * Domain Path:         /languages
 * Requires at least: 6.0
 * WC requires at least: 5.0
 * WC tested up to: 8.9
 * Requires PHP: 7.4
 */

if (!defined('ABSPATH')) {
    exit;
}

// Declare HPOS compatibility
add_action('before_woocommerce_init', function () {
    if (class_exists(\Automattic\WooCommerce\Utilities\FeaturesUtil::class)) {
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility('custom_order_tables', __FILE__, true);
    }
});

// Ensure WooCommerce is active
if (!in_array('woocommerce/woocommerce.php', apply_filters('active_plugins', get_option('active_plugins')))) {
    return;
}

/**
 * Main plugin class to initialize backend and frontend components.
 */
final class BUCE_Plugin
{
    public function __construct()
    {
        if (is_admin()) {
            new BUCE_Admin_Editor();
        } else {
            new BUCE_Frontend_Handler();
        }
    }
}
new BUCE_Plugin();

/**
 * Handles the backend editor interface and AJAX functionality.
 */
class BUCE_Admin_Editor
{
    public function __construct()
    {
        add_action('admin_menu', [$this, 'add_admin_menu']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_scripts']);
        add_action('wp_ajax_buce_fetch_products', [$this, 'fetch_products_ajax_handler']);
        add_action('wp_ajax_buce_save_products', [$this, 'save_products_ajax_handler']);
        add_action('wp_ajax_buce_search_products', [$this, 'search_products_handler']);
    }

    public function add_admin_menu()
    {
        add_submenu_page(
            'woocommerce',
            __('Bulk Upsell & Cross-Sell Editor', 'bulk-upsell-cross-sell-editor'),
            __('Bulk Linked Products', 'bulk-upsell-cross-sell-editor'),
            'manage_woocommerce',
            'buce-editor',
            [$this, 'display_admin_page']
        );
    }

    public function enqueue_admin_scripts($hook)
    {
        if ('woocommerce_page_buce-editor' !== $hook) {
            return;
        }
        $plugin_url = plugin_dir_url(__FILE__);
        wp_enqueue_style('buce-admin-css', $plugin_url . 'admin/css/bulk-upsell-cross-sell-editor.css', [], '1.0.0');
        wp_enqueue_script('buce-admin-js', $plugin_url . 'admin/js/bulk-upsell-cross-sell-editor.js', ['jquery'], '1.0.0', true);
        wp_localize_script('buce-admin-js', 'buce_ajax', ['ajax_url' => admin_url('admin-ajax.php'), 'nonce' => wp_create_nonce('buce-nonce')]);
    }

    public function display_admin_page()
    {
        ?>
        <div class="wrap" id="buce-editor-wrap">
            <h1><?php esc_html_e('Bulk Upsell & Cross-Sell Editor', 'bulk-upsell-cross-sell-editor'); ?></h1>
            <p><?php esc_html_e('Type at least 3 letters of a product name to search and add it as a linked product.', 'bulk-upsell-cross-sell-editor'); ?></p>
            <div class="buce-controls">
                <input type="search" id="buce-search" placeholder="<?php esc_attr_e('Search by name or SKU...', 'bulk-upsell-cross-sell-editor'); ?>">
                <?php wp_dropdown_categories(['show_option_all' => __('All Categories', 'bulk-upsell-cross-sell-editor'), 'taxonomy' => 'product_cat', 'name' => 'buce-category-filter', 'id' => 'buce-category-filter', 'hierarchical' => true, 'value_field' => 'term_id']); ?>
                <label><input type="checkbox" id="buce-include-variations"/> <?php esc_html_e('Include Variations', 'bulk-upsell-cross-sell-editor'); ?></label>
                <button class="button" id="buce-filter-button"><?php esc_html_e('Filter', 'bulk-upsell-cross-sell-editor'); ?></button>
                <button class="button-primary" id="buce-save-button" disabled><?php esc_html_e('Save Changes', 'bulk-upsell-cross-sell-editor'); ?></button>
                <span class="spinner"></span>
            </div>
            <div id="buce-message-container"></div>
            <table class="wp-list-table widefat fixed striped">
                <thead>
                    <tr>
                        <th class="product-name"><?php esc_html_e('Product / Variation (SKU)', 'bulk-upsell-cross-sell-editor'); ?></th>
                        <th><?php esc_html_e('Up-sells', 'bulk-upsell-cross-sell-editor'); ?></th>
                        <th><?php esc_html_e('Cross-sells (with Discount %)', 'bulk-upsell-cross-sell-editor'); ?></th>
                    </tr>
                </thead>
                <tbody id="buce-product-list"></tbody>
            </table>
            <div class="buce-pagination" id="buce-pagination"></div>
        </div>
        <?php
    }

    public function search_products_handler()
    {
        check_ajax_referer('buce-nonce', 'nonce');
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(__('Unauthorized', 'bulk-upsell-cross-sell-editor'), 403);
        }

        $term = isset($_GET['term']) ? sanitize_text_field(wp_unslash($_GET['term'])) : '';
        if (strlen($term) < 3) {
            wp_send_json_error(__('Term too short', 'bulk-upsell-cross-sell-editor'));
        }

        $data_store = WC_Data_Store::load('product');
        $ids = array_unique(array_merge(
            $data_store->search_products($term, '', true, false, 10, [], true),
            $data_store->search_products($term, '', false, false, 10, [], true)
        ));

        $results = [];
        foreach ($ids as $id) {
            $product = wc_get_product($id);
            if (! $product) {
                continue;
            }
            $results[] = [
                'id'   => $product->get_id(),
                'name' => wp_strip_all_tags($product->get_formatted_name()),
                'sku'  => $product->get_sku() ? ' (SKU: ' . $product->get_sku() . ')' : '',
            ];
        }

        wp_send_json_success($results);
    }

    public function fetch_products_ajax_handler()
    {
        check_ajax_referer('buce-nonce', 'nonce');
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(__('Unauthorized', 'bulk-upsell-cross-sell-editor'), 403);
        }

        $paged = isset($_POST['page']) ? absint($_POST['page']) : 1;
        $include_variations = isset($_POST['include_variations']) && $_POST['include_variations'] === 'true';
        $search = isset($_POST['search']) ? sanitize_text_field(wp_unslash($_POST['search'])) : '';
        $cat = isset($_POST['category']) ? absint($_POST['category']) : 0;

        $args = [
            'post_type'      => $include_variations ? ['product', 'product_variation'] : ['product'],
            'posts_per_page' => 20,
            'paged'          => $paged,
            'post_status'    => 'publish',
            'orderby'        => 'title',
            'order'          => 'ASC',
        ];

        if (!empty($search)) {
            $args['s'] = $search;
        }

        // If category provided and we're only searching products (not variations), add a tax_query.
        // tax_query can be slow on very large catalogs; this is necessary for correct category filtering.
        if ($cat > 0 && ! $include_variations) {
            // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_tax_query -- Required for category filtering; documented and intentional.
            $args['tax_query'] = [
                [
                    'taxonomy' => 'product_cat',
                    'field'    => 'term_id',
                    'terms'    => $cat,
                ],
            ];
        }

        $query = new WP_Query($args);
        $html = '';

        if ($query->have_posts()) {
            while ($query->have_posts()) {
                $query->the_post();
                $product = wc_get_product(get_the_ID());
                if (! $product) {
                    continue;
                }

                $upsell_tags = $this->build_tags_html($product->get_upsell_ids(), 'upsell');

                $offers = get_post_meta($product->get_id(), '_custom_cross_sell_offers', true) ?: [];
                $std_ids = $product->get_cross_sell_ids();
                $offer_ids = is_array($offers) ? array_column($offers, 'product_id') : [];
                foreach (array_diff($std_ids, $offer_ids) as $id) {
                    $offers[] = ['product_id' => $id, 'discount_percent' => 0];
                }

                $cs_tags = $this->build_tags_html($offers, 'cross-sell');

                $html .= sprintf(
                    '<tr data-id="%d"><td>%s <br><small>SKU: %s</small></td><td><div class="buce-tag-container" data-type="upsell">%s<input type="text" placeholder="%s"></div></td><td><div class="buce-tag-container" data-type="cross-sell">%s<input type="text" placeholder="%s"></div></td></tr>',
                    esc_attr($product->get_id()),
                    esc_html(wp_strip_all_tags($product->get_formatted_name())),
                    esc_html($product->get_sku() ?: __('N/A', 'bulk-upsell-cross-sell-editor')),
                    $upsell_tags,
                    esc_attr__('Search to add up-sells...', 'bulk-upsell-cross-sell-editor'),
                    $cs_tags,
                    esc_attr__('Search to add cross-sells...', 'bulk-upsell-cross-sell-editor')
                );
            }
        } else {
            $html = '<tr><td colspan="3">' . esc_html__('No products found.', 'bulk-upsell-cross-sell-editor') . '</td></tr>';
        }
        wp_reset_postdata();

        $pagination = paginate_links([
            'base'     => add_query_arg('paged', '%#%'),
            'format'   => '',
            'current'  => max(1, $paged),
            'total'    => $query->max_num_pages,
            'prev_text'=> '«',
            'next_text'=> '»',
            'type'     => 'plain',
        ]);

        wp_send_json_success(['html' => $html, 'pagination' => $pagination]);
    }

    private function build_tags_html($items, $type)
    {
        if (empty($items)) {
            return '';
        }
        $html = '';
        foreach ($items as $item) {
            $id = is_array($item) ? (isset($item['product_id']) ? absint($item['product_id']) : 0) : absint($item);
            $product = wc_get_product($id);
            if (! $product) {
                continue;
            }

            $discount_html = '';
            if ($type === 'cross-sell') {
                $discount = isset($item['discount_percent']) ? absint($item['discount_percent']) : 0;
                /* translators: 1: "Discount" label, 2: Discount value, 3: "Discount percentage" title */
                $discount_html = sprintf(
                    '<span class="buce-discount-wrapper">(%1$s: <input type="number" class="buce-discount-input" value="%2$d" min="0" max="100" title="%3$s"> %%)</span>',
                    esc_html__('Discount', 'bulk-upsell-cross-sell-editor'),
                    esc_attr($discount),
                    esc_attr__('Discount percentage', 'bulk-upsell-cross-sell-editor')
                );
            }

            $html .= sprintf(
                '<span class="buce-tag" data-id="%d" data-name="%s">%s %s<button type="button" class="buce-remove-tag">×</button></span>',
                esc_attr($id),
                esc_attr(wp_strip_all_tags($product->get_formatted_name())),
                esc_html(wp_strip_all_tags($product->get_formatted_name())),
                $discount_html
            );
        }
        return $html;
    }

    /**
     * Recursively sanitizes the product data from the AJAX request.
     *
     * @param array $data The data to sanitize.
     * @return array The sanitized data.
     */
    private function buce_sanitize_products_data(array $data): array
    {
        $sanitized_data = [];
        foreach ($data as $key => $value) {
            $clean_key = sanitize_key((string) $key);
            if (is_array($value)) {
                $sanitized_data[$clean_key] = $this->buce_sanitize_products_data($value);
            } else {
                // Use sanitize_text_field for general text, numbers will be cast later.
                $sanitized_data[$clean_key] = sanitize_text_field(wp_unslash((string) $value));
            }
        }
        return $sanitized_data;
    }

    public function save_products_ajax_handler()
    {
        check_ajax_referer('buce-nonce', 'nonce');
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(__('Unauthorized', 'bulk-upsell-cross-sell-editor'), 403);
        }
        
        // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
        if (!isset($_POST['products']) || !is_array($_POST['products'])) {
            wp_send_json_error(['message' => __('Invalid data provided.', 'bulk-upsell-cross-sell-editor')]);
        }

        // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
        $products_data = $this->buce_sanitize_products_data(wp_unslash($_POST['products'])); // The data is sanitized in the buce_sanitize_products_data method, this is a false positive.

        foreach ($products_data as $data) {
            $id = isset($data['id']) ? absint($data['id']) : 0;
            if (! $id || ! in_array(get_post_type($id), ['product', 'product_variation'], true)) {
                continue;
            }

            $product = wc_get_product($id);
            if (! $product) {
                continue;
            }

            $upsell_ids = isset($data['upsells']) && is_array($data['upsells']) ? array_map('absint', $data['upsells']) : [];
            $product->set_upsell_ids($upsell_ids);

            $cross_sells = isset($data['cross_sells']) && is_array($data['cross_sells']) ? $data['cross_sells'] : [];
            $offers = [];
            $cs_ids = [];

            foreach ($cross_sells as $cs) {
                // be defensive: support both numeric arrays and associative arrays
                if (is_array($cs)) {
                    $cs_id = isset($cs['id']) ? absint($cs['id']) : (isset($cs['product_id']) ? absint($cs['product_id']) : 0);
                    $discount = isset($cs['discount']) ? intval($cs['discount']) : (isset($cs['discount_percent']) ? intval($cs['discount_percent']) : 0);
                } else {
                    $cs_id = absint($cs);
                    $discount = 0;
                }

                if ($cs_id > 0) {
                    $cs_ids[] = $cs_id;
                    $discount = max(0, min(100, $discount));
                    if ($discount > 0) {
                        $offers[] = [
                            'product_id'       => $cs_id,
                            'discount_percent' => $discount,
                        ];
                    }
                }
            }

            update_post_meta($id, '_custom_cross_sell_offers', $offers);
            $product->set_cross_sell_ids(array_unique($cs_ids));
            $product->save();
        }

        wp_send_json_success(['message' => __('Products updated successfully!', 'bulk-upsell-cross-sell-editor')]);
    }
}

/**
 * Handles the frontend display logic.
 */
class BUCE_Frontend_Handler
{
    private $offers = [];
    private $offer_triggers = [];
    private $data_gathered = false;

    public function __construct()
    {
        if (is_admin()) {
            return;
        }

        add_action('woocommerce_before_calculate_totals', [$this, 'calculate_discounts'], 10, 1);
        add_filter('woocommerce_product_get_price', [$this, 'apply_discount_to_price'], 20, 2);
        add_filter('woocommerce_product_variation_get_price', [$this, 'apply_discount_to_price'], 20, 2);
        add_filter('woocommerce_get_price_html', [$this, 'format_discounted_price_html'], 20, 2);
        add_filter('woocommerce_product_is_on_sale', [$this, 'product_is_on_sale_filter'], 20, 2);
        add_filter('woocommerce_sale_flash', [$this, 'custom_sale_flash_text_filter'], 20, 3);

        add_filter('woocommerce_product_get_sale_price', [$this, 'apply_discount_to_price'], 20, 2);
        add_filter('woocommerce_product_variation_get_sale_price', [$this, 'apply_discount_to_price'], 20, 2);

        add_action('woocommerce_before_add_to_cart_form', [$this, 'display_product_page_notice'], 10);
        add_action('woocommerce_cart_collaterals', [$this, 'display_custom_cross_sells']);
    }

    private function ensure_data_is_gathered()
    {
        if ($this->data_gathered || !function_exists('WC') || !WC()->cart) {
            return;
        }

        $this->offers = [];
        $this->offer_triggers = [];
        $cart = WC()->cart;

        if ($cart->is_empty()) {
            $this->data_gathered = true;
            return;
        }

        foreach ($cart->get_cart() as $cart_item) {
            $trigger_product_name = $cart_item['data']->get_name();
            $product_ids_to_check = [$cart_item['product_id']];
            if (! empty($cart_item['variation_id'])) {
                $product_ids_to_check[] = $cart_item['variation_id'];
            }

            foreach ($product_ids_to_check as $product_id) {
                $offers_from_meta = get_post_meta($product_id, '_custom_cross_sell_offers', true);
                if (!empty($offers_from_meta) && is_array($offers_from_meta)) {
                    foreach ($offers_from_meta as $offer) {
                        if (isset($offer['product_id'], $offer['discount_percent']) && $offer['discount_percent'] > 0) {
                            $target_id = absint($offer['product_id']);
                            $discount = absint($offer['discount_percent']);
                            $this->offers[$target_id] = max($this->offers[$target_id] ?? 0, $discount);
                            $this->offer_triggers[$target_id][] = $trigger_product_name;

                            $target_product = wc_get_product($target_id);
                            if ($target_product && $target_product->is_type('variable')) {
                                foreach ($target_product->get_children() as $child_id) {
                                    $this->offers[$child_id] = max($this->offers[$child_id] ?? 0, $discount);
                                    $this->offer_triggers[$child_id][] = $trigger_product_name;
                                }
                            }
                        }
                    }
                }
            }
        }

        $this->data_gathered = true;
    }

    public function calculate_discounts($cart)
    {
        // Reset data_gathered each time to ensure latest cart state is used.
        $this->data_gathered = false;
        $this->ensure_data_is_gathered();

        if (empty($this->offers)) {
            return;
        }

        foreach ($cart->get_cart() as $cart_item_key => $cart_item) {
            $id_to_check = !empty($cart_item['variation_id']) ? $cart_item['variation_id'] : $cart_item['product_id'];
            if (isset($this->offers[$id_to_check])) {
                $discount = $this->offers[$id_to_check];
                $regular_price = (float) $cart_item['data']->get_regular_price();
                $cart_item['data']->set_price($regular_price * (1 - ($discount / 100)));
            }
        }
    }

    public function display_custom_cross_sells()
    {   
        $this->data_gathered = false;
        $this->ensure_data_is_gathered();
        if (empty($this->offers)) {
            return;
        }

        $cs_ids = array_keys($this->offers);
        $cart_ids = [];
        foreach (WC()->cart->get_cart() as $item) {
            $cart_ids[] = $item['product_id'];
            if (! empty($item['variation_id'])) {
                $cart_ids[] = $item['variation_id'];
            }
        }

        $show_ids = array_diff(array_unique($cs_ids), array_unique($cart_ids));
        if (!empty($show_ids)) {
            wc_get_template('cart/cross-sells.php', ['cross_sells' => wc_get_products(['include' => $show_ids])]);
        }
    }

    public function apply_discount_to_price($price, $product)
    {
        $this->data_gathered = false;
        $this->ensure_data_is_gathered();
        $id = $product->get_id();
        if (isset($this->offers[$id])) {
            $regular_price = (float) $product->get_regular_price();
            if ($regular_price > 0) {
                return $regular_price * (1 - ($this->offers[$id] / 100));
            }
        }
        return $price;
    }

    public function format_discounted_price_html($price_html, $product)
    {
        $this->data_gathered = false;
        $this->ensure_data_is_gathered();
        $id = $product->get_id();
        if (isset($this->offers[$id])) {
            $sale_price = (float) $product->get_price();
            $regular_price = (float) $product->get_regular_price();
            if ($regular_price > $sale_price) {
                return wc_format_sale_price(
                    wc_get_price_to_display($product, ['price' => $regular_price]),
                    wc_get_price_to_display($product, ['price' => $sale_price])
                );
            }
        }
        return $price_html;
    }

    public function product_is_on_sale_filter($is_on_sale, $product)
    {
        $this->data_gathered = false;
        $this->ensure_data_is_gathered();
        return isset($this->offers[$product->get_id()]) ? true : $is_on_sale;
    }

    public function custom_sale_flash_text_filter($html, $post, $product)
    {
        $this->data_gathered = false;
        $this->ensure_data_is_gathered();
        $id = $product->get_id();
        if (isset($this->offers[$id])) {
            $tooltip = '';
            if (isset($this->offer_triggers[$id])) {
                $tooltip = sprintf(
                    /* translators: %s: a list of product names. */
                    esc_attr__('Discount applies when you also buy: %s', 'bulk-upsell-cross-sell-editor'),
                    implode(', ', array_unique($this->offer_triggers[$id]))
                );
            }
            /* translators: %d: Discount percentage (e.g., 15) */
            $text = sprintf(esc_html__('Extra %d%% off', 'bulk-upsell-cross-sell-editor'), (int) $this->offers[$id]);
            return sprintf('<span class="onsale cross-sell-discount-badge" title="%s">%s</span>', esc_attr($tooltip), esc_html($text));
        }
        return $html;
    }

    public function display_product_page_notice()
    {
        $this->data_gathered = false;
        $this->ensure_data_is_gathered();
        global $product;
        if (! $product) {
            return;
        }
        $id = $product->get_id();
        if (isset($this->offers[$id]) && isset($this->offer_triggers[$id])) {
            // Sanitize the trigger product names before imploding.
            $sanitized_triggers = array_map('esc_html', array_unique($this->offer_triggers[$id]));
            $names = '<strong>' . implode(', ', $sanitized_triggers) . '</strong>';

            $message = sprintf(
                /* translators: 1: Discount percentage (e.g., 15), 2: a list of product names. */
                __('Special Offer! Get %1$d%% off with %2$s in your cart.', 'bulk-upsell-cross-sell-editor'),
                absint($this->offers[$id]),
                $names
            );
            // wc_print_notice applies wp_kses_post, so the HTML in $names is safe.
            wc_print_notice($message, 'notice');
        }
    }
}
