jQuery(document).ready(function ($) {
    'use strict';

    const editor = {
        wrap: $('#buce-editor-wrap'),
        productList: $('#buce-product-list'),
        searchBox: $('#buce-search'),
        categoryFilter: $('#buce-category-filter'),
        variationsCheckbox: $('#buce-include-variations'),
        filterButton: $('#buce-filter-button'),
        saveButton: $('#buce-save-button'),
        paginationContainer: $('#buce-pagination'),
        messageContainer: $('#buce-message-container'),
        spinner: $('.buce-controls .spinner'),
        searchTimeout: null,
        currentRequest: null,

        init: function () {
            this.bindEvents();
            this.fetchProducts(1);
        },

        bindEvents: function () {
            this.productList
                .on('click', '.buce-remove-tag', this.handleRemoveTag.bind(this))
                .on('input change', '.buce-discount-input', this.handleInputChange.bind(this))
                .on('focus', '.buce-tag-container input[type="text"]', this.handleSearchFocus)
                .on('keyup', '.buce-tag-container input[type="text"]', this.handleSearchInput.bind(this))
                .on('blur', '.buce-tag-container input[type="text"]', this.handleSearchBlur);

            this.filterButton.on('click', (e) => { e.preventDefault(); this.fetchProducts(1); });
            this.saveButton.on('click', this.saveProducts.bind(this));
            this.paginationContainer.on('click', 'a.page-numbers', this.handlePagination.bind(this));
        },
        
        markRowAsDirty: function(el) {
            $(el).closest('tr').addClass('is-dirty');
            this.saveButton.prop('disabled', false);
        },
        
        handleRemoveTag: function (e) {
            e.preventDefault();
            const tag = $(e.currentTarget).closest('.buce-tag');
            this.markRowAsDirty(tag);
            tag.remove();
        },

        handleInputChange: function (e) {
            this.markRowAsDirty(e.currentTarget);
        },

        handlePagination: function (e) {
            e.preventDefault();
            const pageUrl = new URL(e.currentTarget.href);
            this.fetchProducts(pageUrl.searchParams.get('paged') || 1);
        },
        
        handleSearchFocus: () => $('.buce-search-results').remove(),

        handleSearchInput: function (e) {
            const inputField = $(e.currentTarget);
            const term = inputField.val().trim();
            const container = inputField.parent('.buce-tag-container');
            
            clearTimeout(this.searchTimeout);
            container.find('.buce-search-results').remove();
            if (term.length < 3) return;

            this.searchTimeout = setTimeout(() => {
                $.ajax({
                    url: buce_ajax.ajax_url,
                    type: 'GET',
                    data: { action: 'buce_search_products', nonce: buce_ajax.nonce, term: term },
                    success: (response) => {
                        if (response.success && response.data.length) {
                            let resultsHtml = response.data.map(p => `<li data-id="${p.id}" data-name="${$('<textarea />').html(p.name).text()}">${p.name}${p.sku}</li>`).join('');
                            const results = $(`<ul class="buce-search-results">${resultsHtml}</ul>`);
                            container.append(results);
                            results.on('click', 'li', (e) => this.addTag(e, inputField));
                        }
                    }
                });
            }, 400);
        },

        handleSearchBlur: () => setTimeout(() => $('.buce-search-results').remove(), 200),
        
        addTag: function(e, inputField) {
            const li = $(e.currentTarget);
            const container = inputField.parent('.buce-tag-container');
            const { id, name } = li.data();

            if (container.find(`.buce-tag[data-id="${id}"]`).length > 0) return;

            let tagHtml = `<span class="buce-tag" data-id="${id}" data-name="${name}">${name} `;
            if (container.data('type') === 'cross-sell') {
                tagHtml += `<span class="buce-discount-wrapper">(Discount: <input type="number" class="buce-discount-input" value="0" min="0" max="100"> %%)</span>`;
            }
            tagHtml += `<button type="button" class="buce-remove-tag">×</button></span>`;
            
            inputField.before(tagHtml);
            this.markRowAsDirty(inputField);
            inputField.val('').focus();
            container.find('.buce-search-results').remove();
        },

        fetchProducts: function (page = 1) {
            if (this.currentRequest) this.currentRequest.abort();
            this.spinner.css('visibility', 'visible');
            this.saveButton.prop('disabled', true);
            this.productList.css('opacity', 0.5);

            this.currentRequest = $.ajax({
                url: buce_ajax.ajax_url,
                type: 'POST',
                data: { action: 'buce_fetch_products', nonce: buce_ajax.nonce, page: page, search: this.searchBox.val(), category: this.categoryFilter.val(), include_variations: this.variationsCheckbox.is(':checked') },
                success: (response) => {
                    if (response.success) {
                        this.productList.html(response.data.html);
                        this.paginationContainer.html(response.data.pagination);
                    } else {
                        this.productList.html('<tr><td colspan="3">Error fetching products.</td></tr>');
                    }
                },
                error: (jqXHR, status) => { if (status !== 'abort') this.productList.html('<tr><td colspan="3">AJAX error.</td></tr>'); },
                complete: () => { this.spinner.css('visibility', 'hidden'); this.productList.css('opacity', 1); this.currentRequest = null; },
            });
        },

        saveProducts: function () {
            const dirtyRows = this.productList.find('tr.is-dirty');
            if (dirtyRows.length === 0) return;

            this.spinner.css('visibility', 'visible');
            this.saveButton.prop('disabled', true);

            const productsData = dirtyRows.get().map(row => {
                const $row = $(row);
                return {
                    id: $row.data('id'),
                    upsells: $row.find('[data-type="upsell"] .buce-tag').get().map(tag => $(tag).data('id')),
                    cross_sells: $row.find('[data-type="cross-sell"] .buce-tag').get().map(tag => ({ id: $(tag).data('id'), discount: $(tag).find('.buce-discount-input').val() || 0 }))
                };
            });

            $.ajax({
                url: buce_ajax.ajax_url,
                type: 'POST',
                data: { action: 'buce_save_products', nonce: buce_ajax.nonce, products: productsData },
                success: (response) => {
                    this.showMessage(response.data.message || 'Error', response.success ? 'notice-success' : 'notice-error');
                    if(response.success) {
                        dirtyRows.removeClass('is-dirty');
                        this.saveButton.prop('disabled', true);
                    }
                },
                error: () => this.showMessage('AJAX error on save.', 'notice-error'),
                complete: () => this.spinner.css('visibility', 'hidden'),
            });
        },
        
        showMessage: function(message, type) {
            this.messageContainer.html(`<div class="notice ${type} is-dismissible"><p>${message}</p></div>`).hide().fadeIn(200);
            setTimeout(() => this.messageContainer.fadeOut(400, () => $(this).empty()), 4000);
        }
    };
    editor.init();
});