<?php
/*
Plugin Name: Bulk Trash by URL
Description: Bulk‑trash posts, pages and custom post types from pasted URLs. Fast URL mapping, batched processing with pause/resume, and an optional summary.
Version: 1.1
Author: Ivan Trendafilov
Text Domain: bulk-trash-by-url
License: GPLv2 or later
License URI: https://www.gnu.org/licenses/gpl-2.0.html
*/

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly
}

// Configurable maximum URLs allowed per run (default). Can be overridden via option.
if (!defined('BULKTRBY_MAX_URLS')) {
    define('BULKTRBY_MAX_URLS', 7500);
}

// -------- Settings helpers --------
/**
 * Get effective max URLs limit, preferring saved option and falling back to constant.
 */
function bulktrby_get_max_urls() {
    $opt = get_option('bulktrby_max_urls', null);
    if ($opt !== null && $opt !== false && $opt !== '') {
        $v = (int) $opt;
        if ($v > 0) return $v;
    }
    return (int) (defined('BULKTRBY_MAX_URLS') ? BULKTRBY_MAX_URLS : 7500);
}

/**
 * Get mapping page size (per-AJAX mapping batch size). Default 250.
 */
function bulktrby_get_map_batch_size() {
    $opt = get_option('bulktrby_map_batch_size', null);
    $v = ($opt !== null && $opt !== false && $opt !== '') ? (int) $opt : 250;
    if ($v <= 0) $v = 250;
    return $v;
}

/**
 * Whether to show a run summary after completion. Default true.
 */
function bulktrby_get_summary_enabled() {
    $opt = get_option('bulktrby_show_summary', null);
    if ($opt === null || $opt === false || $opt === '') {
        return true; // default
    }
    return (bool) (int) $opt;
}

/**
 * Whether to set matched items to private instead of moving to Trash.
 */
function bulktrby_use_private_mode() {
    $opt = get_option('bulktrby_set_private', null);
    if ($opt !== null && $opt !== false && $opt !== '') {
        return (bool) (int) $opt;
    }
    return false;
}

/**
 * Current action mode used by UI and summary.
 */
function bulktrby_get_action_mode() {
    return bulktrby_use_private_mode() ? 'private' : 'trash';
}

add_action('admin_menu', 'bulktrby_menu');
add_action('admin_enqueue_scripts', 'bulktrby_enqueue_assets');
add_action('wp_ajax_bulktrby_trash_batch', 'bulktrby_ajax_trash_batch');
add_action('wp_ajax_bulktrby_map_batch', 'bulktrby_ajax_map_batch');
add_action('wp_ajax_bulktrby_store_summary', 'bulktrby_ajax_store_summary');

function bulktrby_menu() {
    add_management_page('Bulk Trash by URL', 'Bulk Trash by URL', 'manage_options', 'bulk-trash-by-url', 'bulktrby_page');
}

function bulktrby_page() {
    include_once 'admin/page.php';
}

function bulktrby_enqueue_assets($hook) {
    if ($hook !== 'tools_page_bulk-trash-by-url') {
        return;
    }
    $handle = 'bulktrby-admin';
    $script_path = plugin_dir_path(__FILE__) . 'admin/trasher.js';
    $script_url  = plugin_dir_url(__FILE__) . 'admin/trasher.js';
    $version = file_exists($script_path) ? filemtime($script_path) : '1.1';
    wp_enqueue_script($handle, $script_url, ['jquery'], $version, true);
    wp_localize_script($handle, 'bulktrby', [
        'ajaxUrl'   => admin_url('admin-ajax.php'),
        'nonce'     => wp_create_nonce('bulktrby_trash'),
        'mapNonce'  => wp_create_nonce('bulktrby_map'),
        'summaryNonce' => wp_create_nonce('bulktrby_summary'),
        'batchSize' => apply_filters('bulktrby_batch_size', 25),
        // Map batch size can be overridden via settings; filters remain supported.
        'mapBatchSize' => apply_filters('bulktrby_map_batch_size', (int) bulktrby_get_map_batch_size()),
        'maxUrls'   => (int) bulktrby_get_max_urls(),
        'summaryEnabled' => bulktrby_get_summary_enabled() ? 1 : 0,
        'actionMode' => bulktrby_get_action_mode(),
        'i18n'      => [
            'starting' => __('Starting trash process...', 'bulk-trash-by-url'),
            'paused'   => __('Paused', 'bulk-trash-by-url'),
            'resuming' => __('Resuming...', 'bulk-trash-by-url'),
            'done'     => __('All done!', 'bulk-trash-by-url'),
            'mapping'  => __('Mapping URLs...', 'bulk-trash-by-url'),
            'mapDone'  => __('Mapping complete.', 'bulk-trash-by-url'),
            'startOver' => __('Start Over', 'bulk-trash-by-url'),
            'noItemsSelected' => __('No items selected.', 'bulk-trash-by-url'),
            'noItemsFound' => __('No items found for the provided URLs.', 'bulk-trash-by-url'),
            'setPrivate' => __('Set to private', 'bulk-trash-by-url'),
            'trashedLabel' => __('Trashed', 'bulk-trash-by-url'),
        ],
    ]);
}

function bulktrby_ajax_trash_batch() {
    if (!current_user_can('manage_options')) {
        wp_send_json_error(['message' => __('Insufficient permissions.', 'bulk-trash-by-url')], 403);
    }
    check_ajax_referer('bulktrby_trash', 'nonce');

    // Respect per-run action mode passed from the client; fall back to current option.
    $action_mode = isset($_POST['action_mode']) ? sanitize_text_field( wp_unslash( $_POST['action_mode'] ) ) : '';
    $use_private = ($action_mode === 'private') ? true : bulktrby_use_private_mode();

    $post_ids = isset($_POST['post_ids']) ? array_map('wp_unslash', (array) $_POST['post_ids']) : [];
    $post_ids = array_map('intval', $post_ids);
    $post_ids = array_values(array_unique(array_filter($post_ids)));

    // Defer recounts and cache invalidation within this batch request.
    wp_suspend_cache_invalidation(true);
    wp_defer_term_counting(true);
    wp_defer_comment_counting(true);

    $results = [];
    foreach ($post_ids as $post_id) {
        if ($post_id <= 0) {
            $results[$post_id] = ['status' => 'invalid'];
            continue;
        }
        if (!current_user_can('delete_post', $post_id)) {
            $results[$post_id] = ['status' => 'no_permission'];
            continue;
        }
        if ($use_private) {
            $updated = wp_update_post([
                'ID' => $post_id,
                'post_status' => 'private',
            ], true);
            $ok = !is_wp_error($updated) && $updated;
            $results[$post_id] = ['status' => $ok ? 'privated' : 'failed'];
        } else {
            $ok = wp_trash_post($post_id);
            $results[$post_id] = ['status' => $ok ? 'trashed' : 'failed'];
        }
    }

    // Restore defaults.
    wp_defer_term_counting(false);
    wp_defer_comment_counting(false);
    wp_suspend_cache_invalidation(false);

    wp_send_json_success(['results' => $results]);
}

// Helpers for URL mapping
function bulktrby_normalize_url($url) {
    $url = trim((string) $url);
    if ($url === '') {
        return '';
    }
    if (strpos($url, '//') === 0) {
        $url = (is_ssl() ? 'https:' : 'http:') . $url;
    } elseif ($url[0] === '/') {
        $url = home_url($url);
    }
    $url = esc_url_raw($url);
    $hash_pos = strpos($url, '#');
    if ($hash_pos !== false) {
        $url = substr($url, 0, $hash_pos);
    }
    $disallowed = apply_filters('bulktrby_disallowed_query_args', [
        'utm_source','utm_medium','utm_campaign','utm_term','utm_content','utm_id','utm_name',
        'gclid','fbclid','mc_cid','mc_eid','_hsenc','_hsmi','ref','mkt_tok'
    ]);
    $url = remove_query_arg($disallowed, $url);
    return $url;
}

function bulktrby_try_postid_variants($url) {
    $post_id = url_to_postid($url);
    if ($post_id) {
        return (int) $post_id;
    }
    $relative = wp_make_link_relative($url);
    if ($relative) {
        $abs = home_url($relative);
        $post_id = url_to_postid($abs);
        if ($post_id) {
            return (int) $post_id;
        }
        $ts = user_trailingslashit($abs);
        $uts = untrailingslashit($abs);
        foreach ([$ts, $uts] as $variant) {
            if ($variant && ($pid = url_to_postid($variant))) {
                return (int) $pid;
            }
        }
    }
    $ts = user_trailingslashit($url);
    $uts = untrailingslashit($url);
    foreach ([$ts, $uts] as $variant) {
        if ($variant && ($pid = url_to_postid($variant))) {
            return (int) $pid;
        }
    }
    return 0;
}

function bulktrby_map_urls_to_posts($input_urls) {
    $found_posts = [];
    $not_found_urls = [];
    $normalized = [];
    foreach ($input_urls as $u) {
        $n = bulktrby_normalize_url($u);
        if ($n !== '') {
            $normalized[$n] = true;
        }
    }
    $urls = array_keys($normalized);
    foreach ($urls as $url) {
        $post_id = bulktrby_try_postid_variants($url);
        if ($post_id) {
            if (!isset($found_posts[$post_id])) {
                $found_posts[$post_id] = get_the_title($post_id);
            }
        } else {
            $not_found_urls[] = $url;
        }
    }
    return [
        'found_posts'    => $found_posts,
        'not_found_urls' => array_values($not_found_urls),
    ];
}

function bulktrby_ajax_map_batch() {
    if (!current_user_can('manage_options')) {
        wp_send_json_error(['message' => __('Insufficient permissions.', 'bulk-trash-by-url')], 403);
    }
    check_ajax_referer('bulktrby_map', 'nonce');

    $urls = isset($_POST['urls']) ? array_map('wp_unslash', (array) $_POST['urls']) : [];
    $urls = array_map('trim', $urls);
    $urls = array_filter($urls, function ($v) { return $v !== ''; });

    $results = bulktrby_map_urls_to_posts($urls);
    wp_send_json_success([
        'found_posts'    => $results['found_posts'],
        'not_found_urls' => $results['not_found_urls'],
    ]);
}

function bulktrby_ajax_store_summary() {
    if (!current_user_can('manage_options')) {
        wp_send_json_error(['message' => __('Insufficient permissions.', 'bulk-trash-by-url')], 403);
    }
    check_ajax_referer('bulktrby_summary', 'nonce');

    $uid = get_current_user_id();
    $summary = [
        'time'       => time(),
        'total'      => isset($_POST['total']) ? (int) sanitize_text_field( wp_unslash( $_POST['total'] ) ) : 0,
        'processed'  => isset($_POST['processed']) ? (int) sanitize_text_field( wp_unslash( $_POST['processed'] ) ) : 0,
        'trashed_ids' => array_values( array_filter( array_map( 'intval', array_map( 'wp_unslash', isset($_POST['trashed_ids']) ? (array) $_POST['trashed_ids'] : [] ) ) ) ),
        'failed_ids'  => array_values( array_filter( array_map( 'intval', array_map( 'wp_unslash', isset($_POST['failed_ids']) ? (array) $_POST['failed_ids'] : [] ) ) ) ),
        'no_permission_ids' => array_values( array_filter( array_map( 'intval', array_map( 'wp_unslash', isset($_POST['no_permission_ids']) ? (array) $_POST['no_permission_ids'] : [] ) ) ) ),
        'invalid_ids' => array_values( array_filter( array_map( 'intval', array_map( 'wp_unslash', isset($_POST['invalid_ids']) ? (array) $_POST['invalid_ids'] : [] ) ) ) ),
        'action'     => isset($_POST['action_mode']) ? sanitize_text_field( wp_unslash( $_POST['action_mode'] ) ) : '',
    ];

    set_transient('bulktrby_summary_' . $uid, $summary, 30 * MINUTE_IN_SECONDS);

    $redirect = admin_url('tools.php?page=bulk-trash-by-url&bulktrby_summary=1');
    wp_send_json_success(['redirect' => $redirect]);
}
