/* global jQuery, bulktrby */
jQuery(function ($) {
  // Elements for trashing phase
  const $form = $('#url-bulk-trasher-confirm');
  const $start = $('#ubt-start');
  const $pause = $('#ubt-pause');
  const $resume = $('#ubt-resume');
  const $bar = $('#ubt-bar');
  const $status = $('#ubt-status');

  // Elements for mapping phase
  const $mapBar = $('#ubt-map-bar');
  const $mapStatus = $('#ubt-map-status');
  const $selection = $('#ubt-selection');

  const actionMode = (bulktrby && typeof bulktrby.actionMode === 'string' && bulktrby.actionMode === 'private') ? 'private' : 'trash';
  const successLabel = actionMode === 'private'
    ? ((bulktrby && bulktrby.i18n && bulktrby.i18n.setPrivate) || 'Set to private')
    : ((bulktrby && bulktrby.i18n && bulktrby.i18n.trashedLabel) || 'Trashed');

  let allIds = [];
  let total = 0;
  let processed = 0;
  let trashed = 0;
  let failed = 0;
  let noPerm = 0;
  let invalid = 0;
  let index = 0;
  let paused = false;
  const batchSize = parseInt(bulktrby && bulktrby.batchSize, 10) || 25;

  // Mapping state
  const rawUrls = Array.isArray(window.bulktrbyRawUrls) ? window.bulktrbyRawUrls : null;
  let mapIndex = 0;
  let mapProcessed = 0;
  let mapFoundCount = 0;
  let mapNotFoundCount = 0;
  const mapBatchSize = parseInt(bulktrby && bulktrby.mapBatchSize, 10) || 250;
  const foundPostsById = {};
  // Accumulate unique unmatched URLs for display after mapping
  const notFoundUrlsSet = new Set();
  const $notFoundWrap = $('#ubt-not-found');
  const $notFoundSummary = $('#ubt-not-found-summary');
  const $notFoundList = $('#ubt-not-found-list');

  // Track detailed results for summary view
  const trashedIds = [];
  const failedIds = [];
  const noPermIds = [];
  const invalidIds = [];
  const summaryEnabled = !!(bulktrby && parseInt(bulktrby.summaryEnabled, 10));

  function setMapStatus(text) {
    if ($mapStatus.length) $mapStatus.text(text);
  }

  function updateMapProgress() {
    if (!$mapBar.length || !rawUrls) return;
    const totalUrls = rawUrls.length;
    const pct = totalUrls ? Math.min(100, Math.round((mapProcessed / totalUrls) * 100)) : 0;
    $mapBar.css('width', pct + '%');
    const summary = `Mapped ${mapProcessed}/${totalUrls} — found: ${mapFoundCount}, not matched: ${mapNotFoundCount}`;
    setMapStatus(summary);
  }

  function finishMapping() {
    updateMapProgress();
    // If we have any unmatched URLs, render them
    if ($notFoundWrap.length && notFoundUrlsSet.size > 0) {
      try {
        const urls = Array.from(notFoundUrlsSet);
        // Summary + list
        if ($notFoundSummary.length) {
          $notFoundSummary.text(`Unmatched URLs: ${urls.length}`);
        }
        if ($notFoundList.length) {
          $notFoundList.empty();
          const frag = document.createDocumentFragment();
          urls.forEach(function (u) {
            const li = document.createElement('li');
            li.appendChild(document.createTextNode(u));
            frag.appendChild(li);
          });
          $notFoundList.append(frag);
        }
        $notFoundWrap.show();
      } catch (e) {
        // noop: rendering unmatched URLs should never block the flow
      }
    }
    if (Object.keys(foundPostsById).length === 0) {
      // Nothing found
      if ($selection.length) {
        $selection.hide();
      }
      const $wrap = $('.wrap');
      if ($wrap.length) {
        const $msg = $('<p/>').text((bulktrby && bulktrby.i18n && bulktrby.i18n.noItemsFound) || 'No items found for the provided URLs.');
        const $a = $('<a class="button button-primary"/>')
          .attr('href', (window.ajaxurl || (window.bulktrby && bulktrby.ajaxUrl)).replace('admin-ajax.php', 'tools.php?page=bulk-trash-by-url'))
          .text((bulktrby && bulktrby.i18n && bulktrby.i18n.startOver) || 'Start Over');
        $wrap.append($('<div/>').append($msg)).append($a);
      }
      return;
    }
    // Populate selection form
    const ids = Object.keys(foundPostsById);
    const frag = document.createDocumentFragment();
    ids.forEach(function (id) {
      const title = foundPostsById[id];
      const label = document.createElement('label');
      const cb = document.createElement('input');
      cb.type = 'checkbox';
      cb.name = 'post_ids[]';
      cb.value = id;
      cb.checked = true;
      label.appendChild(cb);
      label.appendChild(document.createTextNode(' ' + title + ' (ID: ' + id + ')'));
      frag.appendChild(label);
      frag.appendChild(document.createElement('br'));
    });
    if ($form.length) {
      $form.empty().append(frag);
    }
    if ($selection.length) {
      $selection.show();
    }
    if ($start.length) {
      $start.prop('disabled', false);
    }
    setMapStatus((bulktrby && bulktrby.i18n ? bulktrby.i18n.mapDone : 'Mapping complete.'));
  }

  function processNextMapBatch() {
    if (!rawUrls) return;
    if (mapIndex >= rawUrls.length) {
      finishMapping();
      return;
    }
    const slice = rawUrls.slice(mapIndex, mapIndex + mapBatchSize);
    mapIndex += slice.length;

    $.ajax({
      method: 'POST',
      url: (bulktrby && bulktrby.ajaxUrl) || ajaxurl,
      dataType: 'json',
      data: {
        action: 'bulktrby_map_batch',
        nonce: bulktrby && bulktrby.mapNonce,
        urls: slice,
      },
    })
      .done(function (resp) {
        if (!resp || resp.success !== true || !resp.data) {
          // consider all in slice as not found to keep progress moving
          mapProcessed += slice.length;
          mapNotFoundCount += slice.length;
          // We only have raw slice values here; add them so the operator sees something actionable
          slice.forEach(function (u) { if (u) notFoundUrlsSet.add(String(u)); });
          updateMapProgress();
          return;
        }
        const fp = resp.data.found_posts || {};
        const nf = Array.isArray(resp.data.not_found_urls) ? resp.data.not_found_urls : [];
        Object.keys(fp).forEach(function (id) {
          // Accumulate titles, last one wins (titles may update)
          foundPostsById[id] = fp[id] || '';
        });
        mapFoundCount = Object.keys(foundPostsById).length;
        mapProcessed += slice.length;
        mapNotFoundCount += nf.length;
        // Accumulate unique unmatched URLs for final display
        nf.forEach(function (u) { if (u) notFoundUrlsSet.add(String(u)); });
        updateMapProgress();
      })
      .fail(function () {
        mapProcessed += slice.length;
        mapNotFoundCount += slice.length;
        slice.forEach(function (u) { if (u) notFoundUrlsSet.add(String(u)); });
        updateMapProgress();
      })
      .always(function () {
        processNextMapBatch();
      });
  }

  // Auto-start mapping if raw URLs present
  if (rawUrls && rawUrls.length && $mapBar.length) {
    setMapStatus((bulktrby && bulktrby.i18n ? bulktrby.i18n.mapping : 'Mapping URLs...'));
    updateMapProgress();
    processNextMapBatch();
  }

  function setStatus(text) {
    $status.text(text);
  }

  function updateProgress() {
    const pct = total ? Math.min(100, Math.round((processed / total) * 100)) : 0;
    $bar.css('width', pct + '%');
    const summary = `Processed ${processed}/${total} — ${successLabel}: ${trashed}, no permission: ${noPerm}, failed: ${failed}`;
    setStatus(summary + (paused ? ' — ' + (bulktrby && bulktrby.i18n ? bulktrby.i18n.paused : 'Paused') : ''));
  }

  function lockUI(running) {
    $form.find('input[type="checkbox"]').prop('disabled', running);
    $start.prop('disabled', running);
    $pause.prop('disabled', !running);
    $resume.toggle(false);
  }

  function finalize() {
    // Lock controls and show final status briefly
    lockUI(false);
    $start.prop('disabled', true).hide();
    $form.find('input[type="checkbox"]').prop('disabled', true);
    $pause.prop('disabled', true);
    $resume.prop('disabled', true).hide();
    setStatus((bulktrby && bulktrby.i18n ? bulktrby.i18n.done : 'All done!') + ` Processed ${processed}/${total}.`);

    if (summaryEnabled) {
      // Store summary on server then redirect to summary page
      try {
        $.ajax({
          method: 'POST',
          url: (bulktrby && bulktrby.ajaxUrl) || ajaxurl,
          dataType: 'json',
          data: {
            action: 'bulktrby_store_summary',
            nonce: bulktrby && bulktrby.summaryNonce,
            total: total,
            processed: processed,
            trashed_ids: trashedIds,
            failed_ids: failedIds,
            no_permission_ids: noPermIds,
            invalid_ids: invalidIds,
            action_mode: actionMode,
          },
        }).done(function(resp){
          var url = (resp && resp.success && resp.data && resp.data.redirect)
            ? resp.data.redirect
            : ((window.ajaxurl || (window.bulktrby && bulktrby.ajaxUrl)) || '').replace('admin-ajax.php', 'tools.php?page=bulk-trash-by-url&bulktrby_summary=1');
          if (url) window.location = url;
        }).fail(function(){
          var fallback = ((window.ajaxurl || (window.bulktrby && bulktrby.ajaxUrl)) || '').replace('admin-ajax.php', 'tools.php?page=bulk-trash-by-url&bulktrby_summary=1');
          if (fallback) window.location = fallback;
        });
      } catch (e) {
        var fallback2 = ((window.ajaxurl || (window.bulktrby && bulktrby.ajaxUrl)) || '').replace('admin-ajax.php', 'tools.php?page=bulk-trash-by-url&bulktrby_summary=1');
        if (fallback2) window.location = fallback2;
      }
    } else {
      // No summary: just show a restart button for convenience
      try {
        var $wrap = $('.wrap');
        if ($wrap.length) {
          var $a = $('<a class="button button-primary"/>')
            .attr('href', (window.ajaxurl || (window.bulktrby && bulktrby.ajaxUrl)).replace('admin-ajax.php', 'tools.php?page=bulk-trash-by-url'))
            .text('Start Over');
          $wrap.append($('<div style="margin-top:12px;"/>').append($a));
        }
      } catch (e) {}
    }
  }

  function processNextBatch() {
    if (paused) return;
    if (index >= total) {
      finalize();
      return;
    }
    const slice = allIds.slice(index, index + batchSize);
    index += slice.length;

    $.ajax({
      method: 'POST',
      url: (bulktrby && bulktrby.ajaxUrl) || ajaxurl,
      dataType: 'json',
      data: {
        action: 'bulktrby_trash_batch',
        nonce: bulktrby && bulktrby.nonce,
        post_ids: slice,
        action_mode: actionMode,
      },
    })
      .done(function (resp) {
        if (!resp || resp.success !== true || !resp.data || !resp.data.results) {
          // count all as failed for visibility
          processed += slice.length;
          failed += slice.length;
          slice.forEach(function (nid) { if (Number.isInteger(nid)) failedIds.push(nid); });
          updateProgress();
          return;
        }
        const results = resp.data.results;
        Object.keys(results).forEach(function (id) {
          const st = results[id] && results[id].status;
          const nid = parseInt(id, 10);
          processed += 1;
          if (st === 'trashed' || st === 'privated') { trashed += 1; if (Number.isInteger(nid)) trashedIds.push(nid); }
          else if (st === 'no_permission') { noPerm += 1; if (Number.isInteger(nid)) noPermIds.push(nid); }
          else if (st === 'invalid') { invalid += 1; if (Number.isInteger(nid)) invalidIds.push(nid); }
          else { failed += 1; if (Number.isInteger(nid)) failedIds.push(nid); }
        });
        updateProgress();
      })
      .fail(function () {
        // Treat this batch as failed
        processed += slice.length;
        failed += slice.length;
        slice.forEach(function (nid) { if (Number.isInteger(nid)) failedIds.push(nid); });
        updateProgress();
      })
      .always(function () {
        // Next batch
        processNextBatch();
      });
  }

  $start.on('click', function () {
    allIds = $form.find('input[name="post_ids[]"]:checked')
      .map(function () { return parseInt($(this).val(), 10); })
      .get()
      .filter(function (v) { return Number.isInteger(v) && v > 0; });

    if (!allIds.length) {
      setStatus((bulktrby && bulktrby.i18n && bulktrby.i18n.noItemsSelected) || 'No items selected.');
      return;
    }
    total = allIds.length;
    processed = trashed = failed = noPerm = invalid = index = 0;
    paused = false;
    lockUI(true);
    setStatus(bulktrby && bulktrby.i18n ? bulktrby.i18n.starting : 'Starting...');
    updateProgress();
    processNextBatch();
  });

  $pause.on('click', function () {
    paused = true;
    $pause.prop('disabled', true);
    $resume.prop('disabled', false).show();
    updateProgress();
  });

  $resume.on('click', function () {
    paused = false;
    $pause.prop('disabled', false);
    $resume.prop('disabled', true).hide();
    setStatus(bulktrby && bulktrby.i18n ? bulktrby.i18n.resuming : 'Resuming...');
    processNextBatch();
  });
});
