<?php
if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

function bulktrby_form() {
    ?>
    <div class="wrap">
        <h1>Bulk Trash by URL</h1>
        <h2 class="nav-tab-wrapper">
            <a href="<?php echo esc_url( admin_url('tools.php?page=bulk-trash-by-url') ); ?>" class="nav-tab nav-tab-active"><?php echo esc_html__('Find Items', 'bulk-trash-by-url'); ?></a>
            <a href="<?php echo esc_url( admin_url('tools.php?page=bulk-trash-by-url&tab=settings') ); ?>" class="nav-tab"><?php echo esc_html__('Settings', 'bulk-trash-by-url'); ?></a>
        </h2>
        <form method="post" id="url-bulk-trasher-input-form">
            <?php wp_nonce_field('bulktrby_find'); ?>
            <textarea id="url_list" name="url_list" rows="10" cols="50" style="width: 100%;" placeholder="<?php
            /* translators: %d: maximum number of URLs allowed per run. */
            echo esc_attr( sprintf( __( 'Paste your URLs here, each on a new line. Max %d URLs.', 'bulk-trash-by-url' ), (int) bulktrby_get_max_urls() ) );
            ?>"></textarea><br>
            <p class="description"><?php
            /* translators: %d: maximum number of URLs allowed per run. */
            echo esc_html( sprintf( __( 'Maximum %d URLs per run. Large lists are mapped and trashed in batches.', 'bulk-trash-by-url' ), (int) bulktrby_get_max_urls() ) );
            ?></p>
            <input type="submit" name="submit_urls" class="button button-primary" value="<?php echo esc_attr__('Find Items', 'bulk-trash-by-url'); ?>">
        </form>
    </div>
    <?php
    // Attach inline script via WP API (no direct <script> tags)
    wp_add_inline_script(
        'bulktrby-admin',
        "(function(){\n  var f = document.getElementById('url-bulk-trasher-input-form');\n  if(!f) return;\n  f.addEventListener('submit', function(e){\n    var max = parseInt((window.bulktrby && bulktrby.maxUrls) || 0, 10) || 0;\n    var v = document.getElementById('url_list').value || '';\n    var lines = v.split(/\\r?\\n/).map(function(s){return s.trim();}).filter(function(s){return s!==''});\n    if(max && lines.length > max){\n      if(!confirm('You entered '+lines.length+' URLs. Only the first '+max+' will be processed. Continue?')){\n        e.preventDefault();\n        return false;\n      }\n    }\n  });\n})();",
        'after'
    );
}

function bulktrby_main() {
    if (isset($_GET['bulktrby_summary'])) {
        bulktrby_display_summary();
        return;
    }
    // Settings tab
    if (isset($_GET['tab']) && $_GET['tab'] === 'settings') {
        bulktrby_settings_page();
        return;
    }
    if (isset($_POST['submit_urls'])) {
        if (!isset($_POST['_wpnonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['_wpnonce'])), 'bulktrby_find')) {
            wp_die(esc_html__('Security check failed.', 'bulk-trash-by-url'));
        }
        // Preserve percent-encoded sequences; sanitize each URL later via esc_url_raw in normalization.
        // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
        $raw = isset($_POST['url_list']) ? (string) wp_unslash($_POST['url_list']) : '';
        $lines = preg_split('/\r\n|\r|\n/', $raw);
        $lines = array_map('trim', $lines);
        $lines = array_filter($lines, function ($v) { return $v !== ''; });
        $total_before = count($lines);
        $max_urls = (int) bulktrby_get_max_urls();
        $lines = array_slice($lines, 0, $max_urls);
        $total_after = count($lines);
        bulktrby_display_mapping_and_trash_ui($lines, $total_before, $total_after);
    } else {
        bulktrby_form();
    }
}

function bulktrby_display_mapping_and_trash_ui($lines, $total_before, $total_after) {
    ?>
    <div class="wrap">
        <h1>Bulk Trash by URL</h1>
        <h2 class="nav-tab-wrapper">
            <a href="<?php echo esc_url( admin_url('tools.php?page=bulk-trash-by-url') ); ?>" class="nav-tab nav-tab-active"><?php echo esc_html__('Find Items', 'bulk-trash-by-url'); ?></a>
            <a href="<?php echo esc_url( admin_url('tools.php?page=bulk-trash-by-url&tab=settings') ); ?>" class="nav-tab"><?php echo esc_html__('Settings', 'bulk-trash-by-url'); ?></a>
        </h2>
        <?php if ($total_before > $total_after): ?>
            <div class="notice notice-warning"><p><?php
            /* translators: 1: number of URLs submitted; 2: maximum number of URLs that will be processed. */
            echo esc_html( sprintf( __( 'You submitted %1$d URLs. Only the first %2$d will be processed due to the current limit.', 'bulk-trash-by-url' ), (int) $total_before, (int) $total_after ) );
            ?></p></div>
        <?php endif; ?>
        <div id="ubt-mapping">
            <h2><?php echo esc_html__('Step 1: Map URLs to posts', 'bulk-trash-by-url'); ?></h2>
            <div id="ubt-map-progress" style="margin-top:8px; max-width:600px;">
                <div style="height:14px; background:#f0f0f1; border:1px solid #dcdcde; position:relative;">
                    <div id="ubt-map-bar" style="height:100%; width:0%; background:#3c8dbc;"></div>
                </div>
                <div id="ubt-map-status" style="margin-top:6px;"></div>
            </div>
            <div id="ubt-not-found" style="margin-top:12px; display:none; max-width:800px;">
                <h3><?php echo esc_html__('Unmatched URLs', 'bulk-trash-by-url'); ?></h3>
                <p id="ubt-not-found-summary"></p>
                <details>
                    <summary><?php echo esc_html__('Show list', 'bulk-trash-by-url'); ?></summary>
                    <ul id="ubt-not-found-list" style="margin-top:8px; list-style:disc; padding-left:20px;"></ul>
                </details>
            </div>
        </div>
        <div id="ubt-selection" style="margin-top:18px; display:none;">
            <h2><?php echo esc_html__('Step 2: Select items to trash', 'bulk-trash-by-url'); ?></h2>
            <form id="url-bulk-trasher-confirm" onsubmit="return false;"></form>
            <div style="margin-top:12px;">
                <button id="ubt-start" class="button button-primary" disabled><?php echo esc_html__('Start Trashing', 'bulk-trash-by-url'); ?></button>
                <button id="ubt-pause" class="button" disabled><?php echo esc_html__('Pause', 'bulk-trash-by-url'); ?></button>
                <button id="ubt-resume" class="button" style="display:none;">&raquo; <?php echo esc_html__('Resume', 'bulk-trash-by-url'); ?></button>
            </div>
            <div id="ubt-progress" style="margin-top:16px; max-width:600px;">
                <div style="height:14px; background:#f0f0f1; border:1px solid #dcdcde; position:relative;">
                    <div id="ubt-bar" style="height:100%; width:0%; background:#2271b1;"></div>
                </div>
                <div id="ubt-status" style="margin-top:6px;"></div>
            </div>
        </div>
    </div>
    <?php
    // Provide raw URLs to the enqueued script without printing <script> tags directly
    wp_add_inline_script(
        'bulktrby-admin',
        'window.bulktrbyRawUrls = ' . wp_json_encode(array_values($lines)) . ';',
        'before'
    );
}

function bulktrby_settings_page() {
    if (!current_user_can('manage_options')) {
        wp_die(esc_html__('Insufficient permissions.', 'bulk-trash-by-url'));
    }
    $saved = false;
    if (isset($_POST['bulktrby_save_settings'])) {
        if (!isset($_POST['_wpnonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['_wpnonce'])), 'bulktrby_save_settings')) {
            wp_die(esc_html__('Security check failed.', 'bulk-trash-by-url'));
        }
        $max_urls = isset($_POST['ubt_max_urls']) ? (int) sanitize_text_field(wp_unslash($_POST['ubt_max_urls'])) : 7500;
        $max_urls = max(1, min(200000, $max_urls));
        update_option('bulktrby_max_urls', $max_urls);

        $map_batch = isset($_POST['ubt_map_batch_size']) ? (int) sanitize_text_field(wp_unslash($_POST['ubt_map_batch_size'])) : 250;
        $map_batch = max(10, min(20000, $map_batch));
        update_option('bulktrby_map_batch_size', $map_batch);

        $show_summary = isset($_POST['ubt_show_summary']) ? 1 : 0;
        update_option('bulktrby_show_summary', $show_summary);

        $set_private = isset($_POST['bulktrby_set_private']) ? 1 : 0;
        update_option('bulktrby_set_private', $set_private);

        $saved = true;
    }
    $max_urls = (int) bulktrby_get_max_urls();
    $map_batch = (int) bulktrby_get_map_batch_size();
    $show_summary = (bool) bulktrby_get_summary_enabled();
    $set_private = bulktrby_use_private_mode();
    ?>
    <div class="wrap">
        <h1>Bulk Trash by URL</h1>
        <h2 class="nav-tab-wrapper">
            <a href="<?php echo esc_url( admin_url('tools.php?page=bulk-trash-by-url') ); ?>" class="nav-tab"><?php echo esc_html__('Find Items', 'bulk-trash-by-url'); ?></a>
            <a href="<?php echo esc_url( admin_url('tools.php?page=bulk-trash-by-url&tab=settings') ); ?>" class="nav-tab nav-tab-active"><?php echo esc_html__('Settings', 'bulk-trash-by-url'); ?></a>
        </h2>
        <?php if ($saved): ?>
            <div class="notice notice-success is-dismissible"><p><?php echo esc_html__('Settings saved.', 'bulk-trash-by-url'); ?></p></div>
        <?php endif; ?>
        <form method="post" action="">
            <?php wp_nonce_field('bulktrby_save_settings'); ?>
            <table class="form-table" role="presentation">
                <tbody>
                    <tr>
                        <th scope="row"><label for="ubt_map_batch_size"><?php echo esc_html__('Mapping Batch Size', 'bulk-trash-by-url'); ?></label></th>
                        <td>
                            <div style="max-width:600px;">
                                <strong id="ubt_map_batch_size_value" style="display:inline-block; min-width:60px;"><?php echo (int) $map_batch; ?></strong>
                                <input type="range" id="ubt_map_batch_size" name="ubt_map_batch_size" min="25" max="20000" step="25" value="<?php echo (int) $map_batch; ?>" style="width:100%;">
                                <p class="description"><?php echo esc_html__('If you notice timeouts or slow responses, try decreasing this value.', 'bulk-trash-by-url'); ?></p>
                            </div>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="ubt_max_urls"><?php echo esc_html__('Max URLs Per Run', 'bulk-trash-by-url'); ?></label></th>
                        <td>
                            <div style="max-width:600px;">
                                <strong id="ubt_max_urls_value" style="display:inline-block; min-width:60px;"><?php echo (int) $max_urls; ?></strong>
                                <input type="range" id="ubt_max_urls" name="ubt_max_urls" min="100" max="50000" step="100" value="<?php echo (int) $max_urls; ?>" style="width:100%;">
                                <p class="description"><?php echo esc_html__('Upper limit of URLs accepted in one run.', 'bulk-trash-by-url'); ?></p>
                            </div>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="ubt_show_summary"><?php echo esc_html__('Show Summary After Run', 'bulk-trash-by-url'); ?></label></th>
                        <td>
                            <label><input type="checkbox" id="ubt_show_summary" name="ubt_show_summary" value="1" <?php checked($show_summary, true); ?>> <?php echo esc_html__('Enable summary screen when processing completes.', 'bulk-trash-by-url'); ?></label>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="bulktrby_set_private"><?php echo esc_html__('Set to Private instead of Trash', 'bulk-trash-by-url'); ?></label></th>
                        <td>
                            <label><input type="checkbox" id="bulktrby_set_private" name="bulktrby_set_private" value="1" <?php checked($set_private, true); ?>> <?php echo esc_html__('When enabled, matched items are set to private (not trashed).', 'bulk-trash-by-url'); ?></label>
                            <p class="description"><?php echo esc_html__('Useful when you need to hide content without moving it to the Trash.', 'bulk-trash-by-url'); ?></p>
                        </td>
                    </tr>
                </tbody>
            </table>
            <p class="submit"><button type="submit" class="button button-primary" name="bulktrby_save_settings" value="1"><?php echo esc_html__('Save Settings', 'bulk-trash-by-url'); ?></button></p>
        </form>
    </div>
    <?php
    // Attach settings UI helpers via WP inline script
    wp_add_inline_script(
        'bulktrby-admin',
        "(function(){\n  function bindRange(id, labelId){\n    var el = document.getElementById(id), out = document.getElementById(labelId);\n    if(!el || !out) return;\n    var update = function(){ out.textContent = el.value; };\n    el.addEventListener('input', update);\n    el.addEventListener('change', update);\n  }\n  bindRange('ubt_map_batch_size', 'ubt_map_batch_size_value');\n  bindRange('ubt_max_urls', 'ubt_max_urls_value');\n})();",
        'after'
    );
}

function bulktrby_display_summary() {
    if (!current_user_can('manage_options')) {
        wp_die(esc_html__('Insufficient permissions.', 'bulk-trash-by-url'));
    }
    $uid = get_current_user_id();
    $summary = get_transient('bulktrby_summary_' . $uid);
    ?>
    <div class="wrap">
        <h1><?php echo esc_html__('Bulk Trash by URL — Summary', 'bulk-trash-by-url'); ?></h1>
        <?php if (!$summary || !is_array($summary)): ?>
            <div class="notice notice-info"><p><?php echo esc_html__('No summary available. Try running a trash operation.', 'bulk-trash-by-url'); ?></p></div>
            <a class="button button-primary" href="<?php echo esc_url(admin_url('tools.php?page=bulk-trash-by-url')); ?>"><?php echo esc_html__('Start Over', 'bulk-trash-by-url'); ?></a>
        <?php else: ?>
            <?php
            $total = isset($summary['total']) ? (int) $summary['total'] : 0;
            $processed = isset($summary['processed']) ? (int) $summary['processed'] : 0;
            $trashed_ids = isset($summary['trashed_ids']) ? (array) $summary['trashed_ids'] : [];
            $failed_ids = isset($summary['failed_ids']) ? (array) $summary['failed_ids'] : [];
            $no_perm_ids = isset($summary['no_permission_ids']) ? (array) $summary['no_permission_ids'] : [];
            $invalid_ids = isset($summary['invalid_ids']) ? (array) $summary['invalid_ids'] : [];
            $action_mode = isset($summary['action']) ? sanitize_text_field($summary['action']) : bulktrby_get_action_mode();
            $success_label = ($action_mode === 'private') ? __('Set to private', 'bulk-trash-by-url') : __('Trashed', 'bulk-trash-by-url');
            ?>
            <p><?php
            /* translators: 1: number of items processed; 2: total number of selected items. */
            echo esc_html( sprintf( __( 'Processed %1$d of %2$d selected items.', 'bulk-trash-by-url' ), (int) $processed, (int) $total ) );
            ?></p>

            <ul>
                <li><?php
                /* translators: %d: number of items successfully processed. */
                echo esc_html( sprintf( __( '%1$s: %2$d', 'bulk-trash-by-url' ), $success_label, count($trashed_ids) ) );
                ?></li>
                <li><?php
                /* translators: %d: number of items the user lacks permission to trash. */
                echo esc_html( sprintf( __( 'No permission: %d', 'bulk-trash-by-url' ), count($no_perm_ids) ) );
                ?></li>
                <li><?php
                /* translators: %d: number of invalid item IDs. */
                echo esc_html( sprintf( __( 'Invalid: %d', 'bulk-trash-by-url' ), count($invalid_ids) ) );
                ?></li>
                <li><?php
                /* translators: %d: number of items that failed to trash. */
                echo esc_html( sprintf( __( 'Failed: %d', 'bulk-trash-by-url' ), count($failed_ids) ) );
                ?></li>
            </ul>

            <?php
            $render_list = function($heading, $ids) {
                if (empty($ids)) return;
                echo '<h2>' . esc_html($heading) . '</h2>';
                echo '<ul style="list-style: disc; padding-left: 20px;">';
                foreach ($ids as $id) {
                    $title = get_the_title($id);
                    $title = $title ? $title : __('(no title)', 'bulk-trash-by-url');
                    echo '<li>' . esc_html($title) . ' (ID: ' . (int) $id . ')</li>';
                }
                echo '</ul>';
            };
            ?>

            <?php $render_list(($action_mode === 'private' ? __('Items set to private', 'bulk-trash-by-url') : __('Items trashed', 'bulk-trash-by-url')), $trashed_ids); ?>
            <?php $render_list(__('No permission', 'bulk-trash-by-url'), $no_perm_ids); ?>
            <?php $render_list(__('Invalid IDs', 'bulk-trash-by-url'), $invalid_ids); ?>
            <?php $render_list(($action_mode === 'private' ? __('Failed to set private', 'bulk-trash-by-url') : __('Failed to trash', 'bulk-trash-by-url')), $failed_ids); ?>

            <div style="margin-top: 16px;">
                <a class="button button-primary" href="<?php echo esc_url(admin_url('tools.php?page=bulk-trash-by-url')); ?>"><?php echo esc_html__('Start Over', 'bulk-trash-by-url'); ?></a>
            </div>
        <?php endif; ?>
    </div>
    <?php
}

bulktrby_main();
