<?php
/**
 * Public-facing functionality of the plugin.
 *
 * @package Aicoso_Bulk_Prod_Price
 */

/**
 * Exit if accessed directly
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}


if ( ! class_exists( 'Aicoso_Bulk_Prod_Price_Public' ) ) {

	/**
	 * This is class for managing front end per quantity functionality
	 *
	 * @name    Aicoso_Bulk_Prod_Price_Public
	 * @package Class
	 */
	class Aicoso_Bulk_Prod_Price_Public {
		/**
		 *
		 * Constructor of the class for initilization.
		 */
		public function __construct() {
			$plugin_enable = get_option( 'aicoso_bulk_prod_price_plugin_enable', 'off' );
			if ( 'on' === $plugin_enable ) {
				add_filter( 'woocommerce_get_price_html', array( $this, 'aicoso_bulk_prod_price_woo_get_price_html' ), 10, 2 );
				add_action( 'wp_enqueue_scripts', array( $this, 'aicoso_bulk_prod_price_wp_enqueue_scripts' ), 10 );
				add_action( 'wp_ajax_aicoso_bulk_prod_price_single_qty', array( $this, 'aicoso_bulk_prod_price_single_qty' ) );
				add_action( 'wp_ajax_nopriv_aicoso_bulk_prod_price_single_qty', array( $this, 'aicoso_bulk_prod_price_single_qty' ) );
				// Use very early priority to ensure our pricing runs first
				add_action( 'woocommerce_before_calculate_totals', array( $this, 'aicoso_bulk_prod_price_woo_before_calculate_totals' ), 5, 1 );

				// Add action to handle cart item quantity update
				add_action( 'woocommerce_after_cart_item_quantity_update', array( $this, 'aicoso_bulk_prod_price_after_quantity_update' ), 10, 4 );
				add_action( 'woocommerce_single_product_summary', array( $this, 'aicoso_bulk_prod_price_woo_single_product_summary' ), 25 );
				add_filter( 'woocommerce_product_is_taxable', array( $this, 'aicoso_bulk_prod_price_woo_product_is_taxable' ), 10, 2 );

				// Add support for cart item price display
				add_filter( 'woocommerce_cart_item_price', array( $this, 'aicoso_bulk_prod_price_cart_item_price' ), 10, 3 );
				add_filter( 'woocommerce_cart_item_subtotal', array( $this, 'aicoso_bulk_prod_price_cart_item_subtotal' ), 10, 3 );

				// Hook into cart loaded from session to apply pricing
				add_action( 'woocommerce_cart_loaded_from_session', array( $this, 'aicoso_bulk_prod_price_cart_loaded_from_session' ), 10, 1 );
			}
		}
		/**
		 * Function to check is product is taxable or not.
		 *
		 * @param boolean $taxable This is boolean value.
		 * @param array   $product Gloabl_product array.
		 */
		public function aicoso_bulk_prod_price_woo_product_is_taxable( $taxable, $product ) {
			$product_id     = $product->get_id();
			$pqp_qty_enable = get_post_meta( $product_id, '_aicoso_bulk_prod_price_per_quantity', true );
			if ( isset( $pqp_qty_enable ) && 'yes' === $pqp_qty_enable ) {
				$tax_calc = get_option( 'aicoso_bulk_prod_price_tax_enable', 'on' );
				if ( 'off' === $tax_calc ) {
					$taxable = false;
				}
			}
			return $taxable;
		}

		/**
		 * Function to check is product is single product summary.
		 */
		public function aicoso_bulk_prod_price_woo_single_product_summary() {
			$ftable_enable = get_option( 'aicoso_bulk_prod_price_ftable_enable', 'off' );
			if ( 'on' === $ftable_enable ) {
				global $post;
				$product_id   = $post->ID;
				$product      = wc_get_product( $product_id );
				$product_type = $product->get_type();
				if ( 'variable' === $product_type ) {
					?>
						<div class="aicoso_bulk_prod_price_per_qty_table"></div>
					<?php
				} else {
					$pqp_qty_enable = get_post_meta( $product_id, '_aicoso_bulk_prod_price_per_quantity', true );
					if ( isset( $pqp_qty_enable ) && 'yes' === $pqp_qty_enable ) {
						$pqp_table_data = get_post_meta( $product_id, '_aicoso_bulk_prod_price_quantity_pricing', true );
						if ( isset( $pqp_table_data ) && ! empty( $pqp_table_data ) ) {
							// Check if custom colors are set to determine table ID
							$iwe_thead_border = get_option( 'aicoso_bulk_prod_price_table_thead_border', '' );
							$iwe_thead_bg     = get_option( 'aicoso_bulk_prod_price_table_thead_bg', '' );
							$iwe_thead_text   = get_option( 'aicoso_bulk_prod_price_table_thead_text', '' );
							$iwe_tbody_border = get_option( 'aicoso_bulk_prod_price_table_tbody_border', '' );
							$iwe_tbody_bg     = get_option( 'aicoso_bulk_prod_price_table_tbody_bg', '' );
							$iwe_tbody_text   = get_option( 'aicoso_bulk_prod_price_table_tbody_text', '' );

							// Determine if we have custom styles
							$has_custom_styles = false;
							$table_id          = '';

							if ( ! empty( $iwe_thead_border ) || ! empty( $iwe_thead_bg ) || ! empty( $iwe_thead_text ) ||
								! empty( $iwe_tbody_border ) || ! empty( $iwe_tbody_bg ) || ! empty( $iwe_tbody_text ) ) {
								$has_custom_styles = true;
								$table_id          = 'aicoso_bulk_prod_price_table_' . $product_id;
							}

							?>
								<div class="aicoso_bulk_prod_price_per_qty_table">
									<table class="aicoso_bulk_prod_price_table"
									<?php
									if ( $has_custom_styles ) {
										echo 'id="' . esc_attr( $table_id ) . '"';
									}
									?>
									>
										<thead>
											<th><?php esc_html_e( 'Quantities', 'bulk-products-pricing' ); ?></th>
											<th><?php esc_html_e( 'Price', 'bulk-products-pricing' ); ?></th>
											<th><?php esc_html_e( 'Pricing Type', 'bulk-products-pricing' ); ?></th>
										</thead>
										<tbody>
											<?php
											foreach ( $pqp_table_data['min'] as $key1 => $value1 ) {
												?>
														<tr>
															<td>
															<?php
																echo esc_html( $value1 . '-' . $pqp_table_data['max'][ $key1 ] );
															?>
															</td>
															<td>
															<?php
															if ( 'pqp_perc_disc' === $pqp_table_data['type'][ $key1 ] ) {
																echo esc_html( $pqp_table_data['price'][ $key1 ] ) . '%';
															} else {
																echo esc_html( get_woocommerce_currency_symbol() ) . esc_html( $pqp_table_data['price'][ $key1 ] );
															}
															?>
															</td>
															<td>
																<?php
																if ( 'pqp_sell_price' === $pqp_table_data['type'][ $key1 ] ) {
																	esc_html_e( 'Selling Price', 'bulk-products-pricing' );
																} elseif ( 'pqp_fixed_disc' === $pqp_table_data['type'][ $key1 ] ) {
																	esc_html_e( 'Fixed Discount', 'bulk-products-pricing' );
																} elseif ( 'pqp_perc_disc' === $pqp_table_data['type'][ $key1 ] ) {
																	esc_html_e( 'Percentage Discount', 'bulk-products-pricing' );
																}
																?>
															</td>
														</tr>
													<?php
											}
											?>
										</tbody>
									</table>
								</div>
							<?php
						}
					}
				}
			}
		}
		public function aicoso_bulk_prod_price_woo_before_calculate_totals( $cart ) {
			// Remove any actions to prevent infinite loops
			if ( did_action( 'woocommerce_before_calculate_totals' ) >= 2 ) {
				return;
			}

			// Check if this is a valid cart object
			if ( ! $cart || ! method_exists( $cart, 'get_cart' ) ) {
				return;
			}

			// Ensure we're working with the latest cart data
			if ( is_admin() && ! wp_doing_ajax() ) {
				return;
			}

			$tax_calc = get_option( 'aicoso_bulk_prod_price_tax_enable', 'on' );
			$is_tax   = false;
			if ( 'on' === $tax_calc ) {
				$is_tax = true;
			}

			foreach ( $cart->get_cart() as $cart_item_key => $cart_item ) {
				$product_id = $cart_item['product_id'];
				$product    = $cart_item['data'];

				if ( ! $product ) {
					continue;
				}

				$product_type = $product->get_type();

				if ( 'variation' === $product_type ) {
					$product_id = $cart_item['variation_id'];
				}

				$qty            = $cart_item['quantity'];
				$pqp_qty_enable = get_post_meta( $product_id, '_aicoso_bulk_prod_price_per_quantity', true );

				if ( isset( $pqp_qty_enable ) && 'yes' === $pqp_qty_enable ) {
					$pqp_table_data = get_post_meta( $product_id, '_aicoso_bulk_prod_price_quantity_pricing', true );

					if ( isset( $pqp_table_data ) && ! empty( $pqp_table_data ) && isset( $pqp_table_data['min'] ) ) {
						$pqp_price = 0;
						$flag      = false;
						$type      = '';

						foreach ( $pqp_table_data['min'] as $key1 => $value1 ) {
							// Convert to integers for proper numeric comparison
							$min_qty  = intval( $value1 );
							$max_qty  = intval( $pqp_table_data['max'][ $key1 ] );
							$cart_qty = intval( $qty );

							if ( $cart_qty >= $min_qty && $cart_qty <= $max_qty ) {
								$pqp_price = floatval( $pqp_table_data['price'][ $key1 ] );
								$type      = $pqp_table_data['type'][ $key1 ];
								$flag      = true;
								break;
							}
						}

						// Get the original product price from the database
						$original_product = wc_get_product( $product_id );

						// Use sale price if available, otherwise use regular price
						if ( $original_product && $original_product->is_on_sale() && $original_product->get_sale_price() ) {
							$original_price = floatval( $original_product->get_sale_price() );
						} else {
							$original_price = $original_product ? floatval( $original_product->get_regular_price() ) : floatval( $product->get_regular_price() );
						}

						// Fallback to get_price() if no price is set
						if ( empty( $original_price ) || 0 === $original_price ) {
							$original_price = $original_product ? floatval( $original_product->get_price() ) : floatval( $product->get_price() );
						}

						if ( $flag ) {
							// Apply tiered pricing
							$new_price = 0;

							if ( 'pqp_sell_price' === $type ) {
								// Direct selling price
								$new_price = floatval( $pqp_price );
							} elseif ( 'pqp_fixed_disc' === $type ) {
								// Fixed discount from original price
								$discount_amount = floatval( $pqp_price );
								$new_price       = max( 0, $original_price - $discount_amount );
							} elseif ( 'pqp_perc_disc' === $type ) {
								// Percentage discount from original price
								$discount_percent = floatval( $pqp_price );
								$discount_amount  = ( $original_price * $discount_percent ) / 100;
								$new_price        = max( 0, $original_price - $discount_amount );
							}

							// Set the new price on the product object in the cart
							// Allow price to be 0 for free products
							if ( $new_price >= 0 ) {
								$product->set_price( $new_price );
							} else {
								// If somehow negative, set to 0
								$product->set_price( 0 );
							}
						} else {
							// No matching tier found - revert to original price
							// This handles quantities outside defined ranges
							$product->set_price( $original_price );
						}
					}
				}
			}
		}
		public function aicoso_bulk_prod_price_single_qty() {
			check_ajax_referer( '_wpdocs-special-string', 'security' );
			$response['result'] = false;
			// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.MissingUnslash -- Sanitized on next lines
			if ( isset( $_POST['product_id'] ) && ! empty( $_POST['product_id'] ) && isset( $_POST['qty'] ) && ! empty( $_POST['qty'] ) ) {
				$product_id     = sanitize_text_field( wp_unslash( $_POST['product_id'] ) );
				$product        = wc_get_product( $product_id );
				$qty            = sanitize_text_field( wp_unslash( $_POST['qty'] ) );
				$pqp_table_data = get_post_meta( $product_id, '_aicoso_bulk_prod_price_quantity_pricing', true );

				if ( isset( $pqp_table_data ) && ! empty( $pqp_table_data ) ) {
					$tax_calc = get_option( 'aicoso_bulk_prod_price_tax_enable', 'on' );

					$is_tax = false;
					if ( 'on' === $tax_calc ) {
						$is_tax = true;
					}
					// Get custom colors or use beautiful defaults
					$iwe_thead_border = get_option( 'aicoso_bulk_prod_price_table_thead_border', '' );
					$iwe_thead_bg     = get_option( 'aicoso_bulk_prod_price_table_thead_bg', '' );
					$iwe_thead_text   = get_option( 'aicoso_bulk_prod_price_table_thead_text', '' );
					$iwe_tbody_border = get_option( 'aicoso_bulk_prod_price_table_tbody_border', '' );
					$iwe_tbody_bg     = get_option( 'aicoso_bulk_prod_price_table_tbody_bg', '' );
					$iwe_tbody_text   = get_option( 'aicoso_bulk_prod_price_table_tbody_text', '' );

					// Build styles for custom colors
					$has_custom_styles = false;
					$custom_css        = '';

					if ( ! empty( $iwe_thead_border ) || ! empty( $iwe_thead_bg ) || ! empty( $iwe_thead_text ) ||
						! empty( $iwe_tbody_border ) || ! empty( $iwe_tbody_bg ) || ! empty( $iwe_tbody_text ) ) {
						$has_custom_styles = true;
						$table_id          = 'aicoso_bulk_prod_price_table_ajax_' . $product_id;
						// Use helper method to generate CSS
						$custom_css        = $this->aicoso_bulk_prod_price_generate_table_css( $table_id, $iwe_thead_border, $iwe_thead_bg, $iwe_thead_text, $iwe_tbody_border, $iwe_tbody_bg, $iwe_tbody_text );
					}
					$pqp_price = 0;
					$flag      = false;
					$type      = '';
					$html      = '';

					foreach ( $pqp_table_data['min'] as $key => $value ) {
						// Convert to integers for proper numeric comparison
						$min_qty   = intval( $value );
						$max_qty   = intval( $pqp_table_data['max'][ $key ] );
						$input_qty = intval( $qty );

						if ( $input_qty >= $min_qty && $input_qty <= $max_qty ) {
							$pqp_price = floatval( $pqp_table_data['price'][ $key ] );
							$type      = $pqp_table_data['type'][ $key ];
							$flag      = true;
							break;
						}
					}
					$ftable_enable = get_option( 'aicoso_bulk_prod_price_ftable_enable', 'off' );
					if ( 'on' === $ftable_enable ) {
						if ( $has_custom_styles ) {
							$html .= $custom_css;
						}
						$table_attrs = $has_custom_styles ? ' id="' . esc_attr( $table_id ) . '"' : '';
						$html       .= '<table class="aicoso_bulk_prod_price_table"' . $table_attrs . '><thead><th>' . esc_html__( 'Quantities', 'bulk-products-pricing' ) . '</th><th>' . esc_html__( 'Price', 'bulk-products-pricing' ) . '</th><th>' . esc_html__( 'Pricing Type', 'bulk-products-pricing' ) . '</th></thead><tbody>';
						foreach ( $pqp_table_data['min'] as $key => $value ) {
							$html .= '<tr><td>' . esc_html( $value ) . '-' . esc_html( $pqp_table_data['max'][ $key ] ) . '</td><td>';

							if ( 'pqp_perc_disc' === $pqp_table_data['type'][ $key ] ) {
								$html .= esc_html( $pqp_table_data['price'][ $key ] ) . '%';
							} else {
								$html .= esc_html( get_woocommerce_currency_symbol() ) . esc_html( $pqp_table_data['price'][ $key ] );
							}

							$html .= '</td><td>';

							if ( 'pqp_sell_price' === $pqp_table_data['type'][ $key ] ) {
								$html .= esc_html__( 'Selling Price', 'bulk-products-pricing' );
							} elseif ( 'pqp_fixed_disc' === $pqp_table_data['type'][ $key ] ) {
								$html .= esc_html__( 'Fixed Discount', 'bulk-products-pricing' );
							} elseif ( 'pqp_perc_disc' === $pqp_table_data['type'][ $key ] ) {
								$html .= esc_html__( 'Percentage Discount', 'bulk-products-pricing' );
							}
							$html .= '</td></tr>';
						}
						$html .= '</tbody></table>';
					}
					if ( $flag ) {

						if ( 'pqp_sell_price' === $type ) {
							if ( WC()->version < '3.0.0' ) {
								if ( $is_tax ) {
									$price = $product->get_display_price( $pqp_price, 1 );
								} else {
									$price = $pqp_price;
								}
								$price              = wc_price( $price );
								$response['result'] = true;
							} else {
								if ( $is_tax ) {
									$args  = array(
										'qty'   => 1,
										'price' => $pqp_price,
									);
									$price = wc_get_price_to_display( $product, $args );
								} else {
									$price = $pqp_price;
								}
								$price              = wc_price( $price );
								$response['result'] = true;
							}
						} elseif ( 'pqp_fixed_disc' === $type ) {
							// Get original price for calculation - use sale price if available
							if ( $product->is_on_sale() && $product->get_sale_price() ) {
								$original_price = floatval( $product->get_sale_price() );
							} else {
								$original_price = floatval( $product->get_regular_price() );
							}

							if ( empty( $original_price ) || 0 === $original_price ) {
								$original_price = floatval( $product->get_price() );
							}

							// Calculate discounted price
							$discounted_price = max( 0, $original_price - $pqp_price );

							if ( WC()->version < '3.0.0' ) {
								if ( $is_tax ) {
									$price = $product->get_display_price( $discounted_price, 1 );
								} else {
									$price = $discounted_price;
								}
								$price              = wc_price( $price );
								$response['result'] = true;
							} else {
								if ( $is_tax ) {
									$args  = array(
										'qty'   => 1,
										'price' => $discounted_price,
									);
									$price = wc_get_price_to_display( $product, $args );
								} else {
									$price = $discounted_price;
								}
								$price              = wc_price( $price );
								$response['result'] = true;
							}
						} elseif ( 'pqp_perc_disc' === $type ) {
							// Get original price for calculation - use sale price if available
							if ( $product->is_on_sale() && $product->get_sale_price() ) {
								$original_price = floatval( $product->get_sale_price() );
							} else {
								$original_price = floatval( $product->get_regular_price() );
							}

							if ( empty( $original_price ) || 0 === $original_price ) {
								$original_price = floatval( $product->get_price() );
							}

							// Calculate discounted price
							$discount_amount  = ( $original_price * $pqp_price ) / 100;
							$discounted_price = max( 0, $original_price - $discount_amount );

							if ( WC()->version < '3.0.0' ) {
								if ( $is_tax ) {
									$price = $product->get_display_price( $discounted_price, 1 );
								} else {
									$price = $discounted_price;
								}
								$price              = wc_price( $price );
								$response['result'] = true;
							} else {
								if ( $is_tax ) {
									$args  = array(
										'qty'   => 1,
										'price' => $discounted_price,
									);
									$price = wc_get_price_to_display( $product, $args );
								} else {
									$price = $discounted_price;
								}
								$price              = wc_price( $price );
								$response['result'] = true;
							}
						}
						$response['data'] = $price;
					} else {
						$response['data'] = wc_price( $product->get_price() );
					}
					$ftable_enable = get_option( 'aicoso_bulk_prod_price_ftable_enable', 'off' );
					if ( 'on' === $ftable_enable ) {
						$response['html'] = $html;
					}
				}
			}
			wp_send_json( $response );
		}
		public function aicoso_bulk_prod_price_wp_enqueue_scripts() {
			if ( is_product() ) {
				global $post;
				$product_id   = $post->ID;
				$product      = wc_get_product( $product_id );
				$product_type = $product->get_type();
				$childarr     = array();
				$aicoso_pqp   = array();
				if ( 'variable' === $product_type ) {
					$childids = $product->get_children();
					foreach ( $childids as $key => $value ) {
						$pqp_qty_enable = get_post_meta( $value, '_aicoso_bulk_prod_price_per_quantity', true );
						if ( ! isset( $pqp_qty_enable ) || empty( $pqp_qty_enable ) ) {
							$pqp_qty_enable = 'no';
						}
						$childarr[ $value ] = $pqp_qty_enable;
					}
					$aicoso_pqp['var_arr'] = $childarr;
				} else {
					$pqp_qty_enable = get_post_meta( $product_id, '_aicoso_bulk_prod_price_per_quantity', true );
					if ( ! isset( $pqp_qty_enable ) || empty( $pqp_qty_enable ) ) {
						$pqp_qty_enable = 'no';
					}
					$aicoso_pqp['pqp_enable'] = $pqp_qty_enable;
				}

				$aicoso_pqp['ajaxurl']    = admin_url( 'admin-ajax.php' );
				$aicoso_pqp['product_id'] = $product_id;
				$aicoso_pqp['type']       = $product_type;
				$aicoso_pqp['ajax_nonce'] = wp_create_nonce( '_wpdocs-special-string' );
				wp_enqueue_script( 'aicoso_bulk_prod_price_product_single', AICOSO_BULK_PROD_PRICE_URL . 'assets/frontend/js/woo-per-qty-public-single.js', array( 'jquery', 'jquery-blockui' ), '1.0.1', true );
				wp_localize_script( 'aicoso_bulk_prod_price_product_single', 'aicoso_pqp', $aicoso_pqp );
				wp_enqueue_style( 'aicoso_bulk_prod_price_product_single_css', AICOSO_BULK_PROD_PRICE_URL . 'assets/frontend/css/woo-per-qty-public-single.css', array(), '1.0.0' );

				// Add dynamic custom colors CSS
				$this->aicoso_bulk_prod_price_add_custom_colors_css( $product_id );
			}

			// Add cart page script for automatic updates
			if ( is_cart() ) {
				wp_enqueue_script( 'aicoso_bulk_prod_price_cart_update', AICOSO_BULK_PROD_PRICE_URL . 'assets/frontend/js/woo-per-qty-cart-update.js', array( 'jquery', 'wc-cart' ), '1.0.0', true );
				$cart_params = array(
					'ajax_url'          => admin_url( 'admin-ajax.php' ),
					'update_cart_nonce' => wp_create_nonce( 'woocommerce-cart' ),
				);
				wp_localize_script( 'aicoso_bulk_prod_price_cart_update', 'aicoso_bulk_prod_price_cart', $cart_params );

				// Add inline script for immediate effect
				wp_add_inline_script(
					'aicoso_bulk_prod_price_cart_update',
					'
					jQuery(document).ready(function($) {
						// Force cart update on quantity change
						var pqpUpdateTimer;
						$(document).on("change", ".quantity input.qty", function() {
							clearTimeout(pqpUpdateTimer);
							var $button = $("[name=\'update_cart\']");
							pqpUpdateTimer = setTimeout(function() {
								$button.prop("disabled", false).attr("aria-disabled", false);
								$button.trigger("click");
							}, 500);
						});
					});
					'
				);
			}
		}
		public function aicoso_bulk_prod_price_woo_get_price_html( $price, $product ) {

			$product_type = $product->get_type();
			if ( 'variation' === $product_type ) {
				return '';
			}
			if ( 'simple' === $product_type ) {
				$product_id = $product->get_id();

				$pqp_qty_enable = get_post_meta( $product_id, '_aicoso_bulk_prod_price_per_quantity', true );
				if ( isset( $pqp_qty_enable ) && 'yes' === $pqp_qty_enable ) {
					$pqp_table_data = get_post_meta( $product_id, '_aicoso_bulk_prod_price_quantity_pricing', true );
					if ( isset( $pqp_table_data ) && ! empty( $pqp_table_data ) ) {

						$pqp_price = 0;
						$flag      = false;
						foreach ( $pqp_table_data['min'] as $key => $value ) {
							if ( 1 === $value ) {
								$pqp_price = $pqp_table_data['price'][ $key ];
								$pqp_type  = $pqp_table_data['type'][ $key ];
								$flag      = true;
								break;
							}
						}
						$tax_calc = get_option( 'aicoso_bulk_prod_price_tax_enable', 'on' );
						$is_tax   = false;
						if ( 'on' === $tax_calc ) {
							$is_tax = true;
						}
						if ( $flag ) {

							if ( 'pqp_sell_price' === $pqp_type ) {
								if ( WC()->version < '3.0.0' ) {
									if ( $is_tax ) {
										$price = $product->get_display_price( $pqp_price, 1 );
									} else {
										$price = $pqp_price;
									}
									$price = wc_price( $price );
								} else {
									if ( $is_tax ) {
										$args  = array(
											'qty'   => 1,
											'price' => $pqp_price,
										);
										$price = wc_get_price_to_display( $product, $args );
									} else {
										$price = $pqp_price;
									}
									$price = wc_price( $price );
								}
							} elseif ( 'pqp_fixed_disc' === $pqp_type ) {
								// Calculate discounted price - use sale price if available
								if ( $product->is_on_sale() && $product->get_sale_price() ) {
									$base_price = floatval( $product->get_sale_price() );
								} else {
									$base_price = floatval( $product->get_regular_price() );
								}
								if ( empty( $base_price ) || 0 === $base_price ) {
									$base_price = floatval( $product->get_price() );
								}
								$discounted_price = max( 0, $base_price - $pqp_price );

								if ( WC()->version < '3.0.0' ) {
									if ( $is_tax ) {
										$price = $product->get_display_price( $discounted_price, 1 );
									} else {
										$price = $discounted_price;
									}
									$price = wc_price( $price );
								} else {
									if ( $is_tax ) {
										$args  = array(
											'qty'   => 1,
											'price' => $discounted_price,
										);
										$price = wc_get_price_to_display( $product, $args );
									} else {
										$price = $discounted_price;
									}
									$price = wc_price( $price );
								}
							} elseif ( 'pqp_perc_disc' === $pqp_type ) {
								// Calculate discounted price - use sale price if available
								if ( $product->is_on_sale() && $product->get_sale_price() ) {
									$base_price = floatval( $product->get_sale_price() );
								} else {
									$base_price = floatval( $product->get_regular_price() );
								}
								if ( empty( $base_price ) || 0 === $base_price ) {
									$base_price = floatval( $product->get_price() );
								}
								$discount_amount  = ( $base_price * $pqp_price ) / 100;
								$discounted_price = max( 0, $base_price - $discount_amount );

								if ( WC()->version < '3.0.0' ) {
									if ( $is_tax ) {
										$price = $product->get_display_price( $discounted_price, 1 );
									} else {
										$price = $discounted_price;
									}
									$price = wc_price( $price );
								} else {
									if ( $is_tax ) {
										$args  = array(
											'qty'   => 1,
											'price' => $discounted_price,
										);
										$price = wc_get_price_to_display( $product, $args );
									} else {
										$price = $discounted_price;
									}
									$price = wc_price( $price );
								}
							}
						}
					}
				}
			}
			return $price;
		}

		/**
		 * Display the correct price in cart for items with quantity pricing
		 */
		public function aicoso_bulk_prod_price_cart_item_price( $price, $cart_item, $cart_item_key ) {
			$product = $cart_item['data'];

			// Get the current price from the product object (already set by calculate_totals)
			if ( $product && method_exists( $product, 'get_price' ) ) {
				$item_price = $product->get_price();
				if ( $item_price ) {
					return wc_price( $item_price );
				}
			}

			return $price;
		}

		/**
		 * Display the correct subtotal in cart for items with quantity pricing
		 */
		public function aicoso_bulk_prod_price_cart_item_subtotal( $subtotal, $cart_item, $cart_item_key ) {
			$product  = $cart_item['data'];
			$quantity = $cart_item['quantity'];

			// Get the current price from the product object (already set by calculate_totals)
			if ( $product && method_exists( $product, 'get_price' ) ) {
				$item_price = $product->get_price();
				if ( $item_price ) {
					return wc_price( $item_price * $quantity );
				}
			}

			return $subtotal;
		}

		/**
		 * Apply pricing when cart is loaded from session
		 */
		public function aicoso_bulk_prod_price_cart_loaded_from_session( $cart ) {
			// Recalculate totals to apply quantity pricing
			if ( ! empty( $cart->get_cart() ) ) {
				$cart->calculate_totals();
			}
		}

		/**
		 * Handle quantity update to immediately apply new pricing
		 */
		public function aicoso_bulk_prod_price_after_quantity_update( $cart_item_key, $quantity, $old_quantity, $cart ) {
			// Get the cart item
			$cart_item = $cart->get_cart_item( $cart_item_key );
			if ( ! $cart_item ) {
				return;
			}

			$product_id = $cart_item['product_id'];
			$product    = $cart_item['data'];

			if ( ! $product ) {
				return;
			}

			// Check if variation
			if ( $product->get_type() === 'variation' ) {
				$product_id = $cart_item['variation_id'];
			}

			// Check if quantity pricing is enabled
			$pqp_qty_enable = get_post_meta( $product_id, '_aicoso_bulk_prod_price_per_quantity', true );

			if ( isset( $pqp_qty_enable ) && 'yes' === $pqp_qty_enable ) {
				$pqp_table_data = get_post_meta( $product_id, '_aicoso_bulk_prod_price_quantity_pricing', true );

				if ( isset( $pqp_table_data ) && ! empty( $pqp_table_data ) && isset( $pqp_table_data['min'] ) ) {
					$pqp_price = 0;
					$flag      = false;
					$type      = '';

					// Find the matching price tier for the NEW quantity
					foreach ( $pqp_table_data['min'] as $key1 => $value1 ) {
						$min_qty = intval( $value1 );
						$max_qty = intval( $pqp_table_data['max'][ $key1 ] );
						$new_qty = intval( $quantity );

						if ( $new_qty >= $min_qty && $new_qty <= $max_qty ) {
							$pqp_price = floatval( $pqp_table_data['price'][ $key1 ] );
							$type      = $pqp_table_data['type'][ $key1 ];
							$flag      = true;
							break;
						}
					}

					// Get original price - use sale price if available
					$original_product = wc_get_product( $product_id );

					if ( $original_product && $original_product->is_on_sale() && $original_product->get_sale_price() ) {
						$original_price = floatval( $original_product->get_sale_price() );
					} else {
						$original_price = $original_product ? floatval( $original_product->get_regular_price() ) : floatval( $product->get_regular_price() );
					}

					if ( empty( $original_price ) || 0 === $original_price ) {
						$original_price = $original_product ? floatval( $original_product->get_price() ) : floatval( $product->get_price() );
					}

					if ( $flag ) {
						// Apply tiered pricing
						$new_price = 0;

						if ( 'pqp_sell_price' === $type ) {
							// Direct selling price
							$new_price = floatval( $pqp_price );
						} elseif ( 'pqp_fixed_disc' === $type ) {
							// Fixed discount from original price
							$discount_amount = floatval( $pqp_price );
							$new_price       = max( 0, $original_price - $discount_amount );
						} elseif ( 'pqp_perc_disc' === $type ) {
							// Percentage discount from original price
							$discount_percent = floatval( $pqp_price );
							$discount_amount  = ( $original_price * $discount_percent ) / 100;
							$new_price        = max( 0, $original_price - $discount_amount );
						}

						// Set the new price immediately
						// Allow price to be 0 for free products
						if ( $new_price >= 0 ) {
							$product->set_price( $new_price );
						} else {
							// If somehow negative, set to 0
							$product->set_price( 0 );
						}
					} else {
						// No matching tier - revert to original price
						$product->set_price( $original_price );
					}

					// Force cart to recalculate
					WC()->cart->calculate_totals();
				}
			}
		}

		/**
		 * Generate table CSS with custom colors
		 *
		 * @param string $table_id Table ID.
		 * @param string $thead_border Header border color.
		 * @param string $thead_bg Header background color.
		 * @param string $thead_text Header text color.
		 * @param string $tbody_border Body border color.
		 * @param string $tbody_bg Body background color.
		 * @param string $tbody_text Body text color.
		 * @return string Generated CSS.
		 */
		private function aicoso_bulk_prod_price_generate_table_css( $table_id, $thead_border, $thead_bg, $thead_text, $tbody_border, $tbody_bg, $tbody_text ) {
			$custom_css = '<style>';

			// Apply thead styles
			if ( ! empty( $thead_bg ) ) {
				$custom_css .= '.aicoso_bulk_prod_price_per_qty_table #' . esc_attr( $table_id ) . ' thead { background: ' . esc_attr( $thead_bg ) . ' !important; }';
				$custom_css .= '.aicoso_bulk_prod_price_per_qty_table #' . esc_attr( $table_id ) . ' thead th { background: transparent !important; }';
			}
			if ( ! empty( $thead_text ) ) {
				$custom_css .= '.aicoso_bulk_prod_price_per_qty_table #' . esc_attr( $table_id ) . ' thead th { color: ' . esc_attr( $thead_text ) . ' !important; }';
			}
			if ( ! empty( $thead_border ) ) {
				$custom_css .= '.aicoso_bulk_prod_price_per_qty_table #' . esc_attr( $table_id ) . ' thead th { border-bottom: 2px solid ' . esc_attr( $thead_border ) . ' !important; }';
				$custom_css .= '.aicoso_bulk_prod_price_per_qty_table #' . esc_attr( $table_id ) . ' thead th:not(:last-child) { border-right: 1px solid ' . esc_attr( $thead_border ) . ' !important; }';
			}

			// Apply tbody styles
			if ( ! empty( $tbody_bg ) ) {
				$custom_css .= '.aicoso_bulk_prod_price_per_qty_table #' . esc_attr( $table_id ) . ' tbody { background: ' . esc_attr( $tbody_bg ) . ' !important; }';
				$custom_css .= '.aicoso_bulk_prod_price_per_qty_table #' . esc_attr( $table_id ) . ' tbody tr { background: ' . esc_attr( $tbody_bg ) . ' !important; }';
				$custom_css .= '.aicoso_bulk_prod_price_per_qty_table #' . esc_attr( $table_id ) . ' tbody td { background: ' . esc_attr( $tbody_bg ) . ' !important; }';
				$custom_css .= '.aicoso_bulk_prod_price_per_qty_table #' . esc_attr( $table_id ) . ' tbody tr:hover { transform: none !important; box-shadow: none !important; }';
				$custom_css .= '.aicoso_bulk_prod_price_per_qty_table #' . esc_attr( $table_id ) . ' tbody tr:hover td { background: ' . esc_attr( $tbody_bg ) . ' !important; filter: brightness(0.92); }';
			}
			if ( ! empty( $tbody_text ) ) {
				$custom_css .= '.aicoso_bulk_prod_price_per_qty_table #' . esc_attr( $table_id ) . ' tbody td { color: ' . esc_attr( $tbody_text ) . ' !important; }';
				$custom_css .= '.aicoso_bulk_prod_price_per_qty_table #' . esc_attr( $table_id ) . ' tbody td:first-child { color: ' . esc_attr( $tbody_text ) . ' !important; }';
				$custom_css .= '.aicoso_bulk_prod_price_per_qty_table #' . esc_attr( $table_id ) . ' tbody td:nth-child(2) { color: ' . esc_attr( $tbody_text ) . ' !important; }';
				$custom_css .= '.aicoso_bulk_prod_price_per_qty_table #' . esc_attr( $table_id ) . ' tbody td:nth-child(3) { color: ' . esc_attr( $tbody_text ) . ' !important; }';
			}
			if ( ! empty( $tbody_border ) ) {
				$custom_css .= '.aicoso_bulk_prod_price_per_qty_table #' . esc_attr( $table_id ) . ' tbody tr { border-bottom: 1px solid ' . esc_attr( $tbody_border ) . ' !important; }';
				$custom_css .= '.aicoso_bulk_prod_price_per_qty_table #' . esc_attr( $table_id ) . ' tbody td { border-right: 1px solid ' . esc_attr( $tbody_border ) . ' !important; }';
				$custom_css .= '.aicoso_bulk_prod_price_per_qty_table #' . esc_attr( $table_id ) . ' { border: 1px solid ' . esc_attr( $tbody_border ) . ' !important; }';
			}

			$custom_css .= '</style>';
			return $custom_css;
		}

		/**
		 * Generate and add custom CSS for pricing table colors
		 *
		 * @param int $product_id Product ID.
		 */
		private function aicoso_bulk_prod_price_add_custom_colors_css( $product_id ) {
			// Get custom colors
			$iwe_thead_border = get_option( 'aicoso_bulk_prod_price_table_thead_border', '' );
			$iwe_thead_bg     = get_option( 'aicoso_bulk_prod_price_table_thead_bg', '' );
			$iwe_thead_text   = get_option( 'aicoso_bulk_prod_price_table_thead_text', '' );
			$iwe_tbody_border = get_option( 'aicoso_bulk_prod_price_table_tbody_border', '' );
			$iwe_tbody_bg     = get_option( 'aicoso_bulk_prod_price_table_tbody_bg', '' );
			$iwe_tbody_text   = get_option( 'aicoso_bulk_prod_price_table_tbody_text', '' );

			// Only generate CSS if custom colors are set
			if ( ! empty( $iwe_thead_border ) || ! empty( $iwe_thead_bg ) || ! empty( $iwe_thead_text ) ||
				! empty( $iwe_tbody_border ) || ! empty( $iwe_tbody_bg ) || ! empty( $iwe_tbody_text ) ) {

				$table_id   = 'aicoso_bulk_prod_price_table_' . $product_id;
				$custom_css = '';

				// Apply thead styles
				if ( ! empty( $iwe_thead_bg ) ) {
					$custom_css .= '.aicoso_bulk_prod_price_per_qty_table #' . $table_id . ' thead { background: ' . esc_attr( $iwe_thead_bg ) . ' !important; }';
					$custom_css .= '.aicoso_bulk_prod_price_per_qty_table #' . $table_id . ' thead th { background: transparent !important; }';
				}
				if ( ! empty( $iwe_thead_text ) ) {
					$custom_css .= '.aicoso_bulk_prod_price_per_qty_table #' . $table_id . ' thead th { color: ' . esc_attr( $iwe_thead_text ) . ' !important; }';
				}
				if ( ! empty( $iwe_thead_border ) ) {
					$custom_css .= '.aicoso_bulk_prod_price_per_qty_table #' . $table_id . ' thead th { border-bottom: 2px solid ' . esc_attr( $iwe_thead_border ) . ' !important; }';
					$custom_css .= '.aicoso_bulk_prod_price_per_qty_table #' . $table_id . ' thead th:not(:last-child) { border-right: 1px solid ' . esc_attr( $iwe_thead_border ) . ' !important; }';
				}

				// Apply tbody styles
				if ( ! empty( $iwe_tbody_bg ) ) {
					$custom_css .= '.aicoso_bulk_prod_price_per_qty_table #' . $table_id . ' tbody { background: ' . esc_attr( $iwe_tbody_bg ) . ' !important; }';
					$custom_css .= '.aicoso_bulk_prod_price_per_qty_table #' . $table_id . ' tbody tr { background: ' . esc_attr( $iwe_tbody_bg ) . ' !important; }';
					$custom_css .= '.aicoso_bulk_prod_price_per_qty_table #' . $table_id . ' tbody td { background: ' . esc_attr( $iwe_tbody_bg ) . ' !important; }';
					$custom_css .= '.aicoso_bulk_prod_price_per_qty_table #' . $table_id . ' tbody tr:hover { transform: none !important; box-shadow: none !important; }';
					$custom_css .= '.aicoso_bulk_prod_price_per_qty_table #' . $table_id . ' tbody tr:hover td { background: ' . esc_attr( $iwe_tbody_bg ) . ' !important; filter: brightness(0.92); }';
				}
				if ( ! empty( $iwe_tbody_text ) ) {
					$custom_css .= '.aicoso_bulk_prod_price_per_qty_table #' . $table_id . ' tbody td { color: ' . esc_attr( $iwe_tbody_text ) . ' !important; }';
					$custom_css .= '.aicoso_bulk_prod_price_per_qty_table #' . $table_id . ' tbody td:first-child { color: ' . esc_attr( $iwe_tbody_text ) . ' !important; }';
					$custom_css .= '.aicoso_bulk_prod_price_per_qty_table #' . $table_id . ' tbody td:nth-child(2) { color: ' . esc_attr( $iwe_tbody_text ) . ' !important; }';
					$custom_css .= '.aicoso_bulk_prod_price_per_qty_table #' . $table_id . ' tbody td:nth-child(3) { color: ' . esc_attr( $iwe_tbody_text ) . ' !important; }';
				}
				if ( ! empty( $iwe_tbody_border ) ) {
					$custom_css .= '.aicoso_bulk_prod_price_per_qty_table #' . $table_id . ' tbody tr { border-bottom: 1px solid ' . esc_attr( $iwe_tbody_border ) . ' !important; }';
					$custom_css .= '.aicoso_bulk_prod_price_per_qty_table #' . $table_id . ' tbody td { border-right: 1px solid ' . esc_attr( $iwe_tbody_border ) . ' !important; }';
					$custom_css .= '.aicoso_bulk_prod_price_per_qty_table #' . $table_id . ' { border: 1px solid ' . esc_attr( $iwe_tbody_border ) . ' !important; }';
				}

				// Add inline style using WordPress standard method
				wp_add_inline_style( 'aicoso_bulk_prod_price_product_single_css', $custom_css );
			}
		}
	}

	new Aicoso_Bulk_Prod_Price_Public();
}
