<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}
/**
 * Plugin installer class
 */
class BulkDe_Installer {

	/**
	 * Install plugin
	 */
	public function install() {
		$this->migrate_from_old_prefix();
		$this->create_tables();
		$this->set_default_options();
	}

	/**
	 * Uninstall plugin
	 */
	public function uninstall() {
		$this->drop_tables();
		$this->delete_options();
	}

	/**
	 * Migrate from old prefix (bpd_) to new prefix (bulkde_)
	 */
	private function migrate_from_old_prefix() {
		global $wpdb;

		$old_table_name = $wpdb->prefix . 'bpd_sites';
		$new_table_name = $wpdb->prefix . 'bulkde_sites';

		// Check if old table exists and new table doesn't
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Migration query only runs once
		$old_table_exists = $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $old_table_name ) );
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Migration query only runs once
		$new_table_exists = $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $new_table_name ) );

		if ( $old_table_exists && ! $new_table_exists ) {
			// Rename the table
			$escaped_old_table_name = esc_sql( $old_table_name );
			$escaped_new_table_name = esc_sql( $new_table_name );
			// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared,WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.DirectDatabaseQuery.SchemaChange -- Table names escaped with esc_sql(), migration query
			$wpdb->query( "RENAME TABLE `{$escaped_old_table_name}` TO `{$escaped_new_table_name}`" );
		}

		// Migrate options
		$old_settings = get_option( 'bpd_settings' );
		if ( $old_settings && ! get_option( 'bulkde_settings' ) ) {
			update_option( 'bulkde_settings', $old_settings );
			delete_option( 'bpd_settings' );
		}

		$old_db_version = get_option( 'bpd_db_version' );
		if ( $old_db_version && ! get_option( 'bulkde_db_version' ) ) {
			update_option( 'bulkde_db_version', $old_db_version );
			delete_option( 'bpd_db_version' );
		}
	}

	/**
	 * Create database tables
	 */
	private function create_tables() {
		global $wpdb;

		$table_name = $wpdb->prefix . 'bulkde_sites';

		$charset_collate = $wpdb->get_charset_collate();

		$sql = "CREATE TABLE $table_name (
            id mediumint(9) NOT NULL AUTO_INCREMENT,
            name varchar(255) NOT NULL,
            url varchar(255) NOT NULL,
            ftp_host varchar(255) NOT NULL,
            ftp_port int(5) DEFAULT 21,
            ftp_username varchar(255) NOT NULL,
            ftp_password text NOT NULL,
            ftp_path varchar(255) DEFAULT '/wp-content/plugins/',
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY name (name),
            KEY url (url)
        ) $charset_collate;";

		require_once ABSPATH . 'wp-admin/includes/upgrade.php';
		dbDelta( $sql );

		// Add version to options
		add_option( 'bulkde_db_version', '1.0.0' );
	}

	/**
	 * Drop database tables
	 */
	private function drop_tables() {
		global $wpdb;

		$table_name = $wpdb->prefix . 'bulkde_sites';
		$escaped_table_name = esc_sql( $table_name );
		// phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared,WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching,WordPress.DB.DirectDatabaseQuery.SchemaChange -- Table name escaped with esc_sql(), direct query required for uninstall, no caching needed for DROP TABLE, schema change required for plugin uninstall
		$wpdb->query( "DROP TABLE IF EXISTS `{$escaped_table_name}`" );
	}

	/**
	 * Set default options
	 */
	private function set_default_options() {
		add_option(
			'bulkde_settings',
			array(
				'max_execution_time' => 300,
				'memory_limit'       => '256M',
				'enable_logging'     => true,
				'log_level'          => 'info',
			)
		);
	}

	/**
	 * Delete options
	 */
	private function delete_options() {
		delete_option( 'bulkde_settings' );
		delete_option( 'bulkde_db_version' );
	}
}
