<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}
/**
 * Admin interface class
 */
class BulkDe_Admin {

	/**
	 * Initialize admin functionality
	 */
	public function init() {
		add_action( 'admin_menu', array( $this, 'add_admin_menu' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_scripts' ) );

		// Add AJAX handlers for import/export
		add_action( 'wp_ajax_bulkde_export_sites', array( $this, 'handle_export_sites' ) );
		add_action( 'wp_ajax_bulkde_import_sites', array( $this, 'handle_import_sites' ) );
	}

	/**
	 * Add admin menu
	 */
	public function add_admin_menu() {
		add_menu_page(
			'Bulk Deployer',
			'Bulk Deployer',
			'manage_options',
			'bulk-deployer',
			array( $this, 'render_admin_page' ),
			'dashicons-upload',
			30
		);
	}

	/**
	 * Enqueue admin scripts and styles
	 */
	public function enqueue_admin_scripts( $hook ) {
		if ( 'toplevel_page_bulk-deployer' !== $hook ) {
			return;
		}

		wp_enqueue_script(
			'bulkde-admin-js',
			BULKDE_PLUGIN_URL . 'assets/js/admin.js',
			array( 'jquery' ),
			BULKDE_PLUGIN_VERSION,
			true
		);

		wp_enqueue_style(
			'bulkde-admin-css',
			BULKDE_PLUGIN_URL . 'assets/css/admin.css',
			array(),
			BULKDE_PLUGIN_VERSION
		);

		// Localize script for AJAX
		wp_localize_script(
			'bulkde-admin-js',
			'bulkde_ajax',
			array(
				'ajax_url' => admin_url( 'admin-ajax.php' ),
				'nonce'    => wp_create_nonce( 'bulkde_nonce' ),
				'strings'  => array(
					'deploying'          => __( 'Deploying plugins...', 'bulk-deployer' ),
					'success'            => __( 'Deployment completed successfully!', 'bulk-deployer' ),
					'error'              => __( 'An error occurred during deployment.', 'bulk-deployer' ),
					'confirm_delete'     => __( 'Are you sure you want to delete this site?', 'bulk-deployer' ),
					'testing_connection' => __( 'Testing connection...', 'bulk-deployer' ),
					'connection_success' => __( 'Connection successful!', 'bulk-deployer' ),
					'connection_failed'  => __( 'Connection failed. Please check your credentials.', 'bulk-deployer' ),
					'exporting'          => __( 'Exporting sites...', 'bulk-deployer' ),
					'importing'          => __( 'Importing sites...', 'bulk-deployer' ),
					'export_success'     => __( 'Sites exported successfully!', 'bulk-deployer' ),
					'import_success'     => __( 'Sites imported successfully!', 'bulk-deployer' ),
					'import_error'       => __( 'Error importing sites. Please check the file format.', 'bulk-deployer' ),
					'confirm_import'     => __( 'This will import the sites. Existing sites with the same name may be updated. Continue?', 'bulk-deployer' ),
					'no_file_selected'   => __( 'Please select a file to import.', 'bulk-deployer' ),
					'invalid_file_type'  => __( 'Please select a valid JSON file.', 'bulk-deployer' ),
				),
			)
		);
	}

	/**
	 * Render admin page
	 */
	public function render_admin_page() {
		$site_manager = new BulkDe_Site_Manager();
		$sites        = $site_manager->get_sites();
		$plugins      = $this->get_available_plugins();

		include BULKDE_PLUGIN_DIR . 'templates/admin-page.php';
	}

	/**
	 * Get available plugins for deployment
	 */
	private function get_available_plugins() {
		$plugins = array();

		// Get plugins from plugins directory
		$plugins_dir = dirname( plugin_dir_path( BULKDE_PLUGIN_FILE ) );
		if ( is_dir( $plugins_dir ) ) {
			$plugin_dirs = glob( $plugins_dir . '/*', GLOB_ONLYDIR );

			foreach ( $plugin_dirs as $plugin_dir ) {
				$plugin_name = basename( $plugin_dir );
				$plugin_file = $plugin_dir . '/' . $plugin_name . '.php';

				if ( file_exists( $plugin_file ) ) {
					$plugin_data = get_plugin_data( $plugin_file );
					
					// Clean description - remove citation part (everything from <cite> onwards)
					$description = $plugin_data['Description'] ?: '';
					if ( ! empty( $description ) ) {
						// Remove HTML tags and citation
						$description = preg_replace( '/<cite>.*$/s', '', $description );
						$description = trim( $description );
					}
					
					$plugins[ $plugin_name ] = array(
						'name'        => $plugin_data['Name'] ?: $plugin_name,
						'description' => $description,
						'version'     => $plugin_data['Version'] ?: '1.0.0',
						'path'        => $plugin_dir,
					);
				}
			}
		}

		return $plugins;
	}

	/**
	 * Handle export sites AJAX request
	 */
	public function handle_export_sites() {
		check_ajax_referer( 'bulkde_nonce', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( 'Unauthorized' );
		}

		$include_passwords = isset( $_POST['include_passwords'] ) && $_POST['include_passwords'] === 'true';
		$selected_sites    = isset( $_POST['selected_sites'] ) ? json_decode( stripslashes( sanitize_text_field( wp_unslash( $_POST['selected_sites'] ) ) ), true ) : array();

		$site_manager = new BulkDe_Site_Manager();
		$all_sites    = $site_manager->get_sites();

		// Filter sites if specific ones were selected
		if ( ! empty( $selected_sites ) ) {
			$all_sites = array_filter(
				$all_sites,
				function ( $site ) use ( $selected_sites ) {
					return in_array( $site['id'], $selected_sites );
				}
			);
		}

		$export_data = array(
			'version'        => '1.0',
			'export_date'    => current_time( 'Y-m-d H:i:s' ),
			'export_options' => array(
				'include_passwords' => $include_passwords,
				'total_sites'       => count( $all_sites ),
			),
			'sites'          => array(),
		);

		foreach ( $all_sites as $site ) {
			$site_data = array(
				'name'         => $site['name'],
				'url'          => $site['url'],
				'ftp_host'     => $site['ftp_host'],
				'ftp_port'     => $site['ftp_port'],
				'ftp_username' => $site['ftp_username'],
				'ftp_path'     => $site['ftp_path'],
			);

			// Include password only if explicitly requested
			if ( $include_passwords ) {
				$site_data['ftp_password'] = $site_manager->decrypt_password( $site['ftp_password'] );
			}

			$export_data['sites'][] = $site_data;
		}

		// Generate filename
		$filename = 'bulkde-sites-export-' . gmdate( 'Y-m-d-H-i-s' ) . '.json';

		// Set headers for file download
		header( 'Content-Type: application/json' );
		header( 'Content-Disposition: attachment; filename="' . $filename . '"' );
		header( 'Cache-Control: no-cache, must-revalidate' );
		header( 'Expires: Sat, 26 Jul 1997 05:00:00 GMT' );

		// Output JSON
		echo wp_json_encode( $export_data, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES );
		wp_die();
	}

	/**
	 * Handle import sites AJAX request
	 */
	public function handle_import_sites() {
		check_ajax_referer( 'bulkde_nonce', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( 'Unauthorized' );
		}

		// Check if file was uploaded
		if ( ! isset( $_FILES['import_file'] ) || ! isset( $_FILES['import_file']['error'] ) || $_FILES['import_file']['error'] !== UPLOAD_ERR_OK ) {
			wp_send_json_error( 'No file uploaded or upload error occurred' );
			return;
		}

		$file = array(
			'name'     => isset( $_FILES['import_file']['name'] ) ? sanitize_file_name( $_FILES['import_file']['name'] ) : '',
			'type'     => isset( $_FILES['import_file']['type'] ) ? sanitize_mime_type( $_FILES['import_file']['type'] ) : '',
			'tmp_name' => isset( $_FILES['import_file']['tmp_name'] ) ? sanitize_text_field( $_FILES['import_file']['tmp_name'] ) : '',
			'error'    => isset( $_FILES['import_file']['error'] ) ? (int) $_FILES['import_file']['error'] : UPLOAD_ERR_NO_FILE,
			'size'     => isset( $_FILES['import_file']['size'] ) ? (int) $_FILES['import_file']['size'] : 0,
		);

		// Validate file type
		$file_extension = pathinfo( $file['name'], PATHINFO_EXTENSION );
		if ( strtolower( $file_extension ) !== 'json' ) {
			wp_send_json_error( 'Invalid file type. Please upload a JSON file.' );
			return;
		}

		// Read file content
		$file_content = file_get_contents( $file['tmp_name'] );
		if ( $file_content === false ) {
			wp_send_json_error( 'Failed to read file content' );
			return;
		}

		// Parse JSON
		$import_data = json_decode( $file_content, true );
		if ( json_last_error() !== JSON_ERROR_NONE ) {
			wp_send_json_error( 'Invalid JSON format: ' . json_last_error_msg() );
			return;
		}

		// Validate import data structure
		if ( ! isset( $import_data['sites'] ) || ! is_array( $import_data['sites'] ) ) {
			wp_send_json_error( 'Invalid import file structure. Missing sites data.' );
			return;
		}

		$site_manager     = new BulkDe_Site_Manager();
		$duplicate_action = isset( $_POST['duplicate_action'] ) ? sanitize_text_field( wp_unslash( $_POST['duplicate_action'] ) ) : 'skip';

		$results = array(
			'total'    => count( $import_data['sites'] ),
			'imported' => 0,
			'skipped'  => 0,
			'updated'  => 0,
			'errors'   => 0,
			'messages' => array(),
		);

		foreach ( $import_data['sites'] as $site_data ) {
			// Validate required fields
			if ( empty( $site_data['name'] ) || empty( $site_data['url'] ) || empty( $site_data['ftp_host'] ) || empty( $site_data['ftp_username'] ) ) {
				++$results['errors'];
				$results['messages'][] = "Skipped site '{$site_data['name']}' - missing required fields";
				continue;
			}

			// Check for existing site with same name
			$existing_sites = $site_manager->get_sites();
			$existing_site  = null;
			foreach ( $existing_sites as $existing ) {
				if ( $existing['name'] === $site_data['name'] ) {
					$existing_site = $existing;
					break;
				}
			}

			// Handle duplicates based on action
			if ( $existing_site ) {
				if ( $duplicate_action === 'skip' ) {
					++$results['skipped'];
					$results['messages'][] = "Skipped '{$site_data['name']}' - already exists";
					continue;
				} elseif ( $duplicate_action === 'update' ) {
					$site_data['id'] = $existing_site['id'];
					// Don't overwrite password if not provided in import
					if ( empty( $site_data['ftp_password'] ) ) {
						unset( $site_data['ftp_password'] );
					}
				}
				// For 'rename' action, the name will be automatically unique
			}

			// Generate unique name if needed
			if ( $duplicate_action === 'rename' && $existing_site ) {
				$base_name = $site_data['name'];
				$counter   = 1;
				do {
					$site_data['name'] = $base_name . ' (' . $counter . ')';
					++$counter;
					$name_exists = false;
					foreach ( $existing_sites as $existing ) {
						if ( $existing['name'] === $site_data['name'] ) {
							$name_exists = true;
							break;
						}
					}
				} while ( $name_exists );
			}

			// Set default values
			$site_data['ftp_port'] = isset( $site_data['ftp_port'] ) ? (int) $site_data['ftp_port'] : 21;
			$site_data['ftp_path'] = isset( $site_data['ftp_path'] ) ? $site_data['ftp_path'] : '/wp-content/plugins/';

			// If no password provided, set empty (will fail validation for new sites)
			if ( ! isset( $site_data['ftp_password'] ) ) {
				$site_data['ftp_password'] = '';
			}

			// Save site
			$result = $site_manager->save_site( $site_data );

			if ( $result ) {
				if ( $existing_site && $duplicate_action === 'update' ) {
					++$results['updated'];
					$results['messages'][] = "Updated '{$site_data['name']}'";
				} else {
					++$results['imported'];
					$results['messages'][] = "Imported '{$site_data['name']}'";
				}
			} else {
				++$results['errors'];
				$results['messages'][] = "Failed to import '{$site_data['name']}' - validation failed or database error";
			}
		}

		wp_send_json_success( $results );
	}
}
