jQuery(document).ready(function($) {
    
    // Global variables
    let currentSiteId = null;
    
    // Initialize the plugin
    initBulkDeployer();
    
    function initBulkDeployer() {
        initTabs();
        initCollapsibleForm();
        bindEvents();
        updateDeployButton();
    }
    
    // Initialize tab switching
    function initTabs() {
        $('.bulkde-tabs .nav-tab').on('click', function(e) {
            e.preventDefault();
            
            const targetTab = $(this).data('tab');
            
            // Remove active class from all tabs and panels
            $('.bulkde-tabs .nav-tab').removeClass('nav-tab-active');
            $('.bulkde-tab-panel').removeClass('bulkde-tab-active');
            
            // Add active class to clicked tab and corresponding panel
            $(this).addClass('nav-tab-active');
            $('#tab-' + targetTab).addClass('bulkde-tab-active');
        });
    }
    
    // Initialize collapsible form
    function initCollapsibleForm() {
        $('.bulkde-form-toggle').on('click', function() {
            const toggle = $(this);
            const formContainer = toggle.next('.bulkde-form-container');
            const isExpanded = toggle.attr('aria-expanded') === 'true';
            
            if (isExpanded) {
                formContainer.slideUp(300);
                toggle.attr('aria-expanded', 'false');
                toggle.find('.bulkde-toggle-icon').text('+');
            } else {
                formContainer.slideDown(300);
                toggle.attr('aria-expanded', 'true');
                toggle.find('.bulkde-toggle-icon').text('−');
            }
        });
    }
    
    function bindEvents() {
        // Site form submission
        $('#bulkde-site-form').on('submit', handleSiteFormSubmit);
        
        // Test connection button
        $('#bulkde-test-connection').on('click', handleTestConnection);
        
        // Clear form button
        $('#bulkde-clear-form').on('click', clearSiteForm);
        
        // Generate standard path button
        $('#bulkde-generate-path').on('click', handleGeneratePath);
        
        // Edit site button
        $(document).on('click', '.bulkde-edit-site', handleEditSite);
        
        // Delete site button
        $(document).on('click', '.bulkde-delete-site', handleDeleteSite);
        
        // Deploy plugins button (arrow button)
        $('#bulkde-deploy-plugins').on('click', handleDeployPlugins);
        
        // Checkbox change events
        $(document).on('change', 'input[name="plugins[]"], input[name="sites[]"]', updateDeployButton);
        
        // Import/Export functionality
        bindImportExportEvents();
    }
    
    function handleSiteFormSubmit(e) {

        e.preventDefault();
        const formData = new FormData(e.target);
        formData.append('action', 'bulkde_save_site');
        formData.append('nonce', bulkde_ajax.nonce);
        
        if (currentSiteId) {
            formData.append('id', currentSiteId);
        }
        
        showLoading(bulkde_ajax.strings.saving || 'Saving site...');
        
        $.ajax({
            url: bulkde_ajax.ajax_url,
            type: 'POST',
            data: formData,
            processData: false,
            contentType: false,
            success: function(response) {
                hideLoading();
                if (response.success) {
                    showNotice('Site saved successfully!', 'success');
                    clearSiteForm();
                    location.reload(); // Refresh to show updated sites list
                } else {
                    showNotice(response.data || 'Failed to save site', 'error');
                }
            },
            error: function() {
                hideLoading();
                showNotice('An error occurred while saving the site', 'error');
            }
        });
    }
    
    function handleTestConnection() {
        const formData = new FormData($('#bulkde-site-form')[0]);
        formData.append('action', 'bulkde_test_connection');
        formData.append('nonce', bulkde_ajax.nonce);
        
        // Add site ID if editing an existing site
        if (currentSiteId) {
            formData.append('id', currentSiteId);
        }
        
        showLoading(bulkde_ajax.strings.testing_connection || 'Testing connection...');
        
        $.ajax({
            url: bulkde_ajax.ajax_url,
            type: 'POST',
            data: formData,
            processData: false,
            contentType: false,
            success: function(response) {
                hideLoading();
                if (response.success) {
                    showNotice(bulkde_ajax.strings.connection_success || 'Connection successful!', 'success');
                } else {
                    showNotice(response.data || bulkde_ajax.strings.connection_failed || 'Connection failed', 'error');
                }
            },
            error: function() {
                hideLoading();
                showNotice('An error occurred while testing the connection', 'error');
            }
        });
    }
    
    function handleEditSite() {
        const siteData = $(this).data('site');
        if (!siteData) return;
        
        // Populate form with site data
        $('#site_name').val(siteData.name);
        $('#site_url').val(siteData.url);
        $('#ftp_host').val(siteData.ftp_host);
        $('#ftp_port').val(siteData.ftp_port);
        $('#ftp_username').val(siteData.ftp_username);
        $('#ftp_password').val(''); // Don't populate password for security
        $('#ftp_path').val(siteData.ftp_path);
        
        currentSiteId = siteData.id;
        
        // Expand form if collapsed
        const formContainer = $('.bulkde-form-container');
        const formToggle = $('.bulkde-form-toggle');
        
        if (formToggle.attr('aria-expanded') === 'false') {
            formContainer.slideDown(300);
            formToggle.attr('aria-expanded', 'true');
            formToggle.find('.bulkde-toggle-icon').text('−');
        }
        
        // Update form title and button
        formToggle.find('.bulkde-toggle-text').text('Edit Site');
        $('.bulkde-form-actions button[type="submit"]').text('Update Site');
        
        // Remove required attribute from password field when editing
        $('#ftp_password').removeAttr('required');
        
        // Update password field label to indicate it's optional
        $('#ftp_password').closest('.bulkde-form-group').find('label').html('FTP Password <small>(leave blank to keep existing)</small>');
        
        // Switch to Site Management tab
        $('.bulkde-tabs .nav-tab[data-tab="sites"]').trigger('click');
        
        // Scroll to form
        $('html, body').animate({
            scrollTop: formContainer.offset().top - 50
        }, 500);
    }
    
    function handleDeleteSite() {
        const siteId = $(this).data('site-id');
        const siteName = $(this).closest('tr').find('td:first').text();
        
        if (!confirm(bulkde_ajax.strings.confirm_delete || 'Are you sure you want to delete this site?')) {
            return;
        }
        
        const formData = new FormData();
        formData.append('action', 'bulkde_delete_site');
        formData.append('nonce', bulkde_ajax.nonce);
        formData.append('id', siteId);
        
        showLoading('Deleting site...');
        
        $.ajax({
            url: bulkde_ajax.ajax_url,
            type: 'POST',
            data: formData,
            processData: false,
            contentType: false,
            success: function(response) {
                hideLoading();
                if (response.success) {
                    showNotice('Site deleted successfully!', 'success');
                    location.reload(); // Refresh to update sites list
                } else {
                    showNotice(response.data || 'Failed to delete site', 'error');
                }
            },
            error: function() {
                hideLoading();
                showNotice('An error occurred while deleting the site', 'error');
            }
        });
    }
    
    function handleDeployPlugins() {
        const selectedPlugins = $('input[name="plugins[]"]:checked').map(function() {
            return this.value;
        }).get();
        
        const selectedSites = $('input[name="sites[]"]:checked').map(function() {
            return this.value;
        }).get();
        

        
        if (selectedPlugins.length === 0) {
            showNotice('Please select at least one plugin to deploy', 'error');
            return;
        }
        
        if (selectedSites.length === 0) {
            showNotice('Please select at least one site to deploy to', 'error');
            return;
        }
        
        const formData = new FormData();
        formData.append('action', 'bulkde_deploy_plugins');
        formData.append('nonce', bulkde_ajax.nonce);
        
        // Ensure we have valid data before sending
        if (selectedPlugins.length > 0) {
            formData.append('plugins', JSON.stringify(selectedPlugins));
        } else {
            formData.append('plugins', JSON.stringify([]));
        }
        
        if (selectedSites.length > 0) {
            formData.append('sites', JSON.stringify(selectedSites));
        } else {
            formData.append('sites', JSON.stringify([]));
        }
        
        showLoading(bulkde_ajax.strings.deploying || 'Deploying plugins...');
        
        $.ajax({
            url: bulkde_ajax.ajax_url,
            type: 'POST',
            data: formData,
            processData: false,
            contentType: false,
            success: function(response) {
                hideLoading();
                if (response.success) {
                    showDeploymentResults(response.data);
                    showNotice(bulkde_ajax.strings.success || 'Deployment completed successfully!', 'success');
                } else {
                    showDeploymentResults(response.data);
                    showNotice(bulkde_ajax.strings.error || 'An error occurred during deployment', 'error');
                }
            },
            error: function() {
                hideLoading();
                showNotice('An error occurred during deployment', 'error');
            }
        });
    }
    
    function showDeploymentResults(data) {
        const resultsContainer = $('#bulkde-results-content');
        let html = '';
        
        // Show summary
        if (data.summary) {
            html += '<div class="bulkde-result-item bulkde-result-summary">';
            html += `<strong>Deployment Summary:</strong> ${data.summary.success} successful, ${data.summary.failure} failed out of ${data.summary.total} total deployments`;
            html += '</div>';
        }
        
        // Show individual results
        if (data.results && data.results.length > 0) {
            data.results.forEach(function(result) {
                const cssClass = result.success ? 'bulkde-result-success' : 'bulkde-result-error';
                html += '<div class="bulkde-result-item ' + cssClass + '">';
                html += '<strong>' + (result.site_name || 'Site ' + result.site_id) + '</strong>';
                if (result.plugin_name) {
                    html += ' - <strong>' + result.plugin_name + '</strong>';
                }
                html += '<br><small>' + result.message + '</small>';
                html += '</div>';
            });
        }
        
        resultsContainer.html(html);
        $('#bulkde-results').show();
        
        // Scroll to results
        $('html, body').animate({
            scrollTop: $('#bulkde-results').offset().top - 50
        }, 500);
    }
    
    function clearSiteForm() {
        $('#bulkde-site-form')[0].reset();
        currentSiteId = null;
        $('.bulkde-form-toggle .bulkde-toggle-text').text('Add New Site');
        $('.bulkde-form-actions button[type="submit"]').text('Save Site');
        
        // Restore required attribute to password field for new sites
        $('#ftp_password').attr('required', 'required');
        
        // Restore original password field label
        $('#ftp_password').closest('.bulkde-form-group').find('label').html('FTP Password *');
    }
    
    function handleGeneratePath(e) {
        e.preventDefault();
        
        const username = $('#ftp_username').val();
        const siteUrl = $('#site_url').val();
        
        if (!username) {
            showNotice('Please enter the FTP username first', 'error');
            $('#ftp_username').focus();
            return;
        }
        
        if (!siteUrl) {
            showNotice('Please enter the site URL first', 'error');
            $('#site_url').focus();
            return;
        }
        
        // Extract domain from URL (remove protocol and www)
        let domain = siteUrl.replace(/^https?:\/\//, '').replace(/^www\./, '');
        
        // Remove trailing slash if present
        domain = domain.replace(/\/$/, '');
        
        // Generate the standard path
        const standardPath = `/home/${username}/${domain}/wp-content/plugins/`;
        
        // Set the path in the input field
        $('#ftp_path').val(standardPath);
        
        showNotice('Standard path generated successfully!', 'success');
    }
    
    function updateDeployButton() {
        const hasPlugins = $('input[name="plugins[]"]:checked').length > 0;
        const hasSites = $('input[name="sites[]"]:checked').length > 0;
        
        $('#bulkde-deploy-plugins').prop('disabled', !(hasPlugins && hasSites));
    }
    
    function showLoading(message) {
        $('#bulkde-loading-message').text(message);
        $('#bulkde-loading-overlay').show();
    }
    
    function hideLoading() {
        $('#bulkde-loading-overlay').hide();
    }
    
    function showNotice(message, type) {
        // Remove existing notices
        $('.bulkde-notice').remove();
        
        // Create new notice
        const notice = $('<div class="bulkde-notice bulkde-notice-' + type + '">' + message + '</div>');
        
        // Insert at the top of the container
        $('.bulkde-container').prepend(notice);
        
        // Auto-hide after 5 seconds
        setTimeout(function() {
            notice.fadeOut(function() {
                $(this).remove();
            });
        }, 5000);
    }
    
    // Utility function to validate form
    function validateForm(form) {
        let isValid = true;
        const requiredFields = form.find('[required]');
        
        requiredFields.each(function() {
            if (!$(this).val()) {
                $(this).addClass('error');
                isValid = false;
            } else {
                $(this).removeClass('error');
            }
        });
        
        return isValid;
    }
    
    // Add form validation
    $('#bulkde-site-form').on('submit', function(e) {
        if (!validateForm($(this))) {
            e.preventDefault();
            showNotice('Please fill in all required fields', 'error');
        }
    });
    
    // Real-time form validation
    $('#bulkde-site-form input[required]').on('blur', function() {
        if (!$(this).val()) {
            $(this).addClass('error');
        } else {
            $(this).removeClass('error');
        }
    });
    
    // Select all/none functionality for checkboxes
    function addSelectAllFunctionality() {
        // Add select all buttons if needed
        if ($('.bulkde-checkbox-group').length > 0) {
            $('.bulkde-checkbox-group').each(function() {
                const container = $(this);
                const checkboxes = container.find('input[type="checkbox"]');
                
                if (checkboxes.length > 3) {
                    const selectAllBtn = $('<button type="button" class="button button-small">Select All</button>');
                    const selectNoneBtn = $('<button type="button" class="button button-small">Select None</button>');
                    
                    container.before(selectAllBtn).before(selectNoneBtn);
                    
                    selectAllBtn.on('click', function() {
                        checkboxes.prop('checked', true);
                        updateDeployButton();
                    });
                    
                    selectNoneBtn.on('click', function() {
                        checkboxes.prop('checked', false);
                        updateDeployButton();
                    });
                }
            });
        }
    }
    
    // Initialize select all functionality
    addSelectAllFunctionality();
    
    // Import/Export Event Bindings
    function bindImportExportEvents() {
        // Export all sites checkbox toggle
        $('#bulkde-export-all-sites').on('change', function() {
            const isChecked = $(this).is(':checked');
            const siteList = $('#bulkde-export-site-list');
            
            if (isChecked) {
                siteList.hide();
                siteList.find('input[type="checkbox"]').prop('checked', false);
            } else {
                siteList.show();
            }
        });
        
        // Export sites button
        $('#bulkde-export-sites').on('click', handleExportSites);
        
        // Import sites button
        $('#bulkde-import-sites').on('click', handleImportSites);
        
        // File input change validation
        $('#bulkde-import-file').on('change', function() {
            const file = this.files[0];
            if (file) {
                if (!file.name.toLowerCase().endsWith('.json')) {
                    showNotice(bulkde_ajax.strings.invalid_file_type || 'Please select a valid JSON file.', 'error');
                    $(this).val('');
                }
            }
        });
    }
    
    // Handle export sites
    function handleExportSites() {
        const includePasswords = $('#bulkde-export-passwords').is(':checked');
        const exportAll = $('#bulkde-export-all-sites').is(':checked');
        
        // Show password warning if passwords are included
        if (includePasswords) {
            if (!confirm('⚠️ WARNING: You are about to export passwords in plain text. This is a security risk!\n\nOnly do this if you trust the destination of this export file.\n\nContinue?')) {
                return;
            }
        }
        
        let selectedSites = [];
        if (!exportAll) {
            selectedSites = $('input[name="export_sites[]"]:checked').map(function() {
                return parseInt(this.value);
            }).get();
            
            if (selectedSites.length === 0) {
                showNotice('Please select at least one site to export.', 'error');
                return;
            }
        }
        
        showLoading(bulkde_ajax.strings.exporting || 'Exporting sites...');
        
        // Create a form to submit the export request
        const form = $('<form>', {
            method: 'POST',
            action: bulkde_ajax.ajax_url,
            style: 'display: none;'
        });
        
        form.append($('<input>', {
            type: 'hidden',
            name: 'action',
            value: 'bulkde_export_sites'
        }));
        
        form.append($('<input>', {
            type: 'hidden',
            name: 'nonce',
            value: bulkde_ajax.nonce
        }));
        
        form.append($('<input>', {
            type: 'hidden',
            name: 'include_passwords',
            value: includePasswords ? 'true' : 'false'
        }));
        
        if (!exportAll && selectedSites.length > 0) {
            form.append($('<input>', {
                type: 'hidden',
                name: 'selected_sites',
                value: JSON.stringify(selectedSites)
            }));
        }
        
        $('body').append(form);
        form.submit();
        form.remove();
        
        // Hide loading after a short delay (since this triggers a download)
        setTimeout(function() {
            hideLoading();
            showNotice(bulkde_ajax.strings.export_success || 'Sites exported successfully!', 'success');
        }, 1000);
    }
    
    // Handle import sites
    function handleImportSites() {
        const fileInput = $('#bulkde-import-file')[0];
        const duplicateAction = $('#bulkde-duplicate-action').val();
        
        if (!fileInput.files.length) {
            showNotice(bulkde_ajax.strings.no_file_selected || 'Please select a file to import.', 'error');
            return;
        }
        
        const file = fileInput.files[0];
        
        // Validate file type
        if (!file.name.toLowerCase().endsWith('.json')) {
            showNotice(bulkde_ajax.strings.invalid_file_type || 'Please select a valid JSON file.', 'error');
            return;
        }
        
        // Confirmation dialog
        if (!confirm(bulkde_ajax.strings.confirm_import || 'This will import the sites. Existing sites with the same name may be updated. Continue?')) {
            return;
        }
        
        showLoading(bulkde_ajax.strings.importing || 'Importing sites...');
        
        const formData = new FormData();
        formData.append('action', 'bulkde_import_sites');
        formData.append('nonce', bulkde_ajax.nonce);
        formData.append('import_file', file);
        formData.append('duplicate_action', duplicateAction);
        
        $.ajax({
            url: bulkde_ajax.ajax_url,
            type: 'POST',
            data: formData,
            processData: false,
            contentType: false,
            success: function(response) {
                hideLoading();
                if (response.success) {
                    showImportResults(response.data);
                    showNotice(bulkde_ajax.strings.import_success || 'Sites imported successfully!', 'success');
                    
                    // Clear the form
                    $('#bulkde-import-form')[0].reset();
                    
                    // Reload the page to show updated sites
                    setTimeout(function() {
                        location.reload();
                    }, 3000);
                } else {
                    showNotice(response.data || bulkde_ajax.strings.import_error || 'Error importing sites', 'error');
                }
            },
            error: function(xhr, status, error) {
                hideLoading();
                console.error('Import error:', error);
                showNotice('An error occurred during import: ' + error, 'error');
            }
        });
    }
    
    // Show import results
    function showImportResults(data) {
        // Remove existing results
        $('.bulkde-import-results').remove();
        
        const resultsHtml = `
            <div class="bulkde-import-results">
                <h4>Import Results</h4>
                <div class="bulkde-import-summary">
                    Total: ${data.total} | Imported: ${data.imported} | Updated: ${data.updated} | Skipped: ${data.skipped} | Errors: ${data.errors}
                </div>
                ${data.messages && data.messages.length > 0 ? `
                    <ul>
                        ${data.messages.map(msg => `<li>${msg}</li>`).join('')}
                    </ul>
                ` : ''}
            </div>
        `;
        
        $('.bulkde-import-section').append(resultsHtml);
    }
}); 