<?php
/**
 * Plugin Name: Bulk Delete Product Images
 * Description: Adds a bulk action to delete featured and gallery images for selected WooCommerce products.
 * Version: 1.0.0
 * Author: Keramaros Antonios
 * Author URI: https://keramaros.gr
 * Requires at least: 5.0
 * Tested up to: 6.8
 * Requires plugin: woocommerce
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 */

if (!defined('ABSPATH')) exit;

// 1. Register new bulk action
add_filter('bulk_actions-edit-product', function ($bulk_actions) {
    $bulk_actions['delete_product_images'] = __('Delete Images', 'bulk-delete-product-images');
    return $bulk_actions;
});

// 2. Handle the bulk action
add_filter('handle_bulk_actions-edit-product', function ($redirect_to, $action, $post_ids) {
    if ($action !== 'delete_product_images') {
        return $redirect_to;
    }

    $deleted_count = 0;

    foreach ($post_ids as $product_id) {
        // Optional: skip non-draft products
        // if (get_post_status($product_id) !== 'draft') continue;

        // Delete featured image
        $thumb_id = get_post_thumbnail_id($product_id);
        if ($thumb_id) {
            wp_delete_attachment($thumb_id, true);
            delete_post_thumbnail($product_id);
        }

        // Delete gallery images
        $gallery = get_post_meta($product_id, '_product_image_gallery', true);
        if ($gallery) {
            $ids = array_filter(array_map('intval', explode(',', $gallery)));
            foreach ($ids as $id) {
                wp_delete_attachment($id, true);
            }
            delete_post_meta($product_id, '_product_image_gallery');
        }

        $deleted_count++;
    }

    $redirect_to = add_query_arg('deleted_product_images', $deleted_count, $redirect_to);
    return $redirect_to;
}, 10, 3);

// 3. Admin notice
add_action('admin_notices', function () {
    if (!empty($_REQUEST['deleted_product_images'])) {
        $count = intval($_REQUEST['deleted_product_images']);
        printf(
            '<div class="updated notice"><p>' .
            esc_html__('Deleted images for %d product(s).', 'bulk-delete-product-images') .
            '</p></div>',
            $count
        );
    }
});
