(function( $ ) {
	'use strict';
	
	var $scanBtn = $('#bctb-scan'),
		$convertAllBtn = $('#bctb-convert-all'),
		$output = $('#bctb-output'),
		$typeSelect = $('#bctb-type-select'),
		$singleConvertLinks = $('.bctb-single-convert'),
		$bulkActionTop = $('#doaction'),
		$bulkActionBottom = $('#doaction2'),
		conversionQueue = [],
		isProcessing = false;
	
	// Initialize hidden editor for block conversion
	$('<div />').attr('id', 'bctb-editor').css('display', 'none').appendTo('body');
	wp.editPost.initializeEditor('bctb-editor');
	
	// Scan button handler
	$scanBtn.on('click', function(e) {
		e.preventDefault();
		$scanBtn.prop('disabled', true);
		$convertAllBtn.hide();
		startScan();
	});
	
	// Scan function
	function startScan( offset = 0, total = -1 ) {
		if ( isProcessing ) return;
		isProcessing = true;
		
		var nonce = $scanBtn.data('nonce');
		var selectedType = $typeSelect.length ? $typeSelect.val() : 'all';
		$output.html( bctbData.scanningMsg );
		
		$.ajax({
			method: 'POST',
			url: bctbData.ajaxUrl,
			data: {
				action: 'bctb_scan',
				offset: offset,
				total: total,
				post_type: selectedType,
				_wpnonce: nonce
			}
		})
		.done(function( data ) {
			isProcessing = false;
			
			if ( typeof data !== 'object' ) {
				$output.html( bctbData.errorMsg );
				return;
			}
			
			if ( data.error ) {
				$output.html( data.message );
				return;
			}
			
			if ( data.offset >= data.total ) {
				$scanBtn.prop('disabled', false);
				$output.html( data.message );
				var selectedType = $typeSelect.length ? $typeSelect.val() : 'all';
				var url = new URL(window.location.href);
				url.searchParams.set('bctb_scan_done', '1');
				if ( selectedType !== 'all' ) {
					url.searchParams.set('bctb_type', selectedType);
				}
				window.location.href = url.toString();
				return;
			}
			
			startScan( data.offset, data.total );
			$output.html( data.message );
		})
		.fail(function() {
			isProcessing = false;
			$output.html( bctbData.errorMsg );
		});
	}
	
	// Bulk convert button handler
	$convertAllBtn.on('click', function(e) {
		e.preventDefault();
		if ( ! confirm( bctbData.confirmMsg ) ) return;
		$convertAllBtn.prop('disabled', true);
		startBulkConvert();
	});
	
	// Single convert link handler
	$singleConvertLinks.on('click', function(e) {
		e.preventDefault();
		var postId = $(this).data('json').post;
		conversionQueue.push( postId );
		processConversionQueue();
	});
	
	// Bulk convert function
	function startBulkConvert( offset = 0, total = -1 ) {
		if ( isProcessing ) return;
		isProcessing = true;
		
		var nonce = $convertAllBtn.data('nonce');
		var selectedType = $typeSelect.length ? $typeSelect.val() : 'all';
		$output.html( bctbData.convertingMsg );
		
		$.ajax({
			method: 'GET',
			url: bctbData.ajaxUrl,
			data: {
				action: 'bctb_bulk_convert',
				offset: offset,
				total: total,
				post_type: selectedType,
				_wpnonce: nonce
			}
		})
		.done(function( data ) {
			isProcessing = false;
			
			if ( typeof data !== 'object' ) {
				$output.html( bctbData.errorMsg );
				return;
			}
			
			if ( data.error ) {
				$output.html( data.message );
				return;
			}
			
			var convertedData = [];
			var postsLength = data.postsData.length;
			
			for ( var i = 0; i < postsLength; i++ ) {
				convertedData.push({
					id: data.postsData[i].id,
					content: convertToBlocks( data.postsData[i].content )
				});
			}
			
			saveBulkConverted( convertedData, data.offset, data.total, data.message );
		})
		.fail(function() {
			isProcessing = false;
			$output.html( bctbData.errorMsg );
		});
	}
	
	// Save bulk converted posts
	function saveBulkConverted( convertedData, offset, total, message ) {
		if ( isProcessing ) return;
		isProcessing = true;
		
		var nonce = $convertAllBtn.data('nonce');
		var ajaxData = {
			action: 'bctb_bulk_convert',
			offset: offset,
			total: total,
			postsData: convertedData,
			_wpnonce: nonce
		};
		
		$.ajax({
			method: 'POST',
			url: bctbData.ajaxUrl,
			data: ajaxData
		})
		.done(function( data ) {
			isProcessing = false;
			
			if ( typeof data !== 'object' ) {
				$output.html( bctbData.errorMsg );
				return;
			}
			
			if ( data.error ) {
				$output.html( data.message );
				return;
			}
			
			if ( data.offset >= data.total ) {
				$convertAllBtn.prop('disabled', false);
				$output.html( bctbData.successMsg );
				return;
			}
			
			startBulkConvert( offset, total );
			$output.html( message );
		})
		.fail(function() {
			isProcessing = false;
			$output.html( bctbData.errorMsg );
		});
	}
	
	// Process conversion queue
	function processConversionQueue() {
		if ( conversionQueue.length === 0 ) {
			$bulkActionTop.prop('disabled', false);
			$bulkActionBottom.prop('disabled', false);
			return;
		}
		
		if ( isProcessing ) return;
		isProcessing = true;
		
		var postId = conversionQueue.shift();
		var $link = $('#bctb-convert-' + postId);
		$link.hide().after( bctbData.convertingSingleMsg );
		
		var jsonData = $link.data('json');
		
		$.ajax({
			method: 'GET',
			url: bctbData.ajaxUrl,
			data: jsonData
		})
		.done(function( data ) {
			isProcessing = false;
			
			if ( typeof data !== 'object' || data.error ) {
				$link.parent().html( bctbData.failedMsg );
				processConversionQueue();
				return;
			}
			
			var convertedContent = convertToBlocks( data.message );
			saveSingleConverted( convertedContent, $link, postId );
		})
		.fail(function() {
			isProcessing = false;
			$link.parent().html( bctbData.failedMsg );
			processConversionQueue();
		});
	}
	
	// Convert content to blocks
	function convertToBlocks( content ) {
		var blocks = wp.blocks.rawHandler({
			HTML: content
		});
		return wp.blocks.serialize(blocks);
	}
	
	// Save single converted post
	function saveSingleConverted( content, $link, postId ) {
		if ( isProcessing ) return;
		isProcessing = true;
		
		var jsonData = $link.data('json');
		jsonData.content = content;
		
		$.ajax({
			method: 'POST',
			url: bctbData.ajaxUrl,
			data: jsonData
		})
		.done(function( data ) {
			isProcessing = false;
			
			$('#bctb-cb-' + postId).prop('checked', false).prop('disabled', true);
			
			if ( typeof data !== 'object' || data.error ) {
				$link.parent().html( bctbData.failedMsg );
				processConversionQueue();
				return;
			}
			
			$link.parent().html( bctbData.convertedSingleMsg );
			processConversionQueue();
		})
		.fail(function() {
			isProcessing = false;
			$('#bctb-cb-' + postId).prop('checked', false).prop('disabled', true);
			$link.parent().html( bctbData.failedMsg );
			processConversionQueue();
		});
	}
	
	// Bulk action handlers
	$bulkActionTop.on('click', function(e) {
		e.preventDefault();
		if ( $('select[name="action"]').val() === 'bulk_convert' ) {
			processBulkSelection();
		}
	});
	
	$bulkActionBottom.on('click', function(e) {
		e.preventDefault();
		if ( $('select[name="action2"]').val() === 'bulk_convert' ) {
			processBulkSelection();
		}
	});
	
	// Process bulk selection
	function processBulkSelection() {
		$('input[name="bulk_convert[]"]').each(function() {
			if ( $(this).prop('checked') ) {
				conversionQueue.push( $(this).val() );
			}
		});
		
		$bulkActionTop.prop('disabled', true);
		$bulkActionBottom.prop('disabled', true);
		processConversionQueue();
	}
	
})( jQuery );
