<?php
/**
 * Bulk Classic to Block Converter
 *
 * @package Bulk_Classic_To_Block
 * @version 1.0.0
 * @author DeepakNess
 * @link https://deepakness.com
 *
 * @wordpress-plugin
 * Plugin Name: Bulk Classic to Block
 * Description: Convert classic editor content to Gutenberg blocks in bulk. Supports all post types including custom post types.
 * Version: 1.0.0
 * Author: DeepakNess
 * Author URI: https://deepakness.com
 * License: GPL-2.0+
 * License URI: http://www.gnu.org/licenses/gpl-2.0.txt
 * Text Domain: bulk-classic-to-block
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Plugin constants
define( 'BCTB_VERSION', '1.0.1' );
define( 'BCTB_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
define( 'BCTB_PLUGIN_URL', plugin_dir_url( __FILE__ ) );
define( 'BCTB_META_FLAG', 'bctb_needs_conversion' );
define( 'BCTB_POST_STATUSES', array( 'publish', 'future', 'draft', 'private' ) );

/**
 * Initialize plugin
 */
function bctb_init() {
	if ( ! bctb_is_block_editor_available() ) {
		return;
	}
	
	add_action( 'admin_menu', 'bctb_add_admin_menu' );
	add_action( 'plugin_action_links_' . plugin_basename( __FILE__ ), 'bctb_add_settings_link' );
	add_action( 'wp_ajax_bctb_scan', 'bctb_handle_scan_request' );
	add_action( 'wp_ajax_bctb_bulk_convert', 'bctb_handle_bulk_convert' );
	add_action( 'wp_ajax_bctb_single_convert', 'bctb_handle_single_convert' );
	add_action( 'post_updated', 'bctb_auto_index_post', 10, 2 );
}
add_action( 'plugins_loaded', 'bctb_init' );

/**
 * Check if block editor is available
 */
function bctb_is_block_editor_available() {
	$has_gutenberg = has_filter( 'replace_editor', 'gutenberg_init' );
	$has_block_editor = version_compare( $GLOBALS['wp_version'], '5.0-beta', '>' );
	
	if ( ! $has_gutenberg && ! $has_block_editor ) {
		return false;
	}
	
	if ( bctb_classic_editor_active() ) {
		$editor_setting = get_option( 'classic-editor-replace' );
		return in_array( $editor_setting, array( 'no-replace', 'block' ), true );
	}
	
	return true;
}

/**
 * Check if classic editor plugin is active
 */
function bctb_classic_editor_active() {
	if ( ! function_exists( 'is_plugin_active' ) ) {
		require_once ABSPATH . 'wp-admin/includes/plugin.php';
	}
	return is_plugin_active( 'classic-editor/classic-editor.php' );
}

/**
 * Get all scanable post types
 */
function bctb_get_scanable_types( $include_private = false ) {
	$types = array( 'post', 'page' );
	
	$custom_types = get_post_types( array(
		'public'   => true,
		'_builtin' => false,
	), 'names' );
	
	if ( ! empty( $custom_types ) ) {
		$types = array_merge( $types, $custom_types );
	}
	
	if ( $include_private ) {
		$private_types = get_post_types( array(
			'public'             => false,
			'_builtin'           => false,
			'publicly_queryable' => true,
		), 'names' );
		
		if ( ! empty( $private_types ) ) {
			$types = array_merge( $types, $private_types );
		}
	}
	
	return $types;
}

/**
 * Add admin menu
 */
function bctb_add_admin_menu() {
	$page = add_management_page(
		__( 'Bulk Classic to Block', 'bulk-classic-to-block' ),
		__( 'Classic to Block', 'bulk-classic-to-block' ),
		'manage_options',
		'bctb-converter',
		'bctb_render_admin_page'
	);
	
	add_action( 'load-' . $page, 'bctb_load_admin_assets' );
}

/**
 * Add settings link to plugins page
 */
function bctb_add_settings_link( $links ) {
	$settings_link = '<a href="' . admin_url( 'tools.php?page=bctb-converter' ) . '">' . __( 'Settings', 'bulk-classic-to-block' ) . '</a>';
	array_unshift( $links, $settings_link );
	return $links;
}

/**
 * Load admin assets
 */
function bctb_load_admin_assets() {
	add_action( 'admin_enqueue_scripts', 'bctb_enqueue_scripts' );
	add_action( 'admin_enqueue_scripts', 'bctb_enqueue_admin_styles' );
}

/**
 * Enqueue scripts and styles
 */
function bctb_enqueue_scripts() {
	wp_register_script(
		'bctb-main',
		BCTB_PLUGIN_URL . 'js/scripts.js',
		array( 'jquery', 'wp-blocks', 'wp-edit-post' ),
		BCTB_VERSION,
		true
	);
	
	wp_localize_script( 'bctb-main', 'bctbData', array(
		'ajaxUrl'              => admin_url( 'admin-ajax.php' ),
		'errorMsg'             => '<div class="error"><p>' . __( 'Server error occurred!', 'bulk-classic-to-block' ) . '</p></div>',
		/* translators: %s: Percentage complete (0-100) */
		'scanningMsg'          => '<p>' . sprintf( __( 'Scanning... %s%%', 'bulk-classic-to-block' ), 0 ) . '</p>',
		/* translators: %s: Percentage complete (0-100) */
		'convertingMsg'        => '<p>' . sprintf( __( 'Converting... %s%%', 'bulk-classic-to-block' ), 0 ) . '</p>',
		'successMsg'           => '<div class="updated"><p>' . __( 'All content successfully converted!', 'bulk-classic-to-block' ) . '</p></div>',
		'confirmMsg'            => __( 'You are about to convert all classic content to blocks. This action is irreversible. Continue?', 'bulk-classic-to-block' ),
		'convertingSingleMsg'   => __( 'Converting...', 'bulk-classic-to-block' ),
		'convertedSingleMsg'    => __( 'Converted', 'bulk-classic-to-block' ),
		'failedMsg'             => __( 'Failed', 'bulk-classic-to-block' ),
	) );
	
	wp_enqueue_script( 'bctb-main' );
}

/**
 * Enqueue admin page styles
 */
function bctb_enqueue_admin_styles() {
	$css = '
		.bctb-controls-box {
			background: #fff;
			border: 1px solid #c3c4c7;
			box-shadow: 0 1px 1px rgba(0,0,0,.04);
			padding: 20px;
			margin: 20px 0;
		}
		.bctb-controls-header {
			border-bottom: 1px solid #dcdcde;
			padding-bottom: 15px;
			margin-bottom: 20px;
		}
		.bctb-controls-header h2 {
			margin: 0 0 5px 0;
			font-size: 18px;
		}
		.bctb-controls-header p {
			margin: 0;
			color: #646970;
		}
		.bctb-controls-body {
			display: flex;
			align-items: flex-end;
			gap: 15px;
			flex-wrap: wrap;
		}
		.bctb-control-group {
			flex: 1;
			min-width: 200px;
		}
		.bctb-control-group label {
			display: block;
			font-weight: 600;
			margin-bottom: 5px;
		}
		.bctb-control-group select {
			width: 100%;
		}
		.bctb-control-actions {
			display: flex;
			gap: 10px;
			align-items: flex-end;
		}
		.bctb-control-actions .button {
			margin: 0;
		}
		@media (max-width: 782px) {
			.bctb-controls-body {
				flex-direction: column;
			}
			.bctb-control-group {
				width: 100%;
			}
			.bctb-control-actions {
				width: 100%;
			}
			.bctb-control-actions .button {
				width: 100%;
			}
		}
	';
	wp_add_inline_style( 'wp-admin', $css );
}

/**
 * Render admin page
 */
function bctb_render_admin_page() {
	// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Reading filter parameter for display, not processing form submission
	$selected_type = ! empty( $_GET['bctb_type'] ) ? sanitize_text_field( wp_unslash( $_GET['bctb_type'] ) ) : 'all';
	$stats = bctb_get_conversion_stats( $selected_type );
	$has_items = bctb_has_convertible_items( $stats );
	?>
	<div id="bctb-wrapper" class="wrap">
		<h1><?php echo esc_html( get_admin_page_title() ); ?></h1>
		
		<?php
		global $bctb_success, $bctb_error;
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Reading display parameter, not processing form submission
		if ( ! empty( $_GET['result'] ) && $_GET['result'] == '1' ) {
			if ( ! empty( $bctb_error ) ) {
				echo '<div class="notice notice-error"><p>' . esc_html( $bctb_error ) . '</p></div>';
			}
			if ( ! empty( $bctb_success ) ) {
				echo '<div class="notice notice-success"><p>' . esc_html( $bctb_success ) . '</p></div>';
			}
		}
		?>
		
		<!-- Warning Box -->
		<div class="notice notice-warning">
			<p><strong><?php esc_html_e( 'Important:', 'bulk-classic-to-block' ); ?></strong> <?php esc_html_e( 'Conversion is irreversible. Always backup your site first.', 'bulk-classic-to-block' ); ?></p>
		</div>
		
		<!-- Main Controls Section -->
		<div class="bctb-controls-box">
			<div class="bctb-controls-header">
				<h2><?php esc_html_e( 'Convert Classic Content to Blocks', 'bulk-classic-to-block' ); ?></h2>
				<p><?php esc_html_e( 'Select a post type and scan for classic editor content that can be converted to Gutenberg blocks.', 'bulk-classic-to-block' ); ?></p>
			</div>
			<div class="bctb-controls-body">
				<div class="bctb-control-group">
					<label for="bctb-type-select"><?php esc_html_e( 'Post Type', 'bulk-classic-to-block' ); ?></label>
					<select id="bctb-type-select" name="bctb_type_select" class="regular-text">
						<option value="all"><?php esc_html_e( 'All Types', 'bulk-classic-to-block' ); ?></option>
						<?php
						$all_types = bctb_get_scanable_types();
						foreach ( $all_types as $type ) {
							$type_obj = get_post_type_object( $type );
							if ( $type_obj ) {
								$selected = ( $selected_type === $type ) ? ' selected="selected"' : '';
								echo '<option value="' . esc_attr( $type ) . '"' . esc_attr( $selected ) . '>' . esc_html( $type_obj->labels->name ) . '</option>';
							}
						}
						?>
					</select>
				</div>
				<div class="bctb-control-actions">
					<button id="bctb-scan" class="button button-primary button-large" data-nonce="<?php echo esc_attr( wp_create_nonce( 'bctb_scan' ) ); ?>"><?php esc_html_e( 'Scan Content', 'bulk-classic-to-block' ); ?></button>
					<?php if ( $has_items ) : ?>
						<button id="bctb-convert-all" class="button button-primary button-large" data-nonce="<?php echo esc_attr( wp_create_nonce( 'bctb_bulk_convert' ) ); ?>"><?php esc_html_e( 'Convert All', 'bulk-classic-to-block' ); ?></button>
					<?php endif; ?>
				</div>
			</div>
		</div>
		
		<!-- Results Section -->
		<div id="bctb-output">
			<?php if ( $has_items ) : ?>
				<div id="bctb-results"><?php bctb_display_stats( $stats ); ?></div>
				<div id="bctb-table"><?php bctb_display_table(); ?></div>
			<?php else : ?>
				<?php
				// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Reading display parameter, not processing form submission
				if ( ! empty( $_GET['bctb_scan_done'] ) ) :
				?>
					<div class="notice notice-info">
						<p><?php esc_html_e( 'No classic content found. All content is already using Gutenberg blocks.', 'bulk-classic-to-block' ); ?></p>
					</div>
				<?php else : ?>
					<div class="notice notice-info">
						<p><?php esc_html_e( 'Click "Scan Content" to find classic editor posts that can be converted.', 'bulk-classic-to-block' ); ?></p>
					</div>
				<?php endif; ?>
			<?php endif; ?>
		</div>
	</div>
	<?php
}

/**
 * Handle scan request
 */
function bctb_handle_scan_request() {
	// Check user capabilities
	if ( ! current_user_can( 'manage_options' ) ) {
		wp_send_json_error( array( 'message' => __( 'Insufficient permissions.', 'bulk-classic-to-block' ) ) );
	}
	
	// Verify nonce
	check_ajax_referer( 'bctb_scan', '_wpnonce' );
	
	$offset = isset( $_REQUEST['offset'] ) ? intval( $_REQUEST['offset'] ) : 0;
	$total_expected = isset( $_REQUEST['total'] ) ? intval( $_REQUEST['total'] ) : -1;
	
	$selected_type = ! empty( $_REQUEST['post_type'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['post_type'] ) ) : 'all';
	$types_to_scan = ( $selected_type === 'all' ) ? bctb_get_scanable_types() : ( post_type_exists( $selected_type ) ? array( $selected_type ) : bctb_get_scanable_types() );
	
	$total_actual = 0;
	foreach ( $types_to_scan as $type ) {
		$counts = wp_count_posts( $type );
		if ( $counts ) {
			foreach ( BCTB_POST_STATUSES as $status ) {
				if ( isset( $counts->$status ) ) {
					$total_actual += (int) $counts->$status;
				}
			}
		}
	}
	
	$response = array(
		'error'   => false,
		'offset'  => $total_actual,
		'total'   => $total_actual,
		'message' => '',
	);
	
	if ( $total_expected != -1 && $total_expected != $total_actual ) {
		$response['error'] = true;
		$response['message'] = '<div class="error"><p>' . __( 'Content changed during scan. Please try again.', 'bulk-classic-to-block' ) . '</p></div>';
		wp_send_json( $response );
	}
	
	$posts = get_posts( array(
		'post_type'      => $types_to_scan,
		'post_status'    => BCTB_POST_STATUSES,
		'posts_per_page' => 10,
		'offset'         => $offset,
	) );
	
	foreach ( $posts as $post ) {
		if ( bctb_is_classic_content( $post->post_content ) ) {
			update_post_meta( $post->ID, BCTB_META_FLAG, '1' );
		}
		$offset++;
	}
	
	$response['offset'] = $offset;
	$percentage = $total_actual > 0 ? (int) ( $offset / $total_actual * 100 ) : 0;
	/* translators: %s: Percentage complete (0-100) */
	$response['message'] = '<p>' . sprintf( __( 'Scanning... %s%%', 'bulk-classic-to-block' ), $percentage ) . '</p>';
	
	wp_send_json( $response );
}

/**
 * Handle bulk convert request
 */
function bctb_handle_bulk_convert() {
	// Check user capabilities
	if ( ! current_user_can( 'manage_options' ) ) {
		wp_send_json_error( array( 'message' => __( 'Insufficient permissions.', 'bulk-classic-to-block' ) ) );
	}
	
	// Verify nonce
	check_ajax_referer( 'bctb_bulk_convert', '_wpnonce' );
	
	if ( ! empty( $_GET['total'] ) ) {
		$offset = isset( $_GET['offset'] ) ? intval( $_GET['offset'] ) : 0;
		$total_expected = intval( $_GET['total'] );
		
		$selected_type = ! empty( $_GET['post_type'] ) ? sanitize_text_field( wp_unslash( $_GET['post_type'] ) ) : 'all';
		$types_to_convert = ( $selected_type === 'all' ) ? bctb_get_scanable_types() : ( post_type_exists( $selected_type ) ? array( $selected_type ) : bctb_get_scanable_types() );
		
		$total_actual = bctb_count_flagged( $types_to_convert );
		if ( $total_expected == -1 ) {
			$total_expected = $total_actual;
		}
		
		$response = array(
			'error'     => false,
			'offset'    => $total_expected,
			'total'     => $total_expected,
			'message'   => '',
			'postsData' => array(),
		);
		
		if ( $total_expected != ( $total_actual + $offset ) ) {
			$response['error'] = true;
			$response['message'] = '<div class="error"><p>' . __( 'Content changed during conversion. Please try again.', 'bulk-classic-to-block' ) . '</p></div>';
			wp_send_json( $response );
		}
		
		// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_key,WordPress.DB.SlowDBQuery.slow_db_query_meta_value -- Required for filtering flagged posts
		$posts = get_posts( array(
			'post_type'      => $types_to_convert,
			'post_status'    => BCTB_POST_STATUSES,
			'posts_per_page' => 10,
			'meta_key'       => BCTB_META_FLAG,
			'meta_value'     => '1',
		) );
		
		$posts_data = array();
		foreach ( $posts as $post ) {
			// Only include posts the user can edit
			if ( current_user_can( 'edit_post', $post->ID ) ) {
				$posts_data[] = array(
					'id'      => $post->ID,
					'content' => wpautop( $post->post_content ),
				);
			}
			$offset++;
		}
		
		$response['postsData'] = $posts_data;
		$response['offset'] = $offset;
		$percentage = $total_expected > 0 ? (int) ( $offset / $total_expected * 100 ) : 0;
		/* translators: %s: Percentage complete (0-100) */
		$response['message'] = '<p>' . sprintf( __( 'Converting... %s%%', 'bulk-classic-to-block' ), $percentage ) . '</p>';
		
		wp_send_json( $response );
	}
	
	if ( ! empty( $_POST['total'] ) ) {
		$response = array(
			'error'  => false,
			'offset' => isset( $_POST['offset'] ) ? intval( $_POST['offset'] ) : 0,
			'total'  => intval( $_POST['total'] ),
		);
		
		if ( ! empty( $_POST['postsData'] ) && is_array( $_POST['postsData'] ) ) {
			// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Data is sanitized in the loop below
			$posts_data = wp_unslash( $_POST['postsData'] );
			foreach ( $posts_data as $post ) {
				if ( ! is_array( $post ) ) {
					continue;
				}
				$post_id = isset( $post['id'] ) ? intval( $post['id'] ) : 0;
				
				// Validate post ID
				if ( $post_id <= 0 ) {
					continue;
				}
				
				// Check if user can edit this post
				if ( ! current_user_can( 'edit_post', $post_id ) ) {
					continue;
				}
				
				// Verify post exists and is valid
				$existing_post = get_post( $post_id );
				if ( ! $existing_post ) {
					continue;
				}
				
				// Sanitize content
				$content = isset( $post['content'] ) ? wp_unslash( $post['content'] ) : '';
				$content = wp_kses_post( $content );
				
				$post_data = array(
					'ID'           => $post_id,
					'post_content' => $content,
				);
				
				if ( ! wp_update_post( $post_data ) ) {
					$response['error'] = true;
					wp_send_json( $response );
				}
			}
		}
		
		wp_send_json( $response );
	}
}

/**
 * Handle single convert request
 */
function bctb_handle_single_convert() {
	// Check user capabilities
	if ( ! current_user_can( 'manage_options' ) ) {
		wp_send_json_error( array( 'message' => __( 'Insufficient permissions.', 'bulk-classic-to-block' ) ) );
	}
	
	$post_id = isset( $_REQUEST['post'] ) ? intval( $_REQUEST['post'] ) : 0;
	
	// Validate post ID
	if ( $post_id <= 0 ) {
		wp_send_json_error( array( 'message' => __( 'Invalid post ID.', 'bulk-classic-to-block' ) ) );
	}
	
	// Verify nonce
	check_ajax_referer( 'bctb_convert_' . $post_id, '_wpnonce' );
	
	// Check if user can edit this post
	if ( ! current_user_can( 'edit_post', $post_id ) ) {
		wp_send_json_error( array( 'message' => __( 'You do not have permission to edit this post.', 'bulk-classic-to-block' ) ) );
	}
	
	$response = array(
		'error'   => false,
		'message' => '',
	);
	
	if ( ! empty( $_GET['post'] ) ) {
		$post = get_post( $post_id );
		if ( ! $post ) {
			wp_send_json_error( array( 'message' => __( 'Post not found.', 'bulk-classic-to-block' ) ) );
		}
		$response['message'] = wpautop( $post->post_content );
		wp_send_json( $response );
	}
	
	if ( ! empty( $_POST['post'] ) ) {
		// Verify post exists
		$existing_post = get_post( $post_id );
		if ( ! $existing_post ) {
			wp_send_json_error( array( 'message' => __( 'Post not found.', 'bulk-classic-to-block' ) ) );
		}
		
		// Sanitize content
		// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Content is sanitized with wp_kses_post() below
		$content = isset( $_POST['content'] ) ? wp_unslash( $_POST['content'] ) : '';
		$content = wp_kses_post( $content );
		
		$post_data = array(
			'ID'           => $post_id,
			'post_content' => $content,
		);
		
		if ( ! wp_update_post( $post_data ) ) {
			wp_send_json_error( array( 'message' => __( 'Failed to update post.', 'bulk-classic-to-block' ) ) );
		}
		
		$response['message'] = $post_id;
		wp_send_json( $response );
	}
}

/**
 * Check if content is classic editor format
 */
function bctb_is_classic_content( $content ) {
	return ! empty( $content ) && strpos( $content, '<!-- wp:' ) === false;
}

/**
 * Get conversion statistics
 */
function bctb_get_conversion_stats( $selected_type = 'all' ) {
	$types = ( $selected_type === 'all' ) ? bctb_get_scanable_types() : ( post_type_exists( $selected_type ) ? array( $selected_type ) : bctb_get_scanable_types() );
	
	$stats = array();
	foreach ( $types as $type ) {
		$type_obj = get_post_type_object( $type );
		if ( $type_obj ) {
			$stats[ $type_obj->labels->name ] = bctb_count_flagged( $type );
		}
	}
	
	return $stats;
}

/**
 * Check if there are convertible items
 */
function bctb_has_convertible_items( $stats ) {
	foreach ( $stats as $count ) {
		if ( $count > 0 ) {
			return true;
		}
	}
	return false;
}

/**
 * Count flagged posts
 */
function bctb_count_flagged( $type ) {
	// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_key,WordPress.DB.SlowDBQuery.slow_db_query_meta_value -- Required for counting flagged posts
	$query = new WP_Query( array(
		'posts_per_page' => -1,
		'post_type'      => $type,
		'meta_key'       => BCTB_META_FLAG,
		'meta_value'     => '1',
	) );
	return $query->post_count;
}

/**
 * Display statistics
 */
function bctb_display_stats( $stats ) {
	$total = array_sum( $stats );
	$output = '<div class="notice notice-info" style="margin: 20px 0;">';
	/* translators: %d: Number of posts */
	$output .= '<p><strong>' . sprintf( __( 'Found %d post(s) ready for conversion:', 'bulk-classic-to-block' ), $total ) . '</strong> ';
	$type_list = array();
	foreach ( $stats as $type => $count ) {
		if ( $count > 0 ) {
			$type_list[] = esc_html( $type ) . ' (' . esc_html( $count ) . ')';
		}
	}
	$output .= implode( ', ', $type_list );
	$output .= '</p></div>';
	echo wp_kses_post( $output );
}

/**
 * Display table
 */
function bctb_display_table() {
	?>
	<div class="meta-box-sortables ui-sortable">
		<?php
		$table = new BCTB_List_Table();
		$table->views();
		?>
		<form method="post">
			<?php
			wp_nonce_field( 'bctb_filter_action', 'bctb_filter_nonce' );
			$table->prepare_items();
			$table->search_box( __( 'Search', 'bulk-classic-to-block' ), 'bctb-search' );
			$table->display();
			?>
		</form>
	</div>
	<?php
}

/**
 * Get status label
 */
function bctb_get_status_label( $status ) {
	$labels = array(
		'any'     => __( 'All', 'bulk-classic-to-block' ),
		'publish' => __( 'Published', 'bulk-classic-to-block' ),
		'future'  => __( 'Scheduled', 'bulk-classic-to-block' ),
		'draft'   => __( 'Draft', 'bulk-classic-to-block' ),
		'private' => __( 'Private', 'bulk-classic-to-block' ),
	);
	return isset( $labels[ $status ] ) ? $labels[ $status ] : $status;
}

/**
 * Auto-index posts on save
 */
function bctb_auto_index_post( $post_id, $post_after ) {
	$types = bctb_get_scanable_types();
	if ( ! in_array( $post_after->post_type, $types, true ) ) {
		return;
	}
	
	if ( bctb_is_classic_content( $post_after->post_content ) ) {
		update_post_meta( $post_id, BCTB_META_FLAG, '1' );
	} else {
		delete_post_meta( $post_id, BCTB_META_FLAG );
	}
}

/**
 * Cleanup on deactivation
 */
function bctb_deactivate() {
	// Delete all meta flags using WordPress functions instead of direct DB query
	// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_key,WordPress.DB.SlowDBQuery.slow_db_query_meta_value -- Required for cleanup on deactivation
	$posts = get_posts( array(
		'posts_per_page' => -1,
		'post_type'      => 'any',
		'meta_key'       => BCTB_META_FLAG,
		'meta_value'     => '1',
	) );
	
	foreach ( $posts as $post ) {
		delete_post_meta( $post->ID, BCTB_META_FLAG );
	}
}
register_deactivation_hook( __FILE__, 'bctb_deactivate' );

// Include WP_List_Table if needed
if ( ! class_exists( 'WP_List_Table' ) ) {
	require_once ABSPATH . 'wp-admin/includes/class-wp-list-table.php';
}

/**
 * Verify filter nonce for list table requests
 *
 * @return bool True if nonce is valid, false otherwise.
 */
function bctb_verify_filter_nonce() {
	if ( isset( $_REQUEST['bctb_filter_nonce'] ) ) {
		return wp_verify_nonce( sanitize_text_field( wp_unslash( $_REQUEST['bctb_filter_nonce'] ) ), 'bctb_filter_action' );
	}
	return false;
}

/**
 * Custom list table class
 */
class BCTB_List_Table extends WP_List_Table {

	public function __construct() {
		parent::__construct( array(
			'singular' => __( 'Item', 'bulk-classic-to-block' ),
			'plural'   => __( 'Items', 'bulk-classic-to-block' ),
			'ajax'     => false,
		) );
	}

	public static function get_query_args() {
		$types = bctb_get_scanable_types();
		
		// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_key,WordPress.DB.SlowDBQuery.slow_db_query_meta_value -- Required for filtering flagged posts
		$args = array(
			'post_type'   => $types,
			'post_status' => BCTB_POST_STATUSES,
			'meta_key'    => BCTB_META_FLAG,
			'meta_value'  => '1',
		);
		
		// Only apply filter parameters if nonce is verified.
		if ( bctb_verify_filter_nonce() ) {
			if ( ! empty( $_REQUEST['bctb_type'] ) ) {
				$requested = sanitize_text_field( wp_unslash( $_REQUEST['bctb_type'] ) );
				if ( post_type_exists( $requested ) ) {
					$args['post_type'] = $requested;
				}
			}
			
			if ( ! empty( $_REQUEST['bctb_status'] ) ) {
				$args['post_status'] = sanitize_text_field( wp_unslash( $_REQUEST['bctb_status'] ) );
			}
			
			if ( ! empty( $_REQUEST['s'] ) ) {
				$args['s'] = sanitize_text_field( wp_unslash( $_REQUEST['s'] ) );
			}
			
			if ( ! empty( $_REQUEST['orderby'] ) && $_REQUEST['orderby'] === 'post_title' ) {
				$args['orderby'] = 'title';
				$args['order'] = ! empty( $_REQUEST['order'] ) && $_REQUEST['order'] === 'asc' ? 'ASC' : 'DESC';
			}
		}
		
		return $args;
	}

	public static function get_items( $per_page = 20, $page = 1 ) {
		$args = self::get_query_args();
		$args['posts_per_page'] = $per_page;
		$args['offset'] = ( $page - 1 ) * $per_page;
		
		$posts = get_posts( $args );
		$results = array();
		
		foreach ( $posts as $post ) {
			$results[] = array(
				'ID'         => $post->ID,
				'post_title' => $post->post_title,
				'post_type'  => $post->post_type,
				'action'     => '',
			);
		}
		
		return $results;
	}

	public static function count_items() {
		$args = self::get_query_args();
		$args['posts_per_page'] = -1;
		$query = new WP_Query( $args );
		return $query->post_count;
	}

	public static function count_by_status( $status ) {
		$types = bctb_get_scanable_types();
		
		// Only apply type filter if nonce is verified.
		if ( bctb_verify_filter_nonce() && ! empty( $_REQUEST['bctb_type'] ) ) {
			$requested = sanitize_text_field( wp_unslash( $_REQUEST['bctb_type'] ) );
			if ( post_type_exists( $requested ) ) {
				$types = array( $requested );
			}
		}
		
		$statuses = ( $status === 'any' ) ? BCTB_POST_STATUSES : array( $status );
		
		// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_key,WordPress.DB.SlowDBQuery.slow_db_query_meta_value -- Required for counting by status
		$args = array(
			'post_type'      => $types,
			'post_status'    => $statuses,
			'posts_per_page' => -1,
			'meta_key'       => BCTB_META_FLAG,
			'meta_value'     => '1',
		);
		
		$query = new WP_Query( $args );
		return $query->post_count;
	}

	public function no_items() {
		esc_html_e( 'No items found.', 'bulk-classic-to-block' );
	}

	public function get_columns() {
		return array(
			'cb'         => '<input type="checkbox" />',
			'post_title' => __( 'Title', 'bulk-classic-to-block' ),
			'post_type'  => __( 'Type', 'bulk-classic-to-block' ),
			'action'     => __( 'Action', 'bulk-classic-to-block' ),
		);
	}

	public function column_cb( $item ) {
		$id = absint( $item['ID'] );
		return sprintf( '<input type="checkbox" id="bctb-cb-%s" name="bulk_convert[]" value="%s" />', $id, $id );
	}

	public function column_post_title( $item ) {
		return '<strong><a href="' . esc_url( get_permalink( $item['ID'] ) ) . '" target="_blank">' . esc_html( $item['post_title'] ) . '</a></strong>';
	}

	public function column_post_type( $item ) {
		$type_obj = get_post_type_object( $item['post_type'] );
		$label = $type_obj ? $type_obj->labels->singular_name : $item['post_type'];
		$url = esc_url( add_query_arg( array( 'bctb_type' => $item['post_type'] ) ) );
		return '<a href="' . $url . '">' . esc_html( $label ) . '</a>';
	}

	public function column_action( $item ) {
		$nonce = wp_create_nonce( 'bctb_convert_' . $item['ID'] );
		$json = json_encode( array(
			'action'  => 'bctb_single_convert',
			'post'    => $item['ID'],
			'_wpnonce' => $nonce,
		) );
		return '<a href="#" id="bctb-convert-' . absint( $item['ID'] ) . '" class="bctb-single-convert" data-json=\'' . esc_attr( $json ) . '\'>' . __( 'Convert', 'bulk-classic-to-block' ) . '</a>';
	}

	public function get_sortable_columns() {
		return array(
			'post_title' => array( 'post_title', false ),
		);
	}

	public function get_bulk_actions() {
		return array(
			'bulk_convert' => __( 'Convert', 'bulk-classic-to-block' ),
		);
	}

	protected function get_views() {
		$statuses = array_merge( array( 'any' ), BCTB_POST_STATUSES );
		$links = array();
		
		foreach ( $statuses as $status ) {
			$count = self::count_by_status( $status );
			if ( $count > 0 ) {
				$label = bctb_get_status_label( $status );
				// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Reading filter parameters for display, not processing form submission
				$requested_status = ! empty( $_REQUEST['bctb_status'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['bctb_status'] ) ) : '';
				$current = ( empty( $requested_status ) && $status === 'any' ) || ( $requested_status === $status );
				
				if ( $current ) {
					$links[ $status ] = '<strong>' . esc_html( $label ) . '</strong> (' . esc_html( $count ) . ')';
				} else {
					// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Reading filter parameters for display, not processing form submission
					$page = isset( $_REQUEST['page'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['page'] ) ) : '';
					$url = $status === 'any' ? '?page=' . esc_attr( $page ) : '?page=' . esc_attr( $page ) . '&bctb_status=' . esc_attr( $status );
					$links[ $status ] = '<a href="' . esc_url( $url ) . '">' . esc_html( $label ) . '</a> (' . esc_html( $count ) . ')';
				}
			}
		}
		
		return $links;
	}

	public function extra_tablenav( $which ) {
		if ( $which !== 'top' || ! $this->has_items() ) {
			return;
		}
		
		$types = bctb_get_scanable_types();
		?>
		<div class="alignleft actions">
			<select name="bctb_type">
				<option value=""><?php esc_html_e( 'All Types', 'bulk-classic-to-block' ); ?></option>
				<?php
				foreach ( $types as $type ) {
					$type_obj = get_post_type_object( $type );
					if ( $type_obj ) {
						// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Reading filter parameters for display, not processing form submission
						$requested_type = ! empty( $_REQUEST['bctb_type'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['bctb_type'] ) ) : '';
						$selected = ( $requested_type === $type ) ? ' selected="selected"' : '';
						echo '<option value="' . esc_attr( $type ) . '"' . esc_attr( $selected ) . '>' . esc_html( $type_obj->labels->name ) . '</option>';
					}
				}
				?>
			</select>
			<?php submit_button( __( 'Filter', 'bulk-classic-to-block' ), 'action', 'bctb_filter', false ); ?>
		</div>
		<?php
	}

	public function prepare_items() {
		$this->_column_headers = array( $this->get_columns(), array(), $this->get_sortable_columns() );
		$this->process_bulk_action();
		
		$per_page = $this->get_items_per_page( 'posts_per_page', 20 );
		$current_page = $this->get_pagenum();
		$total_items = self::count_items();
		
		$this->set_pagination_args( array(
			'total_items' => $total_items,
			'per_page'    => $per_page,
		) );
		
		$this->items = self::get_items( $per_page, $current_page );
	}

	public function process_bulk_action() {
		if ( ( isset( $_POST['action'] ) && $_POST['action'] === 'bulk_convert' ) || ( isset( $_POST['action2'] ) && $_POST['action2'] === 'bulk_convert' ) ) {
			// Verify nonce if form-based submission is used
			if ( isset( $_POST['_wpnonce'] ) ) {
				check_admin_referer( 'bulk-items' );
			}
			
			if ( ! empty( $_POST['bulk_convert'] ) && is_array( $_POST['bulk_convert'] ) ) {
				// Bulk conversion is handled via JavaScript/AJAX
				// This is just for form processing if needed
			}
		}
	}
}
